# 帳票設計書 7-Jinja2テンプレートレスポンス

## 概要

本ドキュメントは、FastAPIフレームワークにおけるJinja2テンプレートレスポンス帳票の設計仕様を定義する。Jinja2Templatesは、Jinja2テンプレートエンジンを使用してHTML帳票を動的に生成するための機能を提供する。

### 本帳票の処理概要

Jinja2テンプレートレスポンスは、HTMLテンプレートにデータをバインドして動的なHTMLページを生成する帳票である。サーバーサイドレンダリング（SSR）を実現し、データベースから取得したデータや計算結果をHTMLとしてクライアントに配信できる。FastAPIはStarletteのJinja2Templatesをそのまま再エクスポートしている。

**業務上の目的・背景**：APIエンドポイントだけでなく、Webページを提供する場合にサーバーサイドでHTMLを生成する必要がある。Jinja2は、Pythonエコシステムで広く使われているテンプレートエンジンであり、条件分岐、ループ、フィルター、マクロなどの強力な機能を持つ。これにより、帳票印刷、ダッシュボード、管理画面などのHTML出力が可能になる。

**帳票の利用シーン**：管理画面の構築、印刷用帳票のHTML生成、メールテンプレート生成、ダッシュボード表示、静的サイト生成時に利用される。

**主要な出力内容**：
1. HTMLテンプレートにデータをバインドした結果
2. 動的なWebページコンテンツ
3. 印刷用帳票HTML
4. メールHTML本文

**帳票の出力タイミング**：エンドポイント関数がTemplateResponseを返却した際に、テンプレートがレンダリングされてHTMLレスポンスが送信される。

**帳票の利用者**：Webブラウザ、メールクライアント、印刷システム

## 帳票種別

HTMLテンプレート出力

## 利用画面

| 画面No | 画面名 | URL/ルーティング | 出力操作 |
|--------|--------|-----------------|---------|
| 1 | 任意のエンドポイント | 開発者が定義 | Jinja2Templatesを使用するエンドポイントへのHTTPリクエスト |

## 出力形式

### 基本仕様

| 項目 | 内容 |
|-----|------|
| ファイル形式 | HTML |
| 用紙サイズ | N/A（テンプレート依存） |
| 向き | N/A |
| ファイル名 | N/A（動的生成） |
| 出力方法 | HTTPレスポンス（HTMLResponse） |
| 文字コード | UTF-8 |

### HTML固有設定

| 項目 | 内容 |
|-----|------|
| Content-Type | text/html; charset=utf-8 |
| テンプレートエンジン | Jinja2 |
| テンプレート拡張子 | .html（通常） |

## 帳票レイアウト

### レイアウト概要

Jinja2テンプレートは、静的HTML部分と動的な変数・ロジック部分で構成される。

```
Jinja2テンプレート構造
├── 継承（extends）
│   └── ベーステンプレート
├── ブロック定義（block）
│   ├── header
│   ├── content
│   └── footer
├── 変数展開（{{ variable }}）
├── 制御構文
│   ├── {% if condition %}
│   ├── {% for item in items %}
│   └── {% include "partial.html" %}
└── フィルター（{{ value|filter }}）
```

### テンプレート構文

| No | 構文 | 説明 | 用途 |
|----|------|------|------|
| 1 | `{{ var }}` | 変数展開 | データ表示 |
| 2 | `{% if %}` | 条件分岐 | 条件付き表示 |
| 3 | `{% for %}` | ループ | リスト表示 |
| 4 | `{% extends %}` | テンプレート継承 | レイアウト共有 |
| 5 | `{% block %}` | ブロック定義 | 差し替え部分 |
| 6 | `{% include %}` | 部分テンプレート | 共通パーツ |
| 7 | `{% macro %}` | マクロ定義 | 再利用可能な関数 |

### フィルター例

| フィルター | 説明 | 例 |
|----------|------|-----|
| escape | HTMLエスケープ | `{{ text\|escape }}` |
| safe | エスケープ無効化 | `{{ html\|safe }}` |
| default | デフォルト値 | `{{ value\|default("N/A") }}` |
| length | 長さ取得 | `{{ items\|length }}` |
| date | 日付フォーマット | `{{ date\|date }}` |

## 出力条件

### 抽出条件

| 条件名 | 説明 | 必須 |
|-------|------|-----|
| Jinja2Templates初期化 | テンプレートディレクトリが設定されていること | Yes |
| テンプレートファイル | 指定されたテンプレートが存在すること | Yes |
| コンテキストデータ | テンプレートに渡すデータ（request含む） | Yes |

### ソート順

N/A（テンプレートロジックに依存）

### 改ページ条件

N/A（HTMLベース）

## データベース参照仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 結合条件 |
|-----------|------|---------|
| N/A | データベースへの直接参照なし（contextで外部から渡される） | - |

### テンプレートコンテキスト

Jinja2Templatesはcontextとして渡された辞書をテンプレートに展開する。

| コンテキストキー | 説明 | 必須 |
|-----------------|------|-----|
| request | StarletteのRequestオブジェクト | Yes |
| その他 | テンプレートで使用する任意のデータ | No |

## 計算仕様

### 計算項目一覧

| 項目名 | 計算式 | 端数処理 | 備考 |
|-------|-------|---------|------|
| テンプレートパス | directory / template_name | N/A | パス結合 |

## 処理フロー

### 出力フロー

```mermaid
flowchart TD
    A[エンドポイント呼び出し] --> B[Jinja2Templates.TemplateResponse呼び出し]
    B --> C[テンプレートファイル読み込み]
    C --> D{テンプレート存在?}
    D -->|Yes| E[テンプレートコンパイル]
    D -->|No| F[TemplateNotFoundエラー]
    E --> G[コンテキストデータバインド]
    G --> H[HTMLレンダリング]
    H --> I[HTMLResponse生成]
    I --> J[レスポンス返却]
```

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 表示メッセージ | 対処方法 |
|----------|---------|--------------|---------|
| TemplateNotFound | テンプレートファイルが存在しない | 500 Internal Server Error | テンプレートパスを確認 |
| UndefinedError | 未定義変数を使用 | 500 Internal Server Error | コンテキストにデータを追加 |
| TemplateSyntaxError | テンプレート構文エラー | 500 Internal Server Error | テンプレート構文を修正 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定データ件数 | テンプレートとコンテキストサイズに依存 |
| 目標出力時間 | 小規模テンプレート: 10ms以下、大規模: 100ms以下 |
| 同時出力数上限 | サーバーリソースに依存 |

## セキュリティ考慮事項

- XSS対策：ユーザー入力は自動エスケープされるが、`|safe`フィルター使用時は注意
- テンプレートインジェクション：ユーザー入力をテンプレート名として使用しない
- 機密情報：コンテキストに機密データを含めない
- CSRF対策：フォームにCSRFトークンを含める

## 備考

- Jinja2TemplatesはStarletteからの再エクスポートであり、FastAPI独自の実装はない
- auto_reload設定でテンプレート変更の自動検出が可能
- カスタムフィルターやグローバル変数の追加が可能
- テンプレートキャッシュにより2回目以降のレンダリングが高速化

---

## コードリーディングガイド

本帳票を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: FastAPIでのインポートを理解する

Jinja2TemplatesがどのようにFastAPIで提供されているかを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | templating.py | `fastapi/templating.py` | Jinja2Templatesの再エクスポート（行1）を確認 |

**読解のコツ**: `from starlette.templating import Jinja2Templates as Jinja2Templates`という1行のみであり、FastAPIは完全にStarletteの実装に依存している。

#### Step 2: 使用例を理解する

Jinja2Templatesの典型的な使用パターンを把握する。

```python
from fastapi import FastAPI, Request
from fastapi.templating import Jinja2Templates

app = FastAPI()
templates = Jinja2Templates(directory="templates")

@app.get("/")
async def read_root(request: Request):
    return templates.TemplateResponse(
        request=request,
        name="index.html",
        context={"title": "Home", "items": [1, 2, 3]}
    )
```

### プログラム呼び出し階層図

```
エンドポイント関数
    │
    └─ templates.TemplateResponse()
           │
           └─ Starlette Jinja2Templates.TemplateResponse()
                  │
                  ├─ jinja2.Environment.get_template()
                  │      └─ テンプレート読み込み・コンパイル
                  │
                  ├─ Template.render(context)
                  │      └─ HTMLレンダリング
                  │
                  └─ HTMLResponse
                         └─ HTTPレスポンス
```

### データフロー図

```
[入力]                        [処理]                         [出力]

HTTPリクエスト ──────────▶ エンドポイント関数 ─────────────▶ HTTPレスポンス
      │                         │                              │
      │                         ▼                              │
      │                 Jinja2Templates                        │
      │                         │                              │
      ▼                         │                              │
Request ─────────────────▶ TemplateResponse()                  │
      │                         │                              │
      │                         ├─ テンプレート読み込み         │
      │                         ├─ コンテキストバインド        │
      │                         ├─ レンダリング               │
      │                         │                              │
      │                         ▼                              ▼
テンプレートファイル ────▶ HTML生成 ────────────────────▶ HTMLResponse
  templates/xxx.html
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| templating.py | `fastapi/templating.py` | ソース | Jinja2Templatesの再エクスポート |
| templating.py | `starlette/templating.py` | 外部依存 | Jinja2Templatesの実装（Starlette） |
| *.html | `templates/` | テンプレート | Jinja2テンプレートファイル（ユーザー定義） |
