# 機能設計書 11-Depends

## 概要

本ドキュメントはFastAPIフレームワークにおける依存性注入（Dependency Injection）機能「Depends」の設計仕様を定義する。

### 本機能の処理概要

Depends機能は、FastAPIのパスオペレーション関数において、共通処理や前処理を宣言的に注入するための機能である。データベース接続、認証処理、設定値の取得など、複数のエンドポイントで共通して必要となる処理を、依存関係として定義し自動的に解決・注入することができる。

**業務上の目的・背景**：Webアプリケーション開発において、認証・認可、データベース接続、設定値の読み込みなど、多くのエンドポイントで共通して必要となる前処理が存在する。これらを各エンドポイントで重複して実装すると、コードの重複、保守性の低下、テスト困難性などの問題が発生する。Depends機能により、これらの共通処理を再利用可能な依存関係として定義し、宣言的に注入することで、コードの重複を排除し、関心の分離を実現する。

**機能の利用シーン**：
- データベースセッションの取得と管理
- ユーザー認証・認可処理の実行
- 設定値や環境変数の読み込み
- リクエストのバリデーション
- 共通のビジネスロジックの実行
- リソースのセットアップとクリーンアップ（yield使用時）

**主要な処理内容**：
1. パスオペレーション関数のパラメータから依存関係を検出
2. 依存関係の解決順序を決定（依存グラフの構築）
3. 各依存関数を呼び出し、戻り値をキャッシュ
4. 解決された値をパスオペレーション関数に注入
5. ジェネレータ依存関係の場合は、リクエスト処理後にクリーンアップを実行

**関連システム・外部連携**：
- Pydantic: バリデーションとデータモデルの連携
- Starlette: Request、Response、WebSocketオブジェクトとの連携
- SecurityBase: セキュリティ依存関係との統合

**権限による制御**：Depends自体は権限制御を行わないが、Security依存関係と組み合わせることでOAuth2スコープによるアクセス制御を実現できる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はバックエンド処理のため、直接関連する画面はない |

## 機能種別

データ連携 / バリデーション / 処理制御

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dependency | Callable[..., Any] \| None | No | 依存関数。呼び出し可能オブジェクト | Noneの場合は型アノテーションから推論 |
| use_cache | bool | No | 同一リクエスト内で結果をキャッシュするか（デフォルト: True） | ブール値 |
| scope | Literal["function", "request"] \| None | No | yieldを使う依存関係のスコープ（デフォルト: None） | "function"または"request" |

### 入力データソース

- パスオペレーション関数のパラメータ定義（型アノテーションとデフォルト値）
- `Annotated`を使用した依存関係の宣言
- 関数のデフォルト値として指定された`Depends()`

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 解決された依存値 | Any | 依存関数の戻り値。パスオペレーション関数のパラメータとして注入される |

### 出力先

- パスオペレーション関数のパラメータ

## 処理フロー

### 処理シーケンス

```
1. パラメータ解析（analyze_param）
   └─ 関数シグネチャから依存関係を検出
2. 依存グラフ構築（get_dependant）
   └─ 依存関係の階層構造を構築
3. 依存関係の解決（solve_dependencies）
   └─ 再帰的に全ての依存関係を解決
4. キャッシュチェック
   └─ use_cache=Trueかつキャッシュ存在時はキャッシュから取得
5. 依存関数の実行
   ├─ 通常関数: 同期/非同期で実行
   └─ ジェネレータ: AsyncExitStackで管理
6. 結果のキャッシュ保存
   └─ 同一リクエスト内での再利用のため
7. 値の注入
   └─ パスオペレーション関数への引数として渡す
```

### フローチャート

```mermaid
flowchart TD
    A[パスオペレーション呼び出し] --> B[依存関係の検出]
    B --> C{依存関係あり?}
    C -->|Yes| D[依存グラフ構築]
    C -->|No| G[パスオペレーション実行]
    D --> E{キャッシュ存在?}
    E -->|Yes| F[キャッシュから取得]
    E -->|No| H{ジェネレータ?}
    H -->|Yes| I[AsyncExitStackで管理]
    H -->|No| J{非同期関数?}
    J -->|Yes| K[awaitで実行]
    J -->|No| L[スレッドプールで実行]
    I --> M[結果をキャッシュ]
    K --> M
    L --> M
    F --> N[次の依存関係]
    M --> N
    N --> O{全て解決?}
    O -->|No| E
    O -->|Yes| G
    G --> P[レスポンス返却]
    P --> Q{yield依存関係あり?}
    Q -->|Yes| R[クリーンアップ実行]
    Q -->|No| S[終了]
    R --> S
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-11-01 | キャッシュ再利用 | use_cache=Trueの場合、同一リクエスト内で同じ依存関数の結果を再利用する | 複数箇所で同じ依存関係を使用する場合 |
| BR-11-02 | スコープ制約 | scope="request"の依存関係は、scope="function"の依存関係に依存できない | yieldを使用する依存関係のネスト時 |
| BR-11-03 | 型推論 | dependency=Noneの場合、型アノテーションから依存関数を推論する | Annotated使用時 |

### 計算ロジック

特になし。

## データベース操作仕様

本機能はデータベース操作を直接行わない。ただし、依存関数内でデータベース操作を行うことは可能。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| DependencyScopeError | スコープエラー | scope="request"の依存がscope="function"に依存 | 依存関係の設計を見直す |
| AssertionError | パラメータエラー | Dependsとデフォルト値を同時に指定 | いずれか一方のみを使用する |
| AssertionError | パラメータエラー | Annotated内とデフォルト値で重複指定 | いずれか一方のみを使用する |

### リトライ仕様

依存関数内でエラーが発生した場合、自動リトライは行わない。必要な場合は依存関数内で実装する。

## トランザクション仕様

本機能はトランザクション管理を直接行わない。依存関数内でyieldを使用することで、リクエスト処理のライフサイクルに合わせたトランザクション管理が可能。

## パフォーマンス要件

- キャッシュ機能により、同一依存関数の重複呼び出しを防止
- 同期関数はスレッドプールで実行し、イベントループをブロックしない
- 依存解決のオーバーヘッドは最小限に抑える設計

## セキュリティ考慮事項

- 依存関数内で認証・認可処理を実装可能
- Security依存関係と組み合わせてOAuth2スコープを管理
- キャッシュされた認証情報は同一リクエスト内でのみ有効

## 備考

- Python 3.9以降ではAnnotatedを使用した宣言が推奨される
- ジェネレータ依存関係は、yieldの前がセットアップ、後がクリーンアップとして動作する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、依存関係を表現するデータ構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | Dependsデータクラスの定義（746-750行目） |
| 1-2 | models.py | `fastapi/dependencies/models.py` | Dependantクラスの構造 |

**読解のコツ**: `Depends`は`@dataclass(frozen=True)`で定義された不変オブジェクト。`dependency`、`use_cache`、`scope`の3つのフィールドを持つ。

#### Step 2: エントリーポイントを理解する

処理の起点となるファイル・関数を特定。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | param_functions.py | `fastapi/param_functions.py` | Depends関数の定義（2209-2286行目） |

**主要処理フロー**:
1. **2209-2256行目**: Depends関数のシグネチャとパラメータ定義
2. **2257-2285行目**: docstringによる使用例の説明
3. **2286行目**: params.Dependsデータクラスのインスタンス化

#### Step 3: 依存関係の検出と解析

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.py | `fastapi/dependencies/utils.py` | analyze_param関数（357-524行目） |
| 3-2 | utils.py | `fastapi/dependencies/utils.py` | get_dependant関数（251-323行目） |

**主要処理フロー**:
- **372-378行目**: Annotatedから依存関係を抽出
- **417-418行目**: params.Dependsインスタンスの検出
- **420-429行目**: デフォルト値からの依存関係検出
- **441-444行目**: dependency=Noneの場合の型推論

#### Step 4: 依存関係の解決

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | utils.py | `fastapi/dependencies/utils.py` | solve_dependencies関数（563-700行目） |
| 4-2 | utils.py | `fastapi/dependencies/utils.py` | _solve_generator関数（543-551行目） |

**主要処理フロー**:
- **593-628行目**: 依存関係の再帰的解決
- **629-649行目**: キャッシュ確認とジェネレータ/コルーチン/同期関数の処理分岐
- **690-693行目**: SecurityScopesパラメータへの値注入

### プログラム呼び出し階層図

```
Depends(dependency, use_cache, scope)
    │
    └─ params.Depends（データクラス生成）
           │
           └─ get_dependant()（依存グラフ構築）
                  │
                  ├─ analyze_param()（パラメータ解析）
                  │      └─ Annotated/デフォルト値から依存検出
                  │
                  └─ solve_dependencies()（依存解決）
                         │
                         ├─ キャッシュ確認
                         │
                         ├─ _solve_generator()（ジェネレータ処理）
                         │      └─ AsyncExitStack管理
                         │
                         └─ run_in_threadpool()（同期関数実行）
```

### データフロー図

```
[入力]                    [処理]                         [出力]

パスオペレーション     analyze_param()              解決された依存値
関数定義        ───▶  依存検出           ───▶     (パラメータに注入)
                           │
                           ▼
型アノテーション     get_dependant()
Annotated          依存グラフ構築
デフォルト値              │
                           ▼
                   solve_dependencies()
                     再帰的解決
                           │
                           ▼
                      キャッシュ
                   (dependency_cache)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| param_functions.py | `fastapi/param_functions.py` | ソース | Depends関数の公開API定義 |
| params.py | `fastapi/params.py` | ソース | Dependsデータクラスの定義 |
| utils.py | `fastapi/dependencies/utils.py` | ソース | 依存関係の検出・解決ロジック |
| models.py | `fastapi/dependencies/models.py` | ソース | Dependantデータモデル |
| routing.py | `fastapi/routing.py` | ソース | ルート登録時の依存関係処理 |
| concurrency.py | `fastapi/concurrency.py` | ソース | 非同期実行ユーティリティ |
