# 機能設計書 12-Security

## 概要

本ドキュメントはFastAPIフレームワークにおけるセキュリティ依存性注入機能「Security」の設計仕様を定義する。

### 本機能の処理概要

Security機能は、Depends機能を拡張し、OAuth2スコープなどのセキュリティコンテキストを管理するための依存性注入機能である。通常のDependsと同様に依存関数を注入しつつ、OAuth2スコープ情報をOpenAPIドキュメントに反映させ、Swagger UIでのインタラクティブな認証テストを可能にする。

**業務上の目的・背景**：現代のWebアプリケーションでは、OAuth2やOpenID Connectなどの標準的な認証・認可プロトコルが広く使用されている。これらのプロトコルでは「スコープ」という概念で権限を管理する。Security機能により、各エンドポイントで必要なスコープを宣言的に定義し、OpenAPIスキーマに自動的に反映させることで、APIのセキュリティ要件を明確に文書化できる。

**機能の利用シーン**：
- OAuth2認証を使用するエンドポイントでのスコープ指定
- 権限ベースのアクセス制御の実装
- OpenAPIドキュメントへのセキュリティ要件の反映
- Swagger UIでの認証フローテスト
- 複数のセキュリティスキームを組み合わせた認証

**主要な処理内容**：
1. セキュリティ依存関数とスコープ情報を受け取る
2. Dependsと同様に依存関係として登録
3. スコープ情報をDependantオブジェクトに伝播
4. OpenAPIスキーマ生成時にセキュリティ要件として出力
5. 依存関数の実行結果を返却

**関連システム・外部連携**：
- OpenAPI: セキュリティスキームとスコープの定義
- Swagger UI: インタラクティブな認証テスト
- OAuth2認証クラス: OAuth2PasswordBearer等との連携

**権限による制御**：scopesパラメータで指定したスコープが、OpenAPIのsecurity要件として反映される。実際のスコープ検証は依存関数内で実装する必要がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | Security依存関係で定義されたスコープがOpenAPIスキーマに反映され、認証UIに表示される |

## 機能種別

セキュリティ / データ連携 / 処理制御

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| dependency | Callable[..., Any] \| None | No | セキュリティ依存関数（OAuth2PasswordBearer等） | 呼び出し可能オブジェクト |
| scopes | Sequence[str] \| None | No | 必要なOAuth2スコープのリスト | 文字列のシーケンス |
| use_cache | bool | No | 同一リクエスト内で結果をキャッシュするか（デフォルト: True） | ブール値 |

### 入力データソース

- パスオペレーション関数のパラメータ定義
- `Annotated`を使用した依存関係の宣言
- OAuth2認証クラスのインスタンス

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 解決された依存値 | Any | セキュリティ依存関数の戻り値（トークン、ユーザー情報等） |
| OpenAPIセキュリティ定義 | dict | OpenAPIスキーマのsecurityセクションに出力される |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（/openapi.json）

## 処理フロー

### 処理シーケンス

```
1. Securityインスタンス生成
   └─ dependency, scopes, use_cacheを保持
2. 依存関係の登録（get_dependant）
   └─ Dependsと同様に依存グラフに追加
3. スコープ情報の伝播
   └─ own_oauth_scopesとしてDependantに設定
4. 依存関係の解決（solve_dependencies）
   └─ Dependsと同じ処理フロー
5. OpenAPIスキーマ生成（get_openapi_security_definitions）
   └─ security_dependenciesからスキーム定義とスコープを抽出
6. 結果の返却
   └─ 依存関数の戻り値をパスオペレーション関数に注入
```

### フローチャート

```mermaid
flowchart TD
    A[Security依存関係宣言] --> B[params.Securityインスタンス生成]
    B --> C[get_dependant呼び出し]
    C --> D{scopesあり?}
    D -->|Yes| E[own_oauth_scopesに設定]
    D -->|No| F[空リストで継続]
    E --> G[依存グラフに追加]
    F --> G
    G --> H[solve_dependencies実行]
    H --> I[依存関数呼び出し]
    I --> J{SecurityScopesパラメータあり?}
    J -->|Yes| K[スコープ情報を注入]
    J -->|No| L[結果を返却]
    K --> L
    L --> M[OpenAPIスキーマ生成]
    M --> N[セキュリティ定義出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-12-01 | スコープ伝播 | 親の依存関係のスコープは子の依存関係に伝播される | ネストした依存関係使用時 |
| BR-12-02 | スコープマージ | 同一セキュリティスキームの複数スコープは統合される | 複数箇所で同じスキームを使用 |
| BR-12-03 | OpenAPI反映 | scopesはOpenAPIのsecurity要件として自動反映される | OpenAPIスキーマ生成時 |

### 計算ロジック

スコープの統合ロジック:
- `own_oauth_scopes`: 当該Security依存関係で指定されたスコープ
- `parent_oauth_scopes`: 親の依存関係から継承されたスコープ
- `oauth_scopes`: `own_oauth_scopes + parent_oauth_scopes`の結合

## データベース操作仕様

本機能はデータベース操作を直接行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTPException 401 | 認証エラー | セキュリティ依存関数で認証失敗時 | 正しい認証情報を提供 |
| AssertionError | パラメータエラー | dependencyが呼び出し不可能 | 正しい依存関数を指定 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- Dependsと同様のキャッシュ機能により、同一リクエスト内での重複実行を防止
- スコープ情報の伝播はメモリ上で効率的に行う

## セキュリティ考慮事項

- scopesパラメータはOpenAPIに反映されるが、実際の検証は依存関数内で実装が必要
- SecurityScopesを使用して、現在のリクエストで必要なスコープを取得可能
- トークンの検証ロジックは依存関数内で適切に実装すること

## 備考

- SecurityはDependsを継承しており、scopesパラメータが追加されている
- 実際のスコープ検証はSecurityScopesを使用して依存関数内で実装する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Securityクラスの構造とDependsとの関係を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | Securityデータクラスの定義（753-756行目） |
| 1-2 | params.py | `fastapi/params.py` | Dependsの継承関係（746-750行目） |

**読解のコツ**: `Security`は`Depends`を継承し、`scopes`パラメータを追加している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | param_functions.py | `fastapi/param_functions.py` | Security関数の定義（2289-2369行目） |

**主要処理フロー**:
1. **2289-2332行目**: Security関数のパラメータ定義とdocstring
2. **2302-2317行目**: scopesパラメータの説明（OAuth2スコープ）
3. **2369行目**: params.Securityインスタンスの生成

#### Step 3: スコープの伝播処理

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.py | `fastapi/dependencies/utils.py` | get_parameterless_sub_dependant関数（112-124行目） |
| 3-2 | utils.py | `fastapi/dependencies/utils.py` | get_dependant関数内のスコープ処理（294-297行目） |

**主要処理フロー**:
- **116-118行目**: params.Securityの場合にスコープを抽出
- **294-297行目**: サブ依存関係のスコープをown_oauth_scopesに設定

#### Step 4: OpenAPIへの反映

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | utils.py | `fastapi/openapi/utils.py` | get_openapi_security_definitions関数（78-101行目） |

**主要処理フロー**:
- **84-91行目**: セキュリティ依存関係からスキーム定義を抽出
- **92-97行目**: 同一スキームのスコープをマージ
- **98-100行目**: OpenAPI形式のセキュリティ要件を構築

### プログラム呼び出し階層図

```
Security(dependency, scopes, use_cache)
    │
    └─ params.Security（データクラス生成）
           │
           └─ get_dependant()
                  │
                  ├─ analyze_param()
                  │      └─ params.Securityインスタンス検出
                  │
                  └─ スコープ処理
                         │
                         ├─ own_oauth_scopes設定
                         │
                         └─ get_flat_dependant()
                                │
                                └─ get_openapi_security_definitions()
                                       └─ OpenAPIスキーマ出力
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Security依存関係        params.Security              セキュリティ依存値
宣言               ───▶  インスタンス生成    ───▶   (トークン等)
  │                           │
  │                           ▼
scopes指定          get_dependant()                 OpenAPIスキーマ
["read", "write"]   own_oauth_scopes設定  ───▶   security要件
                           │
                           ▼
                   get_openapi_security_definitions()
                     スキーム定義とスコープ抽出
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| param_functions.py | `fastapi/param_functions.py` | ソース | Security関数の公開API定義 |
| params.py | `fastapi/params.py` | ソース | Securityデータクラスの定義 |
| utils.py | `fastapi/dependencies/utils.py` | ソース | 依存関係の検出・解決とスコープ処理 |
| utils.py | `fastapi/openapi/utils.py` | ソース | OpenAPIセキュリティ定義の生成 |
| oauth2.py | `fastapi/security/oauth2.py` | ソース | OAuth2セキュリティクラス群 |
| models.py | `fastapi/dependencies/models.py` | ソース | Dependantデータモデル |
