# 機能設計書 13-OAuth2認証

## 概要

本ドキュメントはFastAPIフレームワークにおけるOAuth2認証機能の設計仕様を定義する。

### 本機能の処理概要

OAuth2認証機能は、OAuth2プロトコルに基づく認証・認可を実装するためのクラス群を提供する。OAuth2PasswordBearer、OAuth2AuthorizationCodeBearerなど、OAuth2の各種フローに対応したクラスがあり、これらをDependsまたはSecurity依存関係として使用することで、エンドポイントに認証を適用できる。また、OAuth2PasswordRequestFormによりトークン発行エンドポイントの実装も容易に行える。

**業務上の目的・背景**：OAuth2は、Webアプリケーションやモバイルアプリケーションにおける認証・認可の業界標準プロトコルである。サードパーティサービスとの連携、シングルサインオン、APIのセキュアなアクセス制御など、多くのユースケースで使用される。FastAPIのOAuth2機能により、これらの標準的な認証フローを簡潔に実装できる。

**機能の利用シーン**：
- ユーザー認証（ログイン/ログアウト）
- JWTトークンベースの認証
- サードパーティサービスとのOAuth2連携
- APIのアクセス制御
- Swagger UIでのインタラクティブな認証テスト

**主要な処理内容**：
1. HTTPリクエストからAuthorizationヘッダーを取得
2. Bearerスキームの検証とトークンの抽出
3. 認証失敗時のHTTPException発生（auto_error=Trueの場合）
4. OpenAPIスキーマへのセキュリティスキーム定義の出力
5. トークン発行エンドポイント用のフォームデータ処理

**関連システム・外部連携**：
- OpenAPI: セキュリティスキームの定義
- Swagger UI: OAuth2認証フローのテスト
- JWT: トークンの生成・検証（ユーザー実装）

**権限による制御**：scopesパラメータでOAuth2スコープを定義し、アクセス制御を実現。実際のスコープ検証は開発者が実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 主機能 | OAuth2認証フローのテスト。トークン取得と保持 |
| 3 | OAuth2リダイレクト | 補助機能 | Authorization Codeフローでのリダイレクト処理 |

## 機能種別

セキュリティ / 認証・認可

## 入力仕様

### 入力パラメータ（OAuth2PasswordBearer）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| tokenUrl | str | Yes | トークン取得エンドポイントのURL | 有効なURL形式 |
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| scopes | dict[str, str] \| None | No | OAuth2スコープの定義 | キー：スコープ名、値：説明 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |
| refreshUrl | str \| None | No | トークン更新エンドポイントのURL | 有効なURL形式 |

### 入力パラメータ（OAuth2PasswordRequestForm）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| grant_type | str \| None | No | 認可タイプ（"password"） | パターン: ^password$ |
| username | str | Yes | ユーザー名 | 必須 |
| password | str | Yes | パスワード | 必須 |
| scope | str | No | スコープ（スペース区切り） | 空文字列可 |
| client_id | str \| None | No | クライアントID | 文字列 |
| client_secret | str \| None | No | クライアントシークレット | 文字列 |

### 入力データソース

- HTTPリクエストのAuthorizationヘッダー
- フォームデータ（トークン発行エンドポイント）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| token | str \| None | Bearerトークン文字列（認証成功時） |
| OAuth2PasswordRequestForm | オブジェクト | フォームから抽出されたユーザー名、パスワード、スコープ等 |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス（OAuth2PasswordBearer）

```
1. リクエスト受信
   └─ Authorization: Bearer <token> ヘッダーの確認
2. ヘッダー解析
   └─ get_authorization_scheme_param関数でスキームとトークンを分離
3. スキーム検証
   └─ "bearer"（大文字小文字区別なし）であることを確認
4. 結果判定
   ├─ 認証成功: トークン文字列を返却
   └─ 認証失敗: HTTPException 401（auto_error=True）またはNone（auto_error=False）
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Authorizationヘッダーあり?}
    B -->|No| C{auto_error?}
    B -->|Yes| D[スキームとトークンを分離]
    D --> E{スキームはbearer?}
    E -->|No| C
    E -->|Yes| F[トークンを返却]
    C -->|True| G[HTTPException 401発生]
    C -->|False| H[Noneを返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-13-01 | Bearerスキーム検証 | Authorizationヘッダーのスキームは"bearer"（大文字小文字区別なし）である必要がある | OAuth2PasswordBearer使用時 |
| BR-13-02 | 自動エラー | auto_error=Trueの場合、認証失敗時にHTTPException 401を発生させる | デフォルト動作 |
| BR-13-03 | スコープ分割 | OAuth2PasswordRequestFormのscopeはスペースで分割してリストに変換 | scope文字列の処理時 |
| BR-13-04 | WWW-Authenticateヘッダー | 認証エラー時はWWW-Authenticate: Bearerヘッダーを含める | HTTPException発生時 |

### 計算ロジック

スコープの分割:
```python
self.scopes = scope.split()  # スペース区切りでリストに変換
```

## データベース操作仕様

本機能はデータベース操作を直接行わない。トークンの保存・検証は開発者が実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | Authorizationヘッダーがない | 正しいトークンを送信 |
| HTTP 401 | Unauthorized | スキームがbearerでない | Bearer形式で送信 |
| HTTP 401 | Unauthorized | トークンが空 | 有効なトークンを送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- ヘッダー解析は文字列操作のみで高速
- 各リクエストで認証処理が実行されるため、軽量な実装が重要

## セキュリティ考慮事項

- トークンの検証ロジックは開発者が実装する必要がある
- JWTを使用する場合は、署名の検証、有効期限の確認、スコープの検証を行うこと
- パスワードはハッシュ化して保存すること
- HTTPSを使用してトークンの盗聴を防止すること
- auto_error=Falseの場合、認証の有無を明示的にチェックすること

## 備考

- OAuth2クラスはベースクラスであり、通常はOAuth2PasswordBearerやOAuth2AuthorizationCodeBearerを使用する
- 実際のトークン検証はユーザーの依存関数内で実装する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、OAuth2関連のモデルとクラス構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | models.py | `fastapi/openapi/models.py` | OAuth2モデル、OAuthFlowsモデルの定義（358-389行目） |
| 1-2 | base.py | `fastapi/security/base.py` | SecurityBaseクラスの定義（1-7行目） |

**読解のコツ**: OpenAPIモデルがセキュリティスキームの定義に使用され、SecurityBaseがFastAPIのセキュリティクラスの基底となる。

#### Step 2: OAuth2基底クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | oauth2.py | `fastapi/security/oauth2.py` | OAuth2クラスの定義（306-406行目） |

**主要処理フロー**:
1. **319-375行目**: __init__メソッドでOAuth2Modelを生成
2. **377-397行目**: make_not_authenticated_errorメソッドでエラー生成
3. **399-406行目**: __call__メソッドでAuthorizationヘッダーを取得

#### Step 3: OAuth2PasswordBearerを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | oauth2.py | `fastapi/security/oauth2.py` | OAuth2PasswordBearerクラス（409-514行目） |

**主要処理フロー**:
- **418-486行目**: __init__メソッドでOAuthFlowsModelを生成
- **487-498行目**: OAuthFlowsModel.passwordフローの設定
- **506-514行目**: __call__メソッドでBearerトークンを抽出

#### Step 4: OAuth2PasswordRequestFormを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | oauth2.py | `fastapi/security/oauth2.py` | OAuth2PasswordRequestFormクラス（14-148行目） |

**主要処理フロー**:
- **59-141行目**: __init__メソッドでフォームパラメータを定義
- **142-147行目**: インスタンス属性への値設定
- **145行目**: scopeをスペースで分割してリストに変換

### プログラム呼び出し階層図

```
OAuth2PasswordBearer(tokenUrl, scopes, ...)
    │
    └─ OAuth2.__init__()
           │
           └─ OAuth2Model(flows=OAuthFlowsModel)
                  │
                  └─ __call__(request)
                         │
                         ├─ request.headers.get("Authorization")
                         │
                         └─ get_authorization_scheme_param()
                                │
                                ├─ スキーム検証（"bearer"）
                                │
                                └─ トークン返却 or HTTPException
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          OAuth2PasswordBearer         トークン文字列
Authorization:    ───▶   __call__()          ───▶   または None
Bearer <token>              │
                            ▼
                   get_authorization_scheme_param()
                   スキームとトークン分離
                            │
                            ▼
                   スキーム検証（"bearer"）
                            │
                   ┌────────┴────────┐
                   ▼                 ▼
              認証成功           認証失敗
              token返却     HTTPException 401
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| oauth2.py | `fastapi/security/oauth2.py` | ソース | OAuth2認証クラス群の定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | OpenAPI OAuth2モデルの定義 |
| utils.py | `fastapi/security/utils.py` | ソース | get_authorization_scheme_param関数 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
