# 機能設計書 17-APIキー認証

## 概要

本ドキュメントはFastAPIフレームワークにおけるAPIキー認証機能の設計仕様を定義する。

### 本機能の処理概要

APIキー認証機能は、事前に発行されたAPIキーを使用してAPIアクセスを認証する機能を提供する。APIKeyQuery、APIKeyHeader、APIKeyCookieの3つのクラスがあり、それぞれクエリパラメータ、HTTPヘッダー、Cookieからキーを取得する。OpenAPI仕様に準拠したセキュリティスキーム定義も提供する。

**業務上の目的・背景**：APIキー認証は、シンプルで実装しやすい認証方式であり、マシン間通信、サードパーティAPI連携、レート制限の識別などで広く使用されている。ユーザー認証よりも軽量で、サービス間の認証や公開APIのアクセス制御に適している。

**機能の利用シーン**：
- サードパーティアプリケーションへのAPI提供
- マシン間（M2M）通信の認証
- APIレート制限のためのクライアント識別
- 開発者向けAPIキーの管理
- 内部サービス間の認証

**主要な処理内容**：
1. クエリパラメータ、ヘッダー、またはCookieからAPIキーを取得
2. キーの存在を確認
3. キー値を返却（検証は開発者が実装）
4. 認証失敗時のHTTPException発生（auto_error=Trueの場合）

**関連システム・外部連携**：
- OpenAPI: APIキーセキュリティスキームの定義
- Swagger UI: APIキー入力フォームの表示

**権限による制御**：APIキー認証自体は権限制御を行わない。キーに紐づく権限管理は開発者が実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | APIキー入力フォームの表示と認証テスト |

## 機能種別

セキュリティ / 認証

## 入力仕様

### 入力パラメータ（共通）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | str | Yes | APIキーのパラメータ名/ヘッダー名/Cookie名 | 必須、空文字列不可 |
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |

### 入力データソース

- APIKeyQuery: クエリパラメータ（例: `?api_key=xxx`）
- APIKeyHeader: HTTPヘッダー（例: `X-API-Key: xxx`）
- APIKeyCookie: Cookie（例: `session=xxx`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| api_key | str \| None | APIキー文字列 |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ 指定された場所（クエリ/ヘッダー/Cookie）からキーを取得
2. キー存在確認
   └─ check_api_keyメソッドで確認
3. 結果判定
   ├─ キーあり: キー値を返却
   └─ キーなし: HTTPException 401（auto_error=True）またはNone（auto_error=False）
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{取得場所の判定}
    B -->|APIKeyQuery| C[クエリパラメータ取得]
    B -->|APIKeyHeader| D[ヘッダー取得]
    B -->|APIKeyCookie| E[Cookie取得]
    C --> F{キーが存在?}
    D --> F
    E --> F
    F -->|Yes| G[キー値を返却]
    F -->|No| H{auto_error?}
    H -->|True| I[HTTPException 401発生]
    H -->|False| J[None返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-17-01 | キー位置指定 | nameパラメータで指定した名前のパラメータ/ヘッダー/Cookieからキーを取得 | APIKey*クラス使用時 |
| BR-17-02 | 自動エラー | auto_error=Trueの場合、キーが存在しないときHTTPException 401を発生させる | デフォルト動作 |
| BR-17-03 | WWW-Authenticateヘッダー | RFC 9110に従い、401エラー時はWWW-Authenticate: APIKeyヘッダーを含める | HTTPException発生時 |
| BR-17-04 | OpenAPI反映 | APIキーのin（query/header/cookie）とnameはOpenAPIスキーマに反映される | OpenAPIスキーマ生成時 |

### 計算ロジック

特になし。単純なキー値の取得処理のみ。

## データベース操作仕様

本機能はデータベース操作を直接行わない。APIキーの検証・管理は開発者が実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | 指定された場所にAPIキーが存在しない | 正しいAPIキーを送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- パラメータ/ヘッダー/Cookie取得のみの軽量処理
- 各リクエストで認証処理が実行される

## セキュリティ考慮事項

- APIキーの検証ロジックは開発者が実装する必要がある
- APIキーはデータベース等で安全に管理すること
- HTTPSを使用してキーの盗聴を防止すること
- キーのローテーション機能を検討すること
- クエリパラメータでの送信はURLに残る可能性があるため、ヘッダーまたはCookieの使用を推奨
- レート制限と組み合わせて使用することを推奨

## 備考

- 3つのクラス（APIKeyQuery、APIKeyHeader、APIKeyCookie）は共通の基底クラスAPIKeyBaseを継承
- OpenAPIのapiKeyセキュリティスキームに対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、APIキーのモデル定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | models.py | `fastapi/openapi/models.py` | APIKeyモデルの定義（342-346行目） |
| 1-2 | models.py | `fastapi/openapi/models.py` | APIKeyIn列挙型の定義（336-340行目） |

**読解のコツ**: `APIKeyIn`は`query`、`header`、`cookie`の3つの値を持つ。`APIKey`モデルはこれらの位置とキー名を保持する。

#### Step 2: 基底クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | api_key.py | `fastapi/security/api_key.py` | APIKeyBaseクラスの定義（11-51行目） |

**主要処理フロー**:
1. **12-28行目**: __init__メソッドでAPIKeyモデルを生成
2. **29-43行目**: make_not_authenticated_errorメソッドでエラー生成
3. **45-50行目**: check_api_keyメソッドでキーの存在を確認

#### Step 3: 各クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | api_key.py | `fastapi/security/api_key.py` | APIKeyQueryクラスの定義（53-143行目） |
| 3-2 | api_key.py | `fastapi/security/api_key.py` | APIKeyHeaderクラスの定義（145-231行目） |
| 3-3 | api_key.py | `fastapi/security/api_key.py` | APIKeyCookieクラスの定義（233-319行目） |

**主要処理フロー（APIKeyQuery）**:
- **85-138行目**: __init__メソッドでAPIKeyIn.queryを設定
- **140-142行目**: __call__メソッドでrequest.query_params.get()を実行

**主要処理フロー（APIKeyHeader）**:
- **177-226行目**: __init__メソッドでAPIKeyIn.headerを設定
- **228-230行目**: __call__メソッドでrequest.headers.get()を実行

**主要処理フロー（APIKeyCookie）**:
- **265-314行目**: __init__メソッドでAPIKeyIn.cookieを設定
- **316-318行目**: __call__メソッドでrequest.cookies.get()を実行

### プログラム呼び出し階層図

```
APIKeyQuery(name, ...)     APIKeyHeader(name, ...)    APIKeyCookie(name, ...)
       │                          │                          │
       └────────────────┬─────────┴──────────┬───────────────┘
                        │                    │
                        ▼                    ▼
                 APIKeyBase.__init__()     __call__(request)
                        │                    │
                        ▼                    │
                 APIKey(in=location,        ├─ query_params.get()
                        name=name)          ├─ headers.get()
                                            └─ cookies.get()
                                                    │
                                                    ▼
                                            check_api_key(api_key)
                                                    │
                                            ┌───────┴───────┐
                                            ▼               ▼
                                        キー返却      HTTPException 401
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          APIKeyQuery/Header/Cookie   APIキー文字列
?api_key=xxx      ───▶   __call__()           ───▶   または None
X-API-Key: xxx              │
Cookie: session=xxx         ▼
                   query_params/headers/cookies.get()
                            │
                            ▼
                   check_api_key()
                   キー存在確認
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| api_key.py | `fastapi/security/api_key.py` | ソース | APIキー認証クラス群の定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | APIKeyモデル、APIKeyIn列挙型の定義 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
