# 機能設計書 18-OpenID Connect

## 概要

本ドキュメントはFastAPIフレームワークにおけるOpenID Connect認証機能の設計仕様を定義する。

### 本機能の処理概要

OpenID Connect認証機能は、OpenID Connectプロトコルに基づく認証をサポートするためのスタブクラスを提供する。OpenIdConnectクラスをDependsまたはSecurity依存関係として使用することで、OpenAPI仕様にOpenID Connectセキュリティスキームを定義できる。ただし、完全なOpenID Connectフローの実装は含まれておらず、実際の認証ロジックは開発者が実装する必要がある。

**業務上の目的・背景**：OpenID Connectは、OAuth2を拡張した認証プロトコルであり、Google、Microsoft、Okta等の主要なIDプロバイダーで採用されている。シングルサインオン（SSO）の実現や、外部IDプロバイダーとの連携に使用される。FastAPIのOpenIdConnectクラスは、OpenAPI仕様との統合を容易にし、Swagger UIでのセキュリティスキーム表示をサポートする。

**機能の利用シーン**：
- 外部IDプロバイダー（Google、Microsoft、Okta等）との連携
- シングルサインオン（SSO）の実装
- エンタープライズ環境での認証統合
- OpenAPI仕様でのセキュリティスキーム定義

**主要な処理内容**：
1. HTTPリクエストからAuthorizationヘッダーを取得
2. 認証ヘッダーの存在を確認
3. 認証ヘッダーの値をそのまま返却
4. 認証失敗時のHTTPException発生（auto_error=Trueの場合）

**関連システム・外部連携**：
- OpenAPI: OpenID Connectセキュリティスキームの定義
- IDプロバイダー: Google、Microsoft、Okta等（ユーザー実装）
- JWT: IDトークンの検証（ユーザー実装）

**権限による制御**：OpenID Connect自体は権限制御を行わない。IDトークンに含まれるクレーム（claims）に基づくアクセス制御は開発者が実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | OpenID Connectセキュリティスキームの表示 |

## 機能種別

セキュリティ / 認証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| openIdConnectUrl | str | Yes | OpenID Connect設定URLi（.well-known/openid-configuration） | 必須、有効なURL形式 |
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |

### 入力データソース

- HTTPリクエストのAuthorizationヘッダー

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| authorization | str \| None | Authorizationヘッダーの値（トークン文字列） |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ Authorizationヘッダーの確認
2. ヘッダー検証
   └─ Authorizationヘッダーの存在を確認
3. 結果判定
   ├─ ヘッダーあり: ヘッダー値をそのまま返却
   └─ ヘッダーなし: HTTPException 401（auto_error=True）またはNone（auto_error=False）
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Authorizationヘッダーあり?}
    B -->|No| C{auto_error?}
    B -->|Yes| D[Authorizationヘッダー値を返却]
    C -->|True| E[HTTPException 401発生]
    C -->|False| F[None返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-18-01 | スタブ実装 | FastAPIのOpenIdConnectクラスはスタブであり、完全なOpenID Connectフローを実装していない | 常時 |
| BR-18-02 | ヘッダー透過 | Authorizationヘッダーの値をパースせずそのまま返却する | 認証成功時 |
| BR-18-03 | WWW-Authenticateヘッダー | 認証エラー時はWWW-Authenticate: Bearerヘッダーを含める | HTTPException発生時 |
| BR-18-04 | openIdConnectUrl | OpenAPI仕様に含まれるが、FastAPIでは検証に使用されない | OpenAPIスキーマ生成時 |

### 計算ロジック

特になし。単純なヘッダー取得処理のみ。

## データベース操作仕様

本機能はデータベース操作を直接行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | Authorizationヘッダーがない | 正しい認証情報を送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- ヘッダー取得のみの軽量処理
- 各リクエストで認証処理が実行される

## セキュリティ考慮事項

- FastAPIのOpenIdConnectクラスはスタブであり、実際のトークン検証は開発者が実装する必要がある
- IDトークンの署名検証、発行者（issuer）検証、有効期限確認を必ず実装すること
- HTTPSを使用してトークンの盗聴を防止すること
- IDプロバイダーのJWKS（JSON Web Key Set）を使用して署名を検証すること

## 備考

- OpenIdConnectクラスは警告にある通りスタブ実装であり、サブクラス化して完全な実装を行う必要がある
- python-joseやPyJWT等のライブラリを使用してIDトークンの検証を実装することを推奨

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、OpenID Connectのモデル定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | models.py | `fastapi/openapi/models.py` | OpenIdConnectモデルの定義（392-397行目） |

**読解のコツ**: `OpenIdConnect`モデルは`SecurityBase`を継承し、`openIdConnectUrl`フィールドを追加している。

#### Step 2: OpenIdConnectクラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | open_id_connect_url.py | `fastapi/security/open_id_connect_url.py` | OpenIdConnectクラスの定義（11-94行目） |

**主要処理フロー**:
1. **12-20行目**: クラスのdocstring（スタブ実装である旨の警告）
2. **22-78行目**: __init__メソッドでOpenIdConnectModelを生成
3. **74-77行目**: modelとscheme_name、auto_errorの設定
4. **80-85行目**: make_not_authenticated_errorメソッドでエラー生成
5. **87-94行目**: __call__メソッドでAuthorizationヘッダーを取得

#### Step 3: 認証フローを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | open_id_connect_url.py | `fastapi/security/open_id_connect_url.py` | __call__メソッド（87-94行目） |

**主要処理フロー**:
- **88行目**: Authorizationヘッダーの取得
- **89-93行目**: ヘッダーが存在しない場合の処理
- **94行目**: ヘッダー値の返却

### プログラム呼び出し階層図

```
OpenIdConnect(openIdConnectUrl, scheme_name, description, auto_error)
    │
    └─ __init__()
           │
           └─ OpenIdConnectModel(openIdConnectUrl, description)
                  │
                  └─ __call__(request)
                         │
                         ├─ request.headers.get("Authorization")
                         │
                         └─ 認証判定
                                │
                                ├─ ヘッダーあり → authorization返却
                                │
                                └─ ヘッダーなし → HTTPException 401 / None
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          OpenIdConnect                Authorizationヘッダー値
Authorization:    ───▶   __call__()           ───▶   または None
Bearer eyJhbG...            │
                            ▼
                   headers.get("Authorization")
                            │
                            ▼
                   存在確認
                            │
                   ┌────────┴────────┐
                   ▼                 ▼
              ヘッダーあり      ヘッダーなし
              値を返却      HTTPException 401
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| open_id_connect_url.py | `fastapi/security/open_id_connect_url.py` | ソース | OpenIdConnectクラスの定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | OpenIdConnectモデルの定義 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
