# 機能設計書 19-SecurityScopes

## 概要

本ドキュメントはFastAPIフレームワークにおけるSecurityScopes機能の設計仕様を定義する。

### 本機能の処理概要

SecurityScopes機能は、OAuth2のスコープ情報を依存関係チェーン内で取得・検証するための特別なクラスを提供する。依存関係のパラメータにSecurityScopesを定義することで、その依存関係を呼び出す全ての親依存関係から要求されるスコープを一箇所で取得できる。これにより、細かい粒度のアクセス制御を実現できる。

**業務上の目的・背景**：OAuth2認証では、スコープによってアクセス権限を細かく制御する。例えば「read」スコープは読み取りのみ、「write」スコープは書き込みも許可するといった使い方が一般的である。複数の依存関係が異なるスコープを要求する場合、SecurityScopesを使用することで、全ての必要なスコープを集約し、一括して検証できる。

**機能の利用シーン**：
- OAuth2スコープに基づくきめ細かいアクセス制御
- 依存関係チェーン内でのスコープ要件の集約
- トークンに含まれるスコープと要求スコープの比較検証
- スコープ不足時のエラーメッセージ生成

**主要な処理内容**：
1. 依存関係チェーンから要求されるスコープを収集
2. スコープリストとスコープ文字列（スペース区切り）として提供
3. 開発者が実装するスコープ検証ロジックで使用

**関連システム・外部連携**：
- Security依存関係: スコープ情報の伝播元
- OAuth2認証: トークン内のスコープとの比較
- Depends機能: 依存関係解決時のスコープ注入

**権限による制御**：SecurityScopesに収集されたスコープと、ユーザーのトークンに含まれるスコープを比較することで、アクセス制御を実現する（検証ロジックは開発者が実装）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | Security依存関係で定義されたスコープの表示 |

## 機能種別

セキュリティ / 認可

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| scopes | list[str] \| None | No | スコープのリスト（FastAPIが自動設定） | リスト形式 |

### 入力データソース

- Security依存関係で定義されたscopesパラメータ
- 依存関係チェーンを通じて伝播されるスコープ情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| scopes | list[str] | 依存関係チェーンで要求される全てのスコープのリスト |
| scope_str | str | スコープをスペース区切りで連結した文字列 |

### 出力先

- パスオペレーション関数のパラメータ

## 処理フロー

### 処理シーケンス

```
1. Security依存関係の解析
   └─ Security(dependency, scopes=[...])のスコープを抽出
2. スコープの伝播
   └─ 依存関係チェーンに沿ってスコープを伝播
3. スコープの集約
   └─ get_flat_dependantでparent_oauth_scopesとown_oauth_scopesを結合
4. SecurityScopesの注入
   └─ 依存関数のSecurityScopesパラメータにスコープ情報を注入
```

### フローチャート

```mermaid
flowchart TD
    A[Security依存関係1 scopes=read] --> B[Security依存関係2 scopes=write]
    B --> C[依存関数 security_scopes: SecurityScopes]
    C --> D[security_scopes.scopes = read, write]
    D --> E{トークンのスコープと比較}
    E -->|全スコープあり| F[アクセス許可]
    E -->|スコープ不足| G[HTTPException 401/403]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-19-01 | スコープ集約 | 依存関係チェーン内の全てのSecurity依存関係のスコープが集約される | SecurityScopesパラメータ使用時 |
| BR-19-02 | 自動注入 | SecurityScopesはFastAPIが自動的に値を注入する | 依存関数でSecurityScopes型を宣言時 |
| BR-19-03 | スペース区切り | scope_strはOAuth2仕様に従いスペース区切りで連結される | scope_strプロパティ参照時 |

### 計算ロジック

スコープの集約:
```python
# get_flat_dependantで実行
use_parent_oauth_scopes = (parent_oauth_scopes or []) + (dependant.oauth_scopes or [])
# SecurityScopesインスタンス生成時
scope_str = " ".join(scopes)  # スペース区切りで連結
```

## データベース操作仕様

本機能はデータベース操作を直接行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | 認証なし（スコープ検証前） | 正しい認証情報を送信 |
| HTTP 403 | Forbidden | スコープ不足（ユーザー実装） | 必要なスコープを持つトークンを取得 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- スコープの集約はメモリ上のリスト操作のみで軽量
- 依存関係解決時に一度だけ計算される

## セキュリティ考慮事項

- SecurityScopesで取得したスコープとトークン内のスコープの比較検証は開発者が実装する必要がある
- スコープ不足時は適切なエラー（401または403）を返すこと
- スコープの検証漏れがないよう、全てのエンドポイントで適切にスコープを定義すること
- エラーメッセージにscope_strを含めることで、クライアントに必要なスコープを通知できる

## 備考

- SecurityScopesはFastAPIの特別なクラスであり、通常のPydanticモデルとは異なる
- パラメータ名は任意だが、型アノテーションがSecurityScopesであることが重要
- 依存関数内でSecurityScopesを使用しない場合でも、スコープはOpenAPIスキーマに反映される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、SecurityScopesクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | oauth2.py | `fastapi/security/oauth2.py` | SecurityScopesクラスの定義（623-663行目） |

**読解のコツ**: `SecurityScopes`は`scopes`（リスト）と`scope_str`（スペース区切り文字列）の2つのプロパティを持つ。

#### Step 2: スコープの伝播処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | utils.py | `fastapi/dependencies/utils.py` | get_dependant関数内のスコープ処理（251-300行目） |
| 2-2 | utils.py | `fastapi/dependencies/utils.py` | get_flat_dependant関数（127-178行目） |

**主要処理フロー**:
- **137-139行目**: parent_oauth_scopesとown_oauth_scopesの結合
- **155行目**: Dependantオブジェクトのown_oauth_scopesとparent_oauth_scopes設定
- **294-297行目**: Security依存関係のスコープをsub_own_oauth_scopesに設定

#### Step 3: スコープの注入処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.py | `fastapi/dependencies/utils.py` | solve_dependencies関数（約690行目付近） |

**主要処理フロー**:
- **690-693行目**: security_scopes_param_nameがある場合、SecurityScopesインスタンスを生成して注入

### プログラム呼び出し階層図

```
Security(dependency, scopes=["read", "write"])
    │
    └─ get_dependant()
           │
           └─ analyze_param()
                  └─ params.Security検出
                         │
                         └─ sub_own_oauth_scopes設定
                                │
                                └─ get_flat_dependant()
                                       │
                                       └─ parent_oauth_scopes + own_oauth_scopes
                                              │
                                              └─ solve_dependencies()
                                                     │
                                                     └─ SecurityScopes(scopes=oauth_scopes)
                                                            │
                                                            └─ 依存関数に注入
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Security依存関係         スコープ収集                 SecurityScopes
scopes=["read"]    ───▶  get_flat_dependant()  ───▶  scopes=["read", "write"]
    │                         │                       scope_str="read write"
    ▼                         ▼
Security依存関係         親子スコープ結合
scopes=["write"]        parent + own
    │
    ▼
依存関数                 solve_dependencies()
security_scopes:   ◀──── SecurityScopes注入
SecurityScopes
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| oauth2.py | `fastapi/security/oauth2.py` | ソース | SecurityScopesクラスの定義 |
| utils.py | `fastapi/dependencies/utils.py` | ソース | スコープの伝播と注入処理 |
| models.py | `fastapi/dependencies/models.py` | ソース | Dependantクラス（oauth_scopes保持） |
| params.py | `fastapi/params.py` | ソース | Securityデータクラス（scopesフィールド） |
