# 機能設計書 2-APIRouter

## 概要

本ドキュメントは、FastAPIフレームワークにおけるAPIRouterクラスの機能設計を記述する。パスオペレーション（エンドポイント）をグループ化して管理し、アプリケーションをモジュール化・構造化するためのルーティング機能を提供する。

### 本機能の処理概要

APIRouterは、FastAPIアプリケーションにおいてパスオペレーション（HTTPエンドポイント）を論理的にグループ化し、管理するためのクラスである。複数のルーターを組み合わせることで、大規模アプリケーションをモジュール単位で分割し、保守性と再利用性を向上させる。

**業務上の目的・背景**：大規模なAPIアプリケーションでは、すべてのエンドポイントを単一ファイルに記述することは保守性の低下を招く。APIRouterを活用することで、機能単位（ユーザー管理、商品管理、注文管理等）でエンドポイントを分離し、チーム開発における担当範囲の明確化、コードの再利用性向上、テストの容易化を実現する。これにより、マイクロサービス的なアプローチを単一アプリケーション内で採用可能となる。

**機能の利用シーン**：
- 大規模アプリケーションを複数ファイル・モジュールに分割する際
- 共通のプレフィックス・タグ・依存性を持つエンドポイント群を定義する際
- 機能単位でAPIを開発・テストする際
- 再利用可能なAPIモジュールを作成する際

**主要な処理内容**：
1. ルーターインスタンスの初期化とプレフィックス・タグ等の設定
2. @router.get()、@router.post()等のデコレータによるパスオペレーション登録
3. 依存性・レスポンスクラス・タグの継承と結合
4. FastAPIアプリケーションまたは他のルーターへのインクルード
5. OpenAPIスキーマへのルート情報の反映

**関連システム・外部連携**：FastAPIアプリケーション（include_routerによる統合）、Starletteルーター（基底クラス）、Pydantic（パラメータバリデーション）

**権限による制御**：ルーターレベルで共通の依存性（dependencies引数）を設定することで、配下の全エンドポイントに認証・認可処理を一括適用可能

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | OpenAPI JSON | 補助機能 | 登録されたルーター・エンドポイント情報をOpenAPIスキーマに含める |

## 機能種別

ルーティング管理 / エンドポイントグループ化 / モジュール構造化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| prefix | str | No | ルーターのパスプレフィックス（デフォルト: ""） | "/"で始まり"/"で終わらないこと |
| tags | Optional[list[Union[str, Enum]]] | No | OpenAPIタグリスト | 文字列またはEnumのリスト |
| dependencies | Optional[Sequence[Depends]] | No | 共通依存性リスト | Dependsインスタンスのシーケンス |
| default_response_class | type[Response] | No | デフォルトレスポンスクラス（デフォルト: JSONResponse） | Responseのサブクラス |
| responses | Optional[dict] | No | 追加レスポンス定義（OpenAPI用） | ステータスコードとレスポンス定義の辞書 |
| callbacks | Optional[list[BaseRoute]] | No | OpenAPIコールバック定義 | BaseRouteのリスト |
| redirect_slashes | bool | No | スラッシュリダイレクト有効化（デフォルト: True） | ブール値 |
| route_class | type[APIRoute] | No | カスタムルートクラス（デフォルト: APIRoute） | APIRouteのサブクラス |
| deprecated | Optional[bool] | No | 全エンドポイントを非推奨としてマーク | ブール値またはNone |
| include_in_schema | bool | No | OpenAPIスキーマへの含有（デフォルト: True） | ブール値 |
| generate_unique_id_function | Callable[[APIRoute], str] | No | ユニークID生成関数 | APIRouteを受け取りstrを返す関数 |
| lifespan | Optional[Lifespan] | No | ライフスパンコンテキストマネージャ | Lifespanインスタンス |

### 入力データソース

- コンストラクタ引数（開発者による直接指定）
- パスオペレーションデコレータ（@router.get()等）
- include_router()メソッド呼び出し時の引数

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| routes | list[BaseRoute] | 登録済みルート（APIRoute/APIWebSocketRoute）一覧 |
| prefix | str | 設定されたパスプレフィックス |
| tags | list[Union[str, Enum]] | 設定されたタグ一覧 |

### 出力先

- FastAPIアプリケーション（include_router()経由）
- 他のAPIRouter（ネストされたルーター構成）
- OpenAPIスキーマ

## 処理フロー

### 処理シーケンス

```
1. APIRouterインスタンス生成
   └─ プレフィックス、タグ、依存性等の初期設定
2. パスオペレーション登録
   └─ @router.get(), @router.post()等のデコレータ適用
3. ルート情報の構築
   └─ APIRouteインスタンス生成とroutes配列への追加
4. FastAPIアプリへのインクルード
   └─ app.include_router(router)による統合
5. ルート情報のマージ
   └─ プレフィックス結合、依存性・タグの継承
```

### フローチャート

```mermaid
flowchart TD
    A[APIRouter初期化] --> B[prefix/tags/dependencies設定]
    B --> C[パスオペレーション定義]
    C --> D{@router.get/post等}
    D --> E[add_api_route呼び出し]
    E --> F[APIRoute生成]
    F --> G[routes配列に追加]
    G --> H{他のエンドポイント?}
    H -->|Yes| C
    H -->|No| I[app.include_router]
    I --> J[ルートマージ]
    J --> K[OpenAPIスキーマ反映]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | プレフィックス検証 | prefixは"/"で始まり"/"で終わらないこと | prefix指定時 |
| BR-002 | タグ継承 | ルーターのタグはパスオペレーションのタグと結合される | tags設定時 |
| BR-003 | 依存性継承 | ルーターの依存性はパスオペレーションの依存性の前に適用される | dependencies設定時 |
| BR-004 | レスポンス結合 | ルーターのresponsesとパスオペレーションのresponsesがマージされる | responses設定時 |
| BR-005 | deprecated継承 | ルーターがdeprecatedの場合、配下の全エンドポイントもdeprecated | deprecated=True設定時 |

### 計算ロジック

- パス結合: `self.prefix + path`（add_api_route内）
- タグ結合: ルーターtags + パスオペレーションtags
- 依存性結合: ルーターdependencies + パスオペレーションdependencies

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | AssertionError | prefixが"/"で始まらない | prefixを"/"で開始するよう修正 |
| - | AssertionError | prefixが"/"で終わる | prefixから末尾の"/"を削除 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ルーター初期化とルート登録は同期的に実行される
- 実行時のパスマッチングはStarletteのルーティングエンジンを使用

## セキュリティ考慮事項

- dependencies引数を活用して認証・認可を一元管理することを推奨
- include_in_schema=Falseで内部APIをOpenAPIから除外可能
- 機密性の高いエンドポイントは専用ルーターに分離し、アクセス制御を明確化

## 備考

- ルーターは何重にもネスト可能（router.include_router(sub_router)）
- WebSocketエンドポイントも@router.websocket()で登録可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ルーティングに関連する型定義とデータ構造を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | Dependsクラスの定義を確認 |
| 1-2 | types.py | `fastapi/types.py` | DecoratedCallable、IncEx型の定義 |

**読解のコツ**: Dependsはdataclassとして定義されており、dependency（依存関数）、use_cache（キャッシュ使用）、scope（スコープ）を保持する。

#### Step 2: エントリーポイントを理解する

APIRouterクラスの定義と初期化処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | routing.py | `fastapi/routing.py` | APIRouterクラス定義（664-930行目） |

**主要処理フロー**:
1. **664-689行目**: APIRouterクラスのドキュメントと使用例
2. **691-905行目**: __init__メソッドのパラメータ定義
3. **906-929行目**: 親クラス初期化と属性設定
4. **914-918行目**: prefixのバリデーション（"/"開始・非終了チェック）

#### Step 3: パスオペレーション登録を理解する

ルートの追加処理を詳細に確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | add_api_route()メソッド（950-1030行目） |
| 3-2 | routing.py | `fastapi/routing.py` | api_route()デコレータ（1032-1092行目） |

**主要処理フロー**:
- **984-988行目**: route_classの決定とレスポンスクラス取得
- **990-1001行目**: タグ・依存性・コールバックの結合処理
- **1002-1029行目**: APIRouteインスタンス生成
- **1030行目**: routes配列への追加

#### Step 4: HTTPメソッド別デコレータを理解する

GET/POST等のショートカットメソッドを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | routing.py | `fastapi/routing.py` | get(), post(), put(), delete()等のメソッド |

### プログラム呼び出し階層図

```
APIRouter (routing.py)
    │
    ├─ Router (starlette/routing.py) [継承]
    │
    ├─ @router.get() / @router.post() 等
    │      └─ api_route() デコレータ
    │             └─ add_api_route()
    │                    └─ APIRoute() インスタンス生成
    │                           └─ routes.append()
    │
    ├─ add_api_websocket_route()
    │      └─ APIWebSocketRoute() インスタンス生成
    │
    └─ include_router() [ネスト時]
           └─ 子ルーターのルートをマージ
```

### データフロー図

```
[入力]                     [処理]                           [出力]

@router.get("/items")
         │
         ▼
    api_route()
         │
         ▼
  add_api_route()
         │
         ├─ タグ結合: router.tags + endpoint.tags
         ├─ 依存性結合: router.deps + endpoint.deps
         └─ パス結合: router.prefix + path
         │
         ▼
    APIRoute()
         │
         ▼
  routes.append() ───▶ router.routes配列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| routing.py | `fastapi/routing.py` | ソース | APIRouter、APIRoute、APIWebSocketRoute定義 |
| params.py | `fastapi/params.py` | ソース | Depends、Securityクラス定義 |
| applications.py | `fastapi/applications.py` | ソース | FastAPI.include_router()メソッド |
| types.py | `fastapi/types.py` | ソース | 型定義 |
| utils.py | `fastapi/utils.py` | ソース | generate_unique_id、get_value_or_default等 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | 依存性解決ユーティリティ |
