# 機能設計書 24-HTTPException

## 概要

本ドキュメントは、FastAPIフレームワークが提供するHTTPException例外クラスの設計について記載する。

### 本機能の処理概要

HTTPExceptionは、APIエンドポイント内でHTTPエラーレスポンスを返すために使用する例外クラスである。開発者はこの例外をraiseすることで、特定のHTTPステータスコード、エラー詳細メッセージ、カスタムヘッダーを含むエラーレスポンスをクライアントに返却できる。

**業務上の目的・背景**：REST APIにおいて、正常系以外の処理結果（リソース未検出、認証失敗、権限不足、バリデーションエラー等）を適切にクライアントに伝達する必要がある。HTTPExceptionは、HTTPプロトコルの標準的なステータスコードとJSON形式のエラー詳細を組み合わせることで、クライアントが適切にエラーハンドリングを行えるようにする。

**機能の利用シーン**：
- リソースが見つからない場合（404 Not Found）
- 認証が必要な場合（401 Unauthorized）
- 権限がない場合（403 Forbidden）
- リクエストが不正な場合（400 Bad Request）
- サーバーエラーをクライアントに伝える場合（5xx系）

**主要な処理内容**：
1. HTTPExceptionインスタンスの生成（status_code、detail、headersの設定）
2. 例外のraise
3. FastAPIの例外ハンドラによるキャッチ
4. JSONResponseまたはResponseとしてクライアントに返却

**関連システム・外部連携**：
- StarletteのHTTPException（基底クラス）
- FastAPIの例外ハンドラ機能

**権限による制御**：本機能自体は例外クラスの定義であり、権限制御は含まれない。権限エラーを通知する手段として使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | API全般のエラーレスポンス生成に使用 |

## 機能種別

例外処理 / エラーレスポンス生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| status_code | int | Yes | HTTPステータスコード | 有効なHTTPステータスコード |
| detail | Any | No | エラー詳細情報 | デフォルト: None |
| headers | Optional[dict[str, str]] | No | レスポンスに追加するヘッダー | デフォルト: None |

### 入力データソース

- 開発者がエンドポイント関数内で明示的に指定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| status_code | int | HTTPステータスコード |
| body | dict | {"detail": detail}形式のJSONボディ（ボディが許可されるステータスの場合） |
| headers | dict | カスタムヘッダー（指定された場合） |

### 出力先

- HTTPレスポンス（JSONResponse または Response）

## 処理フロー

### 処理シーケンス

```
1. 開発者がエンドポイント関数内でHTTPExceptionをraise
   └─ status_code、detail、headersを指定
2. FastAPIの例外ハンドラ（http_exception_handler）がキャッチ
   └─ exception_handlers辞書から対応するハンドラを取得
3. ステータスコードの判定
   └─ ボディが許可されるステータスコードかチェック
4. レスポンス生成
   └─ 許可される場合: JSONResponse({"detail": detail})
   └─ 許可されない場合: Response(status_code=status_code)
5. クライアントへレスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[エンドポイント関数実行] --> B{エラー条件発生?}
    B -->|No| C[正常レスポンス返却]
    B -->|Yes| D[HTTPException raise]
    D --> E[http_exception_handler実行]
    E --> F{ボディ許可ステータス?}
    F -->|Yes| G[JSONResponse生成]
    F -->|No| H[空Response生成]
    G --> I{headers指定あり?}
    H --> I
    I -->|Yes| J[ヘッダー追加]
    I -->|No| K[レスポンス返却]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-1 | ボディ制限 | 1xx、204、304などのステータスコードではレスポンスボディを含めない | is_body_allowed_for_status_code()がFalseの場合 |
| BR-24-2 | detail形式 | detailはJSON互換の任意の型を指定可能 | 常時 |
| BR-24-3 | デフォルトハンドラ | FastAPIはHTTPException用のデフォルト例外ハンドラを自動登録 | FastAPI初期化時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベースアクセスを行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | リクエストが不正 | リクエスト内容を修正 |
| 401 | Unauthorized | 認証が必要 | 認証情報を提供 |
| 403 | Forbidden | 権限がない | 適切な権限を取得 |
| 404 | Not Found | リソースが存在しない | リソースIDを確認 |
| 422 | Unprocessable Entity | バリデーションエラー | 入力値を修正 |
| 500 | Internal Server Error | サーバーエラー | ログを確認し対処 |

### リトライ仕様

該当なし（例外定義のためリトライロジックは含まれない）

## トランザクション仕様

該当なし（例外発生時はトランザクションロールバックが発生することがある）

## パフォーマンス要件

- 例外生成とハンドリングは即時完了

## セキュリティ考慮事項

- detailに機密情報（スタックトレース、内部パス等）を含めないよう注意
- 本番環境では詳細なエラー情報の公開を制限すべき

## 備考

- HTTPExceptionはクライアントエラーに使用し、サーバー内部エラーには使用しない想定
- StarletteのHTTPExceptionを継承しており、完全な互換性がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

HTTPException例外クラスの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | exceptions.py | `fastapi/exceptions.py` | HTTPExceptionクラスの定義 |

**主要処理フロー**:
- **17-73行目**: HTTPExceptionクラス定義
- **18-43行目**: クラスドキュメント（使用例含む）
- **45-73行目**: __init__メソッド（status_code、detail、headers）

**読解のコツ**: HTTPExceptionはStarletteのHTTPExceptionを継承しているため、基底クラスの振る舞いも確認するとよい。Annotatedによるパラメータドキュメントが詳細に記載されている。

#### Step 2: 例外ハンドラを理解する

http_exception_handler関数の処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | exception_handlers.py | `fastapi/exception_handlers.py` | http_exception_handler関数 |

**主要処理フロー**:
- **11-17行目**: http_exception_handler関数
- **12行目**: ヘッダーの取得
- **13-14行目**: ボディ許可チェックと空Response返却
- **15-17行目**: JSONResponse返却

#### Step 3: 例外ハンドラ登録を理解する

FastAPIアプリケーションでの例外ハンドラ登録を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | applications.py | `fastapi/applications.py` | 例外ハンドラの登録処理 |

**主要処理フロー**:
- **979-984行目**: exception_handlersの初期化とデフォルトハンドラ登録

#### Step 4: ユーティリティ関数を理解する

ボディ許可チェック関数を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | utils.py | `fastapi/utils.py` | is_body_allowed_for_status_code関数 |

### プログラム呼び出し階層図

```
エンドポイント関数
    │
    └─ raise HTTPException(status_code, detail, headers)
           │
           └─ [FastAPIミドルウェア]
                  │
                  └─ http_exception_handler(request, exc)
                         │
                         ├─ is_body_allowed_for_status_code(status_code)
                         │      └─ True: JSONResponse({"detail": detail})
                         │      └─ False: Response(status_code=status_code)
                         │
                         └─ headers追加（指定時）
```

### データフロー図

```
[入力]                      [処理]                           [出力]

raise HTTPException         http_exception_handler
- status_code ─────────────▶ ボディ許可チェック
- detail                     ↓
- headers                   レスポンス生成 ─────────────────▶ HTTPレスポンス
                             ↓                              - status_code
                            ヘッダー追加                     - body (JSON)
                                                            - headers
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException例外クラスの定義 |
| exception_handlers.py | `fastapi/exception_handlers.py` | ソース | http_exception_handler関数 |
| applications.py | `fastapi/applications.py` | ソース | 例外ハンドラの登録 |
| utils.py | `fastapi/utils.py` | ソース | is_body_allowed_for_status_code関数 |
| responses.py | `starlette/responses.py` | 外部依存 | JSONResponse、Responseクラス |
| exceptions.py | `starlette/exceptions.py` | 外部依存 | StarletteHTTPException（基底クラス） |
