# 機能設計書 26-RequestValidationError

## 概要

本ドキュメントは、FastAPIフレームワークが提供するRequestValidationError例外クラスの設計について記載する。

### 本機能の処理概要

RequestValidationErrorは、リクエストデータのバリデーションに失敗した際に発生する例外クラスである。パスパラメータ、クエリパラメータ、ヘッダー、ボディなどのリクエストデータがPydanticモデルや型ヒントで定義された制約に適合しない場合に自動的に発生し、詳細なエラー情報をクライアントに返却する。

**業務上の目的・背景**：REST APIにおいて、クライアントから送信されたデータの妥当性検証は重要な処理である。不正なデータがビジネスロジックに到達する前に検出し、クライアントに具体的なエラー内容（どのフィールドがどのように不正か）を通知することで、開発効率とユーザー体験を向上させる。

**機能の利用シーン**：
- 必須パラメータが欠落している場合
- データ型が一致しない場合（文字列を数値フィールドに送信等）
- 値が許容範囲外の場合（最小値/最大値制約違反等）
- 正規表現パターンに適合しない場合
- カスタムバリデータがエラーを返した場合

**主要な処理内容**：
1. リクエストデータの型変換・バリデーション実行
2. バリデーションエラーの検出
3. RequestValidationErrorインスタンスの生成（エラーリスト、ボディ、エンドポイントコンテキスト）
4. デフォルト例外ハンドラによる422レスポンス生成

**関連システム・外部連携**：
- Pydanticバリデーション機能
- FastAPI依存性注入システム
- 例外ハンドラ機能

**権限による制御**：本機能自体は例外クラスの定義であり、権限制御は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | API全般のリクエストバリデーションエラー処理に使用 |

## 機能種別

例外処理 / バリデーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| errors | Sequence[Any] | Yes | バリデーションエラーのリスト | - |
| body | Any | No | リクエストボディ（参照用） | デフォルト: None |
| endpoint_ctx | Optional[EndpointContext] | No | エンドポイントのコンテキスト情報 | デフォルト: None |

### 入力データソース

- Pydanticバリデーションエラー
- FastAPIパラメータ解決処理

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| status_code | int | 422 Unprocessable Entity |
| detail | list | バリデーションエラーの詳細リスト |

### 出力先

- HTTPレスポンス（JSONResponse）

## 処理フロー

### 処理シーケンス

```
1. クライアントからリクエスト受信
   └─ パスパラメータ、クエリ、ヘッダー、ボディ等のデータ
2. FastAPIによるパラメータ解決・バリデーション
   └─ Pydanticモデルへの変換、型チェック、制約検証
3. バリデーションエラー検出
   └─ エラーがある場合、RequestValidationErrorを生成
4. 例外ハンドラによる処理
   └─ request_validation_exception_handler関数が呼び出される
5. JSONResponse生成
   └─ status_code=422、detail=エラーリスト
6. クライアントへレスポンス返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B[パラメータ解決開始]
    B --> C{型変換成功?}
    C -->|No| D[バリデーションエラー追加]
    C -->|Yes| E{制約検証成功?}
    E -->|No| D
    E -->|Yes| F{全パラメータ処理完了?}
    F -->|No| B
    F -->|Yes| G{エラーあり?}
    D --> F
    G -->|Yes| H[RequestValidationError raise]
    G -->|No| I[エンドポイント関数実行]
    H --> J[request_validation_exception_handler]
    J --> K[JSONResponse 422]
    K --> L[レスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-26-1 | ステータスコード固定 | RequestValidationErrorは常に422 Unprocessable Entityを返す | 常時 |
| BR-26-2 | エラー形式 | errorsはPydanticバリデーションエラー形式のリスト | 常時 |
| BR-26-3 | デフォルトハンドラ | FastAPIはRequestValidationError用のデフォルト例外ハンドラを自動登録 | FastAPI初期化時 |
| BR-26-4 | エンドポイントコンテキスト | デバッグ用にエンドポイントのファイル・行番号情報を保持 | 例外発生時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベースアクセスを行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 422 | Unprocessable Entity | リクエストバリデーション失敗 | エラー詳細を確認しリクエストを修正 |

### リトライ仕様

該当なし（クライアント側でリクエストを修正して再送信）

## トランザクション仕様

該当なし（バリデーションはエンドポイント関数実行前に行われる）

## パフォーマンス要件

- バリデーション処理は同期的に実行され即時完了

## セキュリティ考慮事項

- エラー詳細には入力値が含まれる場合があるため、機密データの露出に注意
- カスタム例外ハンドラで詳細情報を制限することを検討

## 備考

- ValidationExceptionクラスを継承
- body属性でリクエストボディへの参照を保持
- endpoint_ctx属性でエンドポイントのコンテキスト情報（関数名、パス、ファイル、行番号）を保持

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ValidationExceptionとRequestValidationErrorの継承関係と構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | exceptions.py | `fastapi/exceptions.py` | ValidationException基底クラス |
| 1-2 | exceptions.py | `fastapi/exceptions.py` | RequestValidationErrorクラス |
| 1-3 | exceptions.py | `fastapi/exceptions.py` | EndpointContext型定義 |

**主要処理フロー**:
- **10-15行目**: EndpointContext TypedDict定義（function、path、file、line）
- **164-200行目**: ValidationException基底クラス
- **165-172行目**: __init__メソッド（errors、endpoint_ctx）
- **174-179行目**: エンドポイントコンテキスト属性の設定
- **180-181行目**: errors()メソッド
- **183-199行目**: エラーメッセージフォーマット
- **202-211行目**: RequestValidationErrorクラス
- **203-211行目**: __init__メソッド（errors、body、endpoint_ctx）

**読解のコツ**: RequestValidationErrorはValidationExceptionを継承し、追加でbody属性を持つ。endpoint_ctxによりデバッグ時にエラー発生元のエンドポイント情報を取得できる。

#### Step 2: 例外ハンドラを理解する

request_validation_exception_handler関数の処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | exception_handlers.py | `fastapi/exception_handlers.py` | request_validation_exception_handler関数 |

**主要処理フロー**:
- **20-26行目**: request_validation_exception_handler関数
- **23-25行目**: JSONResponse生成（status_code=422、detail=エラーリスト）

#### Step 3: 例外ハンドラ登録を理解する

FastAPIアプリケーションでの例外ハンドラ登録を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | applications.py | `fastapi/applications.py` | 例外ハンドラの登録処理 |

**主要処理フロー**:
- **983-984行目**: RequestValidationError用のデフォルトハンドラ登録

### プログラム呼び出し階層図

```
リクエスト受信
    │
    └─ FastAPIルーティング
           │
           └─ 依存性解決・パラメータバリデーション
                  │
                  ├─ Pydanticバリデーション
                  │
                  └─ バリデーションエラー発生
                         │
                         └─ raise RequestValidationError(errors, body, endpoint_ctx)
                                │
                                └─ request_validation_exception_handler(request, exc)
                                       │
                                       ├─ jsonable_encoder(exc.errors())
                                       │
                                       └─ JSONResponse(status_code=422, content={"detail": ...})
```

### データフロー図

```
[入力]                      [処理]                           [出力]

リクエストデータ            パラメータ解決
- path params              ↓
- query params ───────────▶ Pydanticバリデーション
- headers                   ↓
- body                     バリデーションエラー検出
                            ↓
                           RequestValidationError
                            ↓
                           request_validation_exception_handler
                            ↓
                           JSONResponse ──────────────────▶ HTTPレスポンス 422
                                                           {"detail": [...]}
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| exceptions.py | `fastapi/exceptions.py` | ソース | ValidationException、RequestValidationErrorクラス定義 |
| exception_handlers.py | `fastapi/exception_handlers.py` | ソース | request_validation_exception_handler関数 |
| applications.py | `fastapi/applications.py` | ソース | 例外ハンドラの登録 |
| encoders.py | `fastapi/encoders.py` | ソース | jsonable_encoder関数 |
| routing.py | `fastapi/routing.py` | ソース | パラメータ解決・バリデーション処理 |
| responses.py | `starlette/responses.py` | 外部依存 | JSONResponseクラス |
