# 機能設計書 28-例外ハンドラ

## 概要

本ドキュメントは、FastAPIフレームワークが提供する例外ハンドラ機能の設計について記載する。

### 本機能の処理概要

例外ハンドラは、アプリケーション内で発生した例外をキャッチし、適切なHTTPレスポンスに変換する機能である。FastAPIは、HTTPException、RequestValidationError、WebSocketRequestValidationError用のデフォルト例外ハンドラを提供し、開発者はカスタム例外ハンドラを登録して例外処理をカスタマイズできる。

**業務上の目的・背景**：APIアプリケーションにおいて、例外発生時の一貫したエラーレスポンス形式を提供することは、クライアント側のエラーハンドリング実装を容易にし、APIの使いやすさを向上させる。また、例外処理を一元化することで、エラーログの記録、通知、メトリクス収集などの横断的関心事を効率的に実装できる。

**機能の利用シーン**：
- HTTPExceptionの標準的なJSONレスポンス生成
- バリデーションエラーの詳細なエラー情報返却
- WebSocketバリデーションエラー時の接続クローズ
- カスタム例外クラスの処理
- エラーログの一元記録
- エラー通知（Slack、メール等）

**主要な処理内容**：
1. FastAPIアプリケーション初期化時にデフォルト例外ハンドラを登録
2. リクエスト処理中に例外が発生した場合、対応するハンドラを検索
3. ハンドラ関数を実行し、HTTPレスポンスを生成
4. レスポンスをクライアントに返却

**関連システム・外部連携**：
- StarletteのExceptionMiddleware
- Pydanticバリデーション
- JSONエンコーダ機能

**権限による制御**：本機能自体には権限制御は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | API全般のエラーレスポンス生成に使用 |

## 機能種別

例外処理 / ミドルウェア

## 入力仕様

### 入力パラメータ

#### http_exception_handler

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| request | Request | Yes | HTTPリクエストオブジェクト | - |
| exc | HTTPException | Yes | 発生した例外 | - |

#### request_validation_exception_handler

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| request | Request | Yes | HTTPリクエストオブジェクト | - |
| exc | RequestValidationError | Yes | 発生した例外 | - |

#### websocket_request_validation_exception_handler

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| websocket | WebSocket | Yes | WebSocket接続オブジェクト | - |
| exc | WebSocketRequestValidationError | Yes | 発生した例外 | - |

### 入力データソース

- 発生した例外オブジェクト
- リクエスト/WebSocket接続オブジェクト

## 出力仕様

### 出力データ

#### http_exception_handler

| 項目名 | 型 | 説明 |
|--------|-----|------|
| response | Response/JSONResponse | HTTPレスポンス |

#### request_validation_exception_handler

| 項目名 | 型 | 説明 |
|--------|-----|------|
| response | JSONResponse | HTTPレスポンス（status_code=422） |

#### websocket_request_validation_exception_handler

| 項目名 | 型 | 説明 |
|--------|-----|------|
| - | None | WebSocket接続をクローズ |

### 出力先

- HTTPレスポンス / WebSocketクローズフレーム

## 処理フロー

### 処理シーケンス

```
1. FastAPIアプリケーション初期化
   └─ デフォルト例外ハンドラを exception_handlers 辞書に登録
2. リクエスト処理中に例外発生
   └─ ExceptionMiddlewareが例外をキャッチ
3. 例外タイプに対応するハンドラを検索
   └─ exception_handlers 辞書から検索
4. ハンドラ関数を実行
   └─ HTTPException: http_exception_handler
   └─ RequestValidationError: request_validation_exception_handler
   └─ WebSocketRequestValidationError: websocket_request_validation_exception_handler
5. レスポンス生成・返却
   └─ 生成されたレスポンスをクライアントに返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト処理開始] --> B{例外発生?}
    B -->|No| C[正常レスポンス返却]
    B -->|Yes| D[ExceptionMiddlewareが例外キャッチ]
    D --> E{例外タイプ判定}
    E -->|HTTPException| F[http_exception_handler]
    E -->|RequestValidationError| G[request_validation_exception_handler]
    E -->|WebSocketRequestValidationError| H[websocket_request_validation_exception_handler]
    E -->|その他| I{カスタムハンドラあり?}
    I -->|Yes| J[カスタムハンドラ実行]
    I -->|No| K[デフォルトエラー処理]
    F --> L{ボディ許可ステータス?}
    L -->|Yes| M[JSONResponse生成]
    L -->|No| N[空Response生成]
    G --> O[JSONResponse 422]
    H --> P[WebSocket.close]
    M --> Q[レスポンス返却]
    N --> Q
    O --> Q
    J --> Q
    K --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-28-1 | デフォルトハンドラ自動登録 | FastAPI初期化時に3つのデフォルトハンドラが登録される | FastAPI初期化時 |
| BR-28-2 | カスタムハンドラ優先 | 同じ例外タイプにカスタムハンドラが登録された場合、カスタムハンドラが優先される | カスタムハンドラ登録時 |
| BR-28-3 | ボディ許可チェック | 1xx、204、304等のステータスコードではレスポンスボディを含めない | http_exception_handler |
| BR-28-4 | WebSocketバリデーションエラー | WS_1008_POLICY_VIOLATIONコードで接続をクローズ | websocket_request_validation_exception_handler |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベースアクセスを行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 例外ハンドラエラー | 例外ハンドラ自体でエラー発生 | ハンドラ実装を確認 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 例外ハンドラは非同期関数として実装され、即時実行完了

## セキュリティ考慮事項

- エラーレスポンスに機密情報（スタックトレース、内部パス等）を含めないよう注意
- 本番環境では詳細なエラー情報の公開を制限すべき

## 備考

- @app.exception_handler()デコレータでカスタムハンドラを登録可能
- exception_handlersパラメータでFastAPI初期化時にハンドラを一括登録可能
- 500またはException用のハンドラはServerErrorMiddlewareで処理される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: デフォルト例外ハンドラを理解する

FastAPIが提供するデフォルト例外ハンドラの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | exception_handlers.py | `fastapi/exception_handlers.py` | 3つのデフォルト例外ハンドラ |

**主要処理フロー**:
- **1-8行目**: インポート文
- **11-17行目**: http_exception_handler関数
- **12行目**: 例外からヘッダー取得
- **13-14行目**: ボディ許可チェック、空Response返却
- **15-17行目**: JSONResponse返却
- **20-26行目**: request_validation_exception_handler関数
- **23-25行目**: 422ステータスでJSONResponse返却
- **29-34行目**: websocket_request_validation_exception_handler関数
- **32-33行目**: WS_1008_POLICY_VIOLATIONでWebSocketクローズ

**読解のコツ**: 各ハンドラは非同期関数として定義されている。http_exception_handlerはis_body_allowed_for_status_codeで条件分岐している。

#### Step 2: 例外ハンドラ登録を理解する

FastAPIアプリケーションでの例外ハンドラ登録を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | applications.py | `fastapi/applications.py` | 例外ハンドラの登録処理 |

**主要処理フロー**:
- **15-18行目**: デフォルト例外ハンドラのインポート
- **490-505行目**: exception_handlersパラメータの定義
- **979-990行目**: デフォルトハンドラの登録
- **982行目**: HTTPException用ハンドラをsetdefault
- **983-984行目**: RequestValidationError用ハンドラをsetdefault
- **986-989行目**: WebSocketRequestValidationError用ハンドラをsetdefault
- **1003-1016行目**: ミドルウェアスタックの構築

**読解のコツ**: setdefaultを使用しているため、ユーザーがカスタムハンドラを登録した場合はそちらが優先される。

#### Step 3: ユーティリティ関数を理解する

例外ハンドラで使用されるユーティリティを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.py | `fastapi/utils.py` | is_body_allowed_for_status_code関数 |
| 3-2 | encoders.py | `fastapi/encoders.py` | jsonable_encoder関数 |

### プログラム呼び出し階層図

```
FastAPI.__init__()
    │
    └─ exception_handlers辞書の初期化
           │
           ├─ HTTPException → http_exception_handler
           ├─ RequestValidationError → request_validation_exception_handler
           └─ WebSocketRequestValidationError → websocket_request_validation_exception_handler

リクエスト処理
    │
    └─ ExceptionMiddleware
           │
           └─ 例外発生時
                  │
                  └─ exception_handlers[例外タイプ](request, exc)
                         │
                         ├─ http_exception_handler
                         │      ├─ is_body_allowed_for_status_code()
                         │      └─ JSONResponse or Response
                         │
                         ├─ request_validation_exception_handler
                         │      ├─ jsonable_encoder()
                         │      └─ JSONResponse(422)
                         │
                         └─ websocket_request_validation_exception_handler
                                └─ WebSocket.close()
```

### データフロー図

```
[入力]                          [処理]                        [出力]

例外発生
- HTTPException              http_exception_handler
  - status_code ───────────▶ ボディ許可チェック ──────────▶ Response/JSONResponse
  - detail                    ↓
  - headers                  ヘッダー追加

- RequestValidationError     request_validation_exception_handler
  - errors ────────────────▶ jsonable_encoder ────────────▶ JSONResponse 422
                              ↓
                             {"detail": [...]}

- WebSocketRequestValidationError  websocket_request_validation_exception_handler
  - errors ────────────────▶ WebSocket.close ─────────────▶ クローズフレーム
                              ↓                             (WS_1008_POLICY_VIOLATION)
                             reason: jsonable_encoder(errors)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| exception_handlers.py | `fastapi/exception_handlers.py` | ソース | デフォルト例外ハンドラの定義 |
| applications.py | `fastapi/applications.py` | ソース | 例外ハンドラの登録 |
| exceptions.py | `fastapi/exceptions.py` | ソース | 例外クラスの定義 |
| utils.py | `fastapi/utils.py` | ソース | is_body_allowed_for_status_code関数 |
| encoders.py | `fastapi/encoders.py` | ソース | jsonable_encoder関数 |
| websockets.py | `fastapi/websockets.py` | ソース | WebSocketクラス |
| responses.py | `starlette/responses.py` | 外部依存 | JSONResponse、Responseクラス |
| status.py | `starlette/status.py` | 外部依存 | WS_1008_POLICY_VIOLATION定数 |
