# 機能設計書 3-Request/Response処理

## 概要

本ドキュメントは、FastAPIフレームワークにおけるHTTPリクエストの受信とレスポンスの送信処理機能の設計を記述する。JSONResponse、HTMLResponse等の各種レスポンス形式に対応し、効率的なHTTP通信を実現する。

### 本機能の処理概要

Request/Response処理は、FastAPIアプリケーションにおけるHTTP通信の基盤機能である。クライアントからのリクエストを受信し、パスオペレーション関数で処理した結果を適切な形式でレスポンスとして返却する。Starletteのリクエスト/レスポンスクラスを基盤として、FastAPI固有の拡張（高性能JSONシリアライザ対応等）を提供する。

**業務上の目的・背景**：Webアプリケーションの中核機能として、クライアントとサーバー間のデータ通信を担う。RESTful APIにおいては、JSON形式でのデータ交換が主流であるが、HTMLページ返却、ファイルダウンロード、ストリーミング等の多様なユースケースに対応する必要がある。本機能により、開発者はビジネスロジックに集中でき、通信層の詳細を意識せずに済む。

**機能の利用シーン**：
- APIエンドポイントでJSONデータを返却する際
- HTMLテンプレートをレンダリングして返却する際
- ファイルダウンロード機能を実装する際
- リダイレクト処理を行う際
- ストリーミングレスポンスを実装する際

**主要な処理内容**：
1. HTTPリクエストの受信とパース（ヘッダー、ボディ、クエリパラメータ等）
2. リクエストデータのバリデーション（Pydantic連携）
3. レスポンスオブジェクトの生成（ステータスコード、ヘッダー、ボディ）
4. レスポンスボディのシリアライズ（JSON、HTML等）
5. HTTPレスポンスの送信

**関連システム・外部連携**：Starlette（Request/Responseベースクラス）、Pydantic（データバリデーション）、ujson/orjson（高性能JSONシリアライザ）

**権限による制御**：本機能自体は権限制御を持たないが、依存性注入を通じてリクエストレベルでの認証・認可チェックが可能

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | API連携 | HTMLResponseを使用してSwagger UIのHTML画面を返却 |
| 2 | ReDoc | API連携 | HTMLResponseを使用してReDocのHTML画面を返却 |
| 3 | OAuth2リダイレクト | API連携 | HTMLResponseを使用してリダイレクト画面を返却 |
| 4 | アイテム詳細画面 | API連携 | HTMLResponseを使用してレンダリング結果を返却 |
| 5 | OpenAPI JSON | API連携 | JSONResponseを使用してOpenAPIスキーマをJSON形式で返却 |

## 機能種別

HTTP通信 / データシリアライズ / レスポンス生成

## 入力仕様

### 入力パラメータ（Request）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| scope | Scope | Yes | ASGIスコープ（リクエストメタデータ） | 辞書型 |
| receive | Receive | Yes | ASGIメッセージ受信関数 | 呼び出し可能 |
| send | Send | No | ASGIメッセージ送信関数 | 呼び出し可能 |

### 入力パラメータ（Response）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| content | Any | No | レスポンスボディコンテンツ | 型に応じたシリアライズ |
| status_code | int | No | HTTPステータスコード（デフォルト: 200） | 100-599の整数 |
| headers | Optional[Mapping] | No | レスポンスヘッダー | キー・値が文字列の辞書 |
| media_type | Optional[str] | No | Content-Type | MIMEタイプ文字列 |
| background | Optional[BackgroundTask] | No | バックグラウンドタスク | BackgroundTaskインスタンス |

### 入力データソース

- ASGIサーバー（uvicorn等）からのスコープ・受信関数
- パスオペレーション関数の戻り値

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| body | bytes | シリアライズされたレスポンスボディ |
| status_code | int | HTTPステータスコード |
| headers | MutableHeaders | レスポンスヘッダー |
| media_type | str | Content-Typeヘッダー値 |

### 出力先

- ASGIサーバー（send関数経由）
- クライアント（HTTPレスポンス）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ ASGIスコープとreceive関数からRequestオブジェクト生成
2. リクエストデータ取得
   └─ headers, query_params, path_params, body等のプロパティアクセス
3. パスオペレーション実行
   └─ エンドポイント関数の呼び出しと戻り値取得
4. レスポンス生成
   └─ 戻り値からResponseオブジェクト生成（型に応じてシリアライズ）
5. レスポンス送信
   └─ send関数経由でASGIサーバーにレスポンス送信
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[Requestオブジェクト生成]
    B --> C[パスマッチング]
    C --> D[依存性解決]
    D --> E[パスオペレーション実行]
    E --> F{戻り値の型判定}
    F -->|Response| G[直接返却]
    F -->|dict/list| H[JSONResponse生成]
    F -->|str| I[PlainTextResponse生成]
    F -->|bytes| J[Response生成]
    G --> K[レスポンス送信]
    H --> K
    I --> K
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 自動JSON変換 | dict/list/Pydanticモデルは自動的にJSONResponseに変換 | response_class未指定時 |
| BR-002 | Responseバイパス | Responseオブジェクトを返却した場合はそのまま返却 | 戻り値がResponseインスタンス時 |
| BR-003 | ステータスコード継承 | レスポンスのステータスコードはstatus_code引数または200 | status_code指定時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | リクエストボディのパースエラー | 正しい形式のリクエストを送信 |
| 422 | Unprocessable Entity | バリデーションエラー | バリデーション要件を満たすデータを送信 |
| 500 | Internal Server Error | レスポンスシリアライズエラー | シリアライズ可能なデータを返却 |

### リトライ仕様

該当なし（リクエスト単位での処理のため）

## トランザクション仕様

該当なし

## パフォーマンス要件

- ujson/orjson使用時、標準JSONライブラリの2-10倍の性能
- ストリーミングレスポンスでメモリ効率の良い大容量データ転送

## セキュリティ考慮事項

- CORSヘッダーの適切な設定（CORSMiddleware使用推奨）
- Content-Typeヘッダーの正確な設定
- セキュリティヘッダー（X-Content-Type-Options等）の付与推奨

## 備考

- FastAPIのレスポンスクラスはStarletteのクラスを再エクスポート
- UJSONResponse、ORJSONResponseは高性能JSON処理用の独自拡張

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

リクエスト/レスポンスの基本構造を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | requests.py | `fastapi/requests.py` | Starletteからの再エクスポートを確認 |
| 1-2 | responses.py | `fastapi/responses.py` | 各種レスポンスクラスの定義 |

**読解のコツ**: FastAPIのrequests.pyはStarletteのRequest/HTTPConnectionを単純に再エクスポートしている。responses.pyには独自のUJSONResponse、ORJSONResponseが追加されている。

#### Step 2: エントリーポイントを理解する

リクエスト処理のエントリーポイントを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | routing.py | `fastapi/routing.py` | request_response関数（80-117行目） |
| 2-2 | routing.py | `fastapi/routing.py` | get_request_handler関数（248-403行目） |

**主要処理フロー**:
1. **80-89行目**: request_response関数の定義（ASGIアプリケーションラッパー）
2. **91-116行目**: AsyncExitStackによる依存性管理とレスポンス処理
3. **248-262行目**: get_request_handlerのパラメータ定義
4. **291-337行目**: リクエストボディの読み取りとパース

#### Step 3: レスポンスシリアライズを理解する

レスポンスのシリアライズ処理を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | serialize_response関数（190-232行目） |
| 3-2 | encoders.py | `fastapi/encoders.py` | jsonable_encoder関数 |

**主要処理フロー**:
- **190-202行目**: serialize_responseのパラメータ定義
- **203-229行目**: バリデーションとシリアライズ処理
- **232行目**: フィールドなしの場合はjsonable_encoder使用

### プログラム呼び出し階層図

```
ASGIサーバー (uvicorn等)
    │
    └─ FastAPI.__call__()
           │
           └─ request_response() [routing.py:80-117]
                  │
                  ├─ Request生成 [starlette/requests.py]
                  │
                  ├─ get_request_handler() [routing.py:248-403]
                  │      │
                  │      ├─ リクエストボディ読み取り
                  │      ├─ solve_dependencies()
                  │      ├─ run_endpoint_function()
                  │      └─ serialize_response()
                  │             └─ jsonable_encoder()
                  │
                  └─ Response送信 [starlette/responses.py]
```

### データフロー図

```
[入力]                     [処理]                           [出力]

HTTPリクエスト ───▶ Requestオブジェクト ───▶ パスオペレーション
                                                    │
                                                    ▼
                                            戻り値（dict等）
                                                    │
                                                    ▼
                                         serialize_response()
                                                    │
                                                    ▼
                                         JSONResponse ───▶ HTTPレスポンス
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| requests.py | `fastapi/requests.py` | ソース | Request、HTTPConnection再エクスポート |
| responses.py | `fastapi/responses.py` | ソース | 各種Responseクラス定義 |
| routing.py | `fastapi/routing.py` | ソース | request_response、get_request_handler |
| encoders.py | `fastapi/encoders.py` | ソース | jsonable_encoder |
| exception_handlers.py | `fastapi/exception_handlers.py` | ソース | 例外時のレスポンス生成 |
