# 機能設計書 30-WebSocketDisconnect

## 概要

本ドキュメントは、FastAPIフレームワークが提供するWebSocketDisconnect例外クラスの設計について記載する。

### 本機能の処理概要

WebSocketDisconnectは、WebSocket接続がクライアントによって切断された際に発生する例外クラスである。開発者はこの例外をキャッチすることで、クライアント切断を検出し、適切なクリーンアップ処理（リソース解放、他クライアントへの通知等）を実行できる。

**業務上の目的・背景**：WebSocket通信において、クライアントが予期せず切断される状況（ネットワーク障害、ブラウザクローズ、タイムアウト等）は頻繁に発生する。WebSocketDisconnectは、このような切断イベントを検出し、サーバー側でリソースリークを防止し、他の接続クライアントへの適切な通知を可能にする。

**機能の利用シーン**：
- クライアントがブラウザを閉じた場合
- ネットワーク接続が切断された場合
- クライアントが明示的に接続を閉じた場合
- 接続タイムアウトが発生した場合
- サーバー側から接続を閉じた後のクリーンアップ

**主要な処理内容**：
1. WebSocket receive操作中にクライアント切断を検出
2. WebSocketDisconnect例外の発生
3. 例外をキャッチしてクリーンアップ処理を実行
4. リソースの解放、状態の更新

**関連システム・外部連携**：
- StarletteのWebSocketDisconnect例外
- WebSocket接続管理機能

**権限による制御**：本機能自体は例外クラスの定義であり、権限制御は含まれない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | WebSocket通信を行う画面全般で使用 |

## 機能種別

例外処理 / 接続管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| code | int | No | WebSocketクローズコード | デフォルト: 1000 |
| reason | str | No | クローズ理由 | デフォルト: None |

### 入力データソース

- WebSocket接続のクローズイベント

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| code | int | クローズコード |
| reason | str | クローズ理由 |

### 出力先

- 例外として発生（エンドポイント関数内でキャッチ）

## 処理フロー

### 処理シーケンス

```
1. WebSocketエンドポイント関数内でメッセージ受信待機
   └─ await websocket.receive_text/bytes/json()
2. クライアント切断検出
   └─ WebSocket接続のクローズフレーム受信
3. WebSocketDisconnect例外発生
   └─ codeとreasonを含む例外オブジェクト
4. 例外ハンドリング
   └─ try-except でキャッチ
5. クリーンアップ処理
   └─ リソース解放、状態更新、他クライアントへの通知等
```

### フローチャート

```mermaid
flowchart TD
    A[WebSocket接続確立] --> B[メッセージ受信ループ]
    B --> C{receive_*待機}
    C -->|メッセージ受信| D[メッセージ処理]
    D --> B
    C -->|クライアント切断| E[WebSocketDisconnect発生]
    E --> F{try-exceptあり?}
    F -->|Yes| G[except WebSocketDisconnect]
    F -->|No| H[例外伝播]
    G --> I[クリーンアップ処理]
    I --> J[接続終了]
    H --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-30-1 | 切断検出 | receive操作中にクライアント切断があるとWebSocketDisconnectが発生 | receive_*呼び出し時 |
| BR-30-2 | コードデフォルト | codeのデフォルト値は1000（Normal Closure） | code未指定時 |
| BR-30-3 | クリーンアップ推奨 | WebSocketDisconnectをキャッチして適切なクリーンアップを実装すべき | 常時 |
| BR-30-4 | 再接続不可 | 切断後の同一WebSocketオブジェクトでの再接続は不可 | 切断後 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベースアクセスを行わないが、クリーンアップ処理で実装可能）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1000 | Normal Closure | 正常終了 | クリーンアップ処理を実行 |
| 1001 | Going Away | ページ遷移、サーバーシャットダウン | クリーンアップ処理を実行 |
| 1006 | Abnormal Closure | 予期しない切断（コードなし） | ログ記録、リトライロジック検討 |
| 1011 | Internal Error | サーバー内部エラーによる切断 | サーバーログを確認 |

### リトライ仕様

該当なし（切断後はクライアント側で再接続を実装）

## トランザクション仕様

該当なし

## パフォーマンス要件

- 切断検出は即時
- クリーンアップ処理は可能な限り高速に完了させる

## セキュリティ考慮事項

- 切断時のクリーンアップでセッション情報を適切に削除すること
- ログにreasonの内容を記録する場合、機密情報の露出に注意

## 備考

- WebSocketDisconnectはStarletteから再エクスポートされている
- 例外をキャッチしない場合、エンドポイント関数は正常終了扱い
- WebSocketStateでは切断後はDISCONNECTED状態となる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

WebSocketDisconnect例外クラスの定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | websockets.py | `fastapi/websockets.py` | WebSocketDisconnectのインポート |

**主要処理フロー**:
- **2行目**: StarletteのWebSocketDisconnectをインポート

**読解のコツ**: FastAPIのWebSocketDisconnectはStarletteからの完全な再エクスポートである。詳細な実装と振る舞いはstarlette.websocketsを参照する必要がある。

#### Step 2: 使用例を理解する

WebSocketエンドポイントでのWebSocketDisconnect使用パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | routing.py | `fastapi/routing.py` | websocketデコレータのドキュメント |

**読解のコツ**: @app.websocketデコレータのドキュメントにWebSocketの使用例が含まれている。WebSocketDisconnectの典型的な使用パターンは、メッセージ受信ループをtry-exceptで囲む形式である。

```python
@app.websocket("/ws")
async def websocket_endpoint(websocket: WebSocket):
    await websocket.accept()
    try:
        while True:
            data = await websocket.receive_text()
            await websocket.send_text(f"Message: {data}")
    except WebSocketDisconnect:
        print("Client disconnected")
```

### プログラム呼び出し階層図

```
WebSocketエンドポイント関数
    │
    └─ try:
           │
           └─ while True:
                  │
                  └─ await websocket.receive_text/bytes/json()
                         │
                         └─ [クライアント切断検出]
                                │
                                └─ raise WebSocketDisconnect(code, reason)
                                       │
       except WebSocketDisconnect:     │
           │ ◀─────────────────────────┘
           │
           └─ クリーンアップ処理
                  │
                  ├─ リソース解放
                  ├─ 状態更新
                  └─ 他クライアントへの通知
```

### データフロー図

```
[入力]                          [処理]                        [出力]

WebSocket接続                   receive_*()待機
                                ↓
クライアント切断 ─────────────▶ クローズフレーム検出
                                ↓
                               WebSocketDisconnect生成
                               - code
                               - reason
                                ↓
                               except WebSocketDisconnect
                                ↓
                               クリーンアップ処理 ─────────▶ リソース解放
                                                            状態更新
                                                            通知送信
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| websockets.py | `fastapi/websockets.py` | ソース | WebSocketDisconnectのインポート |
| routing.py | `fastapi/routing.py` | ソース | WebSocketエンドポイントの定義 |
| websockets.py | `starlette/websockets.py` | 外部依存 | WebSocketDisconnect例外クラスの実装 |
| status.py | `starlette/status.py` | 外部依存 | WebSocketクローズコード定数 |

### 典型的な使用パターン

#### パターン1: 基本的なクリーンアップ

```python
@app.websocket("/ws/{client_id}")
async def websocket_endpoint(websocket: WebSocket, client_id: str):
    await websocket.accept()
    connected_clients[client_id] = websocket
    try:
        while True:
            data = await websocket.receive_text()
            # メッセージ処理
    except WebSocketDisconnect:
        del connected_clients[client_id]
        print(f"Client {client_id} disconnected")
```

#### パターン2: 他クライアントへの通知

```python
@app.websocket("/chat")
async def chat_endpoint(websocket: WebSocket):
    await websocket.accept()
    clients.append(websocket)
    try:
        while True:
            data = await websocket.receive_text()
            for client in clients:
                await client.send_text(data)
    except WebSocketDisconnect:
        clients.remove(websocket)
        for client in clients:
            await client.send_text("A user left the chat")
```
