# 機能設計書 32-jsonable_encoder

## 概要

本ドキュメントは、FastAPIのjsonable_encoder機能について、その設計意図、処理フロー、実装詳細を記述した機能設計書である。

### 本機能の処理概要

jsonable_encoderは、PythonオブジェクトをJSON互換の形式に変換するユーティリティ関数である。Pydanticモデル、datetime、UUID、Enum、dataclassなど、標準のJSONエンコーダーでは処理できない型を適切に変換する。

**業務上の目的・背景**：WebAPIでは、レスポンスとしてJSONを返すことが一般的であるが、Pythonの多くの型（datetime、UUID、Pydanticモデルなど）はそのままではJSONにシリアライズできない。jsonable_encoderは、これらの型を自動的にJSON互換の形式（dict、list、str、int、float、bool、None）に変換することで、開発者がシリアライズを意識せずにAPIを開発できるようにする。

**機能の利用シーン**：
- APIレスポンスとしてPydanticモデルを返す場合
- データベースに保存する前にPythonオブジェクトをJSON形式に変換する場合
- datetime、UUIDなどの特殊な型を含むデータのシリアライズ
- SQLAlchemyオブジェクトの内部属性を除外したシリアライズ

**主要な処理内容**：
1. カスタムエンコーダーの適用チェック
2. Pydanticモデルの場合は`model_dump()`を呼び出し
3. dataclassの場合は`asdict()`で辞書に変換
4. Enum、PurePath、基本型（str、int、float、None）の処理
5. dict、list等のコレクション型の再帰的な変換
6. 組み込みエンコーダーによる型別変換

**関連システム・外部連携**：Swagger UI、ReDoc等のAPIドキュメント表示で内部的に使用される。

**権限による制御**：権限制御機能はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | Swagger UIの設定パラメータをJSON互換形式に変換 |

## 機能種別

データ変換 / シリアライゼーション

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| obj | Any | Yes | JSON変換対象のオブジェクト | なし |
| include | Optional[IncEx] | No | Pydanticモデルで含めるフィールド | set/dict形式 |
| exclude | Optional[IncEx] | No | Pydanticモデルで除外するフィールド | set/dict形式 |
| by_alias | bool | No | エイリアス名を使用するか（デフォルト: True） | なし |
| exclude_unset | bool | No | 未設定フィールドを除外（デフォルト: False） | なし |
| exclude_defaults | bool | No | デフォルト値のフィールドを除外（デフォルト: False） | なし |
| exclude_none | bool | No | Noneのフィールドを除外（デフォルト: False） | なし |
| custom_encoder | Optional[dict[Any, Callable]] | No | カスタムエンコーダー辞書 | なし |
| sqlalchemy_safe | bool | No | SQLAlchemy内部属性を除外（デフォルト: True） | なし |

### 入力データソース

- 開発者が直接関数に渡すPythonオブジェクト
- FastAPI内部でレスポンス生成時に自動的に呼び出される

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | Any | JSON互換の値（dict、list、str、int、float、bool、None） |

### 出力先

- 戻り値として呼び出し元に返される
- 最終的にはJSONレスポンスとしてクライアントに送信

## 処理フロー

### 処理シーケンス

```
1. カスタムエンコーダーチェック
   └─ type(obj)がcustom_encoderに存在すればそのエンコーダーを使用
2. include/excludeの正規化
   └─ set/dict以外の場合はsetに変換
3. 型別処理分岐
   ├─ BaseModel → model_dump()で変換後、再帰的にjsonable_encoder
   ├─ dataclass → asdict()で変換後、再帰的にjsonable_encoder
   ├─ Enum → value属性を取得
   ├─ PurePath → str()で文字列化
   ├─ 基本型 → そのまま返却
   ├─ dict → キー・値を再帰的に変換
   ├─ list/set等 → 各要素を再帰的に変換
   └─ その他 → ENCODERS_BY_TYPEで変換、なければdict/varsを試行
```

### フローチャート

```mermaid
flowchart TD
    A[入力: obj] --> B{カスタムエンコーダー?}
    B -->|Yes| C[カスタムエンコーダー適用]
    B -->|No| D{BaseModel?}
    D -->|Yes| E[model_dump後、再帰呼び出し]
    D -->|No| F{dataclass?}
    F -->|Yes| G[asdict後、再帰呼び出し]
    F -->|No| H{Enum?}
    H -->|Yes| I[obj.value返却]
    H -->|No| J{基本型?}
    J -->|Yes| K[そのまま返却]
    J -->|No| L{dict?}
    L -->|Yes| M[キー・値を再帰変換]
    L -->|No| N{list/set等?}
    N -->|Yes| O[各要素を再帰変換]
    N -->|No| P{ENCODERS_BY_TYPEに存在?}
    P -->|Yes| Q[型別エンコーダー適用]
    P -->|No| R[dict/vars試行]
    C --> Z[出力: JSON互換値]
    E --> Z
    G --> Z
    I --> Z
    K --> Z
    M --> Z
    O --> Z
    Q --> Z
    R --> Z
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-32-001 | SQLAlchemy安全モード | `_sa`で始まるキーを除外 | sqlalchemy_safe=True時 |
| BR-32-002 | エイリアス優先 | by_alias=Trueでエイリアス名を使用 | Pydanticモデル変換時 |
| BR-32-003 | None除外 | exclude_none=TrueでNone値を除外 | すべての変換時 |
| BR-32-004 | Pydantic v1非対応 | Pydantic v1モデルは例外発生 | is_pydantic_v1_model_instance判定時 |

### 計算ロジック

Decimalの変換ロジック（`decimal_encoder`）:
- 指数部が0以上の整数 → int型に変換
- それ以外 → float型に変換

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | jsonable_encoder自体はDB操作を行わない |

### テーブル別操作詳細

jsonable_encoderはデータベース操作機能を持たない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | PydanticV1NotSupportedError | Pydantic v1モデルを変換しようとした | Pydantic v2モデルに更新 |
| - | ValueError | dict()もvars()も失敗した | シリアライズ可能なオブジェクトを使用 |

### リトライ仕様

リトライ機能は提供しない。

## トランザクション仕様

トランザクション管理は不要。

## パフォーマンス要件

- 再帰的な処理のため、深くネストしたオブジェクトでは処理時間が増加
- 大量のデータを変換する場合はメモリ使用量に注意

## セキュリティ考慮事項

- SecretStr、SecretBytesは文字列に変換されるため、ログ出力時は注意
- sqlalchemy_safe=Trueでデータベース内部情報の漏洩を防止

## 備考

- ENCODERS_BY_TYPEに定義された型は自動的に変換される
- カスタムエンコーダーで任意の型の変換ロジックを追加可能
- FastAPI内部でレスポンス生成時に自動的に使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

エンコーダーの型マッピングとサポートする型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | encoders.py | `fastapi/encoders.py` | ENCODERS_BY_TYPE辞書（67-94行目） |

**読解のコツ**:
- ENCODERS_BY_TYPEは型をキー、変換関数を値とする辞書
- bytes、datetime、UUID、IPAddress等の変換ルールが定義されている

**主要処理フロー**:
1. **36-37行目**: `isoformat`関数 - datetime系の変換
2. **42-64行目**: `decimal_encoder`関数 - Decimal型の変換
3. **67-94行目**: `ENCODERS_BY_TYPE`辞書 - 型別エンコーダーマッピング
4. **97-108行目**: エンコーダーをクラスタプルでグループ化

#### Step 2: メイン関数を理解する

jsonable_encoder関数の処理フローを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | encoders.py | `fastapi/encoders.py` | jsonable_encoder関数（111-346行目） |

**主要処理フロー**:
- **111-200行目**: 関数シグネチャとパラメータ定義
- **213-224行目**: カスタムエンコーダーの適用
- **225-240行目**: Pydanticモデル（BaseModel）の処理
- **241-254行目**: dataclassの処理
- **255-262行目**: Enum、PurePath、基本型の処理
- **263-297行目**: dict型の再帰的処理
- **298-314行目**: list/set等のコレクション型の処理
- **316-320行目**: ENCODERS_BY_TYPEによる変換
- **321-325行目**: Pydantic v1モデルのエラー処理
- **326-346行目**: dict()、vars()によるフォールバック

#### Step 3: 使用箇所を理解する

jsonable_encoderがどこで使用されているかを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | レスポンスシリアライズでの使用（42行目のimport） |

**主要処理フロー**:
- **42行目**: `from fastapi.encoders import jsonable_encoder`

### プログラム呼び出し階層図

```
jsonable_encoder(obj)
    │
    ├─ custom_encoder適用
    │
    ├─ BaseModel処理
    │      └─ obj.model_dump()
    │             └─ jsonable_encoder(obj_dict) [再帰]
    │
    ├─ dataclass処理
    │      └─ dataclasses.asdict(obj)
    │             └─ jsonable_encoder(obj_dict) [再帰]
    │
    ├─ Enum処理 → obj.value
    │
    ├─ PurePath処理 → str(obj)
    │
    ├─ 基本型処理 → そのまま返却
    │
    ├─ dict処理
    │      └─ jsonable_encoder(key) [再帰]
    │      └─ jsonable_encoder(value) [再帰]
    │
    ├─ list/set等処理
    │      └─ jsonable_encoder(item) [再帰]
    │
    └─ ENCODERS_BY_TYPE処理
           └─ encoder(obj)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

Python対象オブジェクト
    │
    ▼
jsonable_encoder ─────────────────────────────────────▶ JSON互換値
    │                                                    (dict/list/
    ├─ BaseModel ─▶ model_dump() ─▶ dict                 str/int/
    ├─ dataclass ─▶ asdict() ─▶ dict                     float/bool/
    ├─ Enum ─▶ value                                     None)
    ├─ datetime ─▶ isoformat()
    ├─ UUID ─▶ str()
    ├─ Decimal ─▶ int/float
    └─ 基本型 ─▶ そのまま
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| encoders.py | `fastapi/encoders.py` | ソース | jsonable_encoder関数の定義 |
| routing.py | `fastapi/routing.py` | ソース | レスポンスシリアライズでの使用 |
| types.py | `fastapi/types.py` | ソース | IncEx型の定義 |
| _compat/__init__.py | `fastapi/_compat/__init__.py` | ソース | Pydantic互換レイヤー |
