# 機能設計書 37-GZipミドルウェア

## 概要

本ドキュメントは、FastAPIのGZipミドルウェア機能について、その設計意図、処理フロー、実装詳細を記述した機能設計書である。

### 本機能の処理概要

GZipミドルウェアは、HTTPレスポンスボディをGZip圧縮することで、転送データ量を削減し、ネットワークパフォーマンスを向上させる機能を提供する。

**業務上の目的・背景**：WebAPIでは大量のJSONデータやテキストデータを返却することが多く、ネットワーク帯域を効率的に使用することが重要である。GZipミドルウェアは、レスポンスを自動的に圧縮することで、クライアントへのデータ転送時間を短縮し、帯域使用量を削減する。特にモバイル環境や低速回線での利用時に効果を発揮する。

**機能の利用シーン**：
- 大量のJSONデータを返すAPI
- テキストベースのレスポンス（HTML、XML等）
- 帯域幅に制限がある環境でのAPI利用
- CDNとの連携によるキャッシュ効率向上

**主要な処理内容**：
1. クライアントのAccept-Encodingヘッダーを確認
2. レスポンスサイズが閾値を超えるか判定
3. 条件を満たす場合にGZip圧縮を適用
4. Content-Encodingヘッダーを付与

**関連システム・外部連携**：ブラウザやHTTPクライアントのGZip解凍機能と連携。

**権限による制御**：権限制御機能はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | GZipミドルウェアは画面を持たないバックエンド機能 |

## 機能種別

パフォーマンス最適化 / データ圧縮

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| minimum_size | int | No | 圧縮を適用する最小レスポンスサイズ（バイト）（デフォルト: 500） | 正の整数 |

### 入力データソース

- FastAPIアプリケーションの初期化時に`app.add_middleware(GZipMiddleware, minimum_size=...)`で設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 圧縮レスポンスボディ | bytes | GZip圧縮されたレスポンスデータ |
| Content-Encoding | str | "gzip"（圧縮が適用された場合） |
| Vary | str | "Accept-Encoding"（キャッシュ制御用） |

### 出力先

- HTTPレスポンスボディおよびヘッダー

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ Accept-Encodingヘッダーを確認
2. GZip対応チェック
   └─ クライアントがgzipをサポートしているか判定
3. エンドポイント処理
   └─ レスポンスを生成
4. 圧縮判定
   ├─ レスポンスサイズがminimum_sizeを超えるか
   └─ ストリーミングレスポンスでないか
5. 圧縮適用（条件を満たす場合）
   ├─ GZip圧縮を実行
   └─ Content-Encodingヘッダーを付与
6. レスポンス送信
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Accept-Encoding: gzip?}
    B -->|No| C[圧縮なしで処理]
    B -->|Yes| D[エンドポイント処理]
    D --> E{レスポンスサイズ >= minimum_size?}
    E -->|No| F[圧縮なしで送信]
    E -->|Yes| G{ストリーミング?}
    G -->|Yes| H[ストリーミング圧縮]
    G -->|No| I[一括GZip圧縮]
    H --> J[Content-Encoding: gzip付与]
    I --> J
    J --> K[レスポンス送信]
    C --> L[レスポンス送信]
    F --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-37-001 | 最小サイズ閾値 | minimum_size未満のレスポンスは圧縮しない | すべてのレスポンス |
| BR-37-002 | クライアント対応 | Accept-Encoding: gzipを送信するクライアントのみ | すべてのリクエスト |
| BR-37-003 | Varyヘッダー | キャッシュ制御のためVary: Accept-Encodingを付与 | 圧縮適用時 |

### 計算ロジック

- 圧縮率は一般的にテキストデータで60-80%削減
- バイナリデータ（画像等）は圧縮効率が低い

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | GZipミドルウェア自体はDB操作を行わない |

### テーブル別操作詳細

GZipミドルウェアはデータベース操作機能を持たない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 圧縮失敗時は非圧縮で返却 | 内部でエラーハンドリング |

### リトライ仕様

リトライ機能は提供しない。

## トランザクション仕様

トランザクション管理は不要。

## パフォーマンス要件

- 圧縮処理によるCPU負荷と転送時間短縮のトレードオフ
- minimum_sizeの調整で最適なバランスを設定
- 小さいレスポンスは圧縮オーバーヘッドの方が大きい場合がある

## セキュリティ考慮事項

- BREACH攻撃への考慮（HTTPSと組み合わせる場合）
- 機密データの圧縮は慎重に検討

## 備考

- StarletteのGZipMiddlewareを再エクスポート
- 画像や動画など既に圧縮されているデータには効果が薄い
- CDNが圧縮を行う場合は重複して適用しないよう注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: GZipMiddlewareの再エクスポートを理解する

FastAPIがStarletteからGZipMiddlewareをどのように再エクスポートしているかを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | gzip.py | `fastapi/middleware/gzip.py` | GZipMiddlewareの再エクスポート（1行目） |

**読解のコツ**:
- `from starlette.middleware.gzip import GZipMiddleware as GZipMiddleware`の形式で再エクスポート
- `# noqa`は未使用import警告の抑制

**主要処理フロー**:
1. **1行目**: Starletteから`GZipMiddleware`を再エクスポート

### プログラム呼び出し階層図

```
リクエスト受信
    │
    ▼
GZipMiddleware
    │
    ├─ Accept-Encoding確認
    │      └─ gzip対応判定
    │
    ├─ エンドポイント処理呼び出し
    │
    ├─ レスポンスサイズ判定
    │      └─ minimum_sizeとの比較
    │
    └─ 圧縮処理
           ├─ GZip圧縮
           └─ Content-Encoding: gzip付与
                  │
                  ▼
           レスポンス送信
```

### データフロー図

```
[入力]                    [処理]                           [出力]

HTTPリクエスト
    │
    ├─ Accept-Encoding: gzip
    │
    ▼
GZipMiddleware ─────────────────────────────────────▶ HTTPレスポンス
    │                                                    │
    ├─ レスポンスデータ                                    ├─ 圧縮データ（gzip）
    │      │                                             ├─ Content-Encoding: gzip
    │      ▼                                             └─ Vary: Accept-Encoding
    │   サイズ判定
    │      │
    │      ▼
    └── GZip圧縮
           │
           └─ 圧縮率: 約60-80%削減（テキストデータ）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| gzip.py | `fastapi/middleware/gzip.py` | ソース | GZipMiddlewareの再エクスポート |
| applications.py | `fastapi/applications.py` | ソース | ミドルウェア登録機能 |
