# 機能設計書 4-Pathパラメータ

## 概要

本ドキュメントは、FastAPIフレームワークにおけるPathパラメータ機能の設計を記述する。URLパスの一部として値を受け取るパラメータ定義機能であり、型検証、バリデーション、OpenAPIスキーマ生成を自動で実施する。

### 本機能の処理概要

Pathパラメータは、URLパスの動的部分（例：`/items/{item_id}`の`item_id`）から値を抽出し、パスオペレーション関数の引数として渡す機能である。Python型ヒントとPydanticのバリデーション機能を活用し、型変換とバリデーションを自動的に行う。

**業務上の目的・背景**：RESTful API設計において、リソースの識別子はURLパスに含めるのが一般的である。例えば、ユーザーID、商品ID、注文番号等をパスパラメータとして受け取ることで、直感的で一貫性のあるAPI設計が可能になる。本機能により、開発者はパス変数の抽出・型変換・バリデーションを手動で実装する必要がなくなり、ボイラープレートコードを大幅に削減できる。

**機能の利用シーン**：
- リソース詳細取得API（`/users/{user_id}`）
- リソース更新・削除API（`/items/{item_id}`）
- ネストされたリソースアクセス（`/users/{user_id}/orders/{order_id}`）
- 複数パスパラメータを持つAPI

**主要な処理内容**：
1. URLパスからパラメータ値の抽出
2. 型ヒントに基づく型変換（str→int等）
3. バリデーションルールに基づく値検証
4. OpenAPIスキーマへのパラメータ情報追加
5. バリデーションエラー時の422レスポンス生成

**関連システム・外部連携**：Pydantic（FieldInfo基底クラス、バリデーション）、OpenAPI（スキーマ生成）、Starlette（パスコンバーター）

**権限による制御**：本機能自体は権限制御を持たない

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | アイテム詳細画面 | 補助機能 | URLパス内の{id}パラメータからアイテムIDを取得 |

## 機能種別

パラメータ抽出 / バリデーション / 型変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| default | Any | No | デフォルト値（Pathでは常に必須のため...が設定される） | 必須パラメータ |
| alias | Optional[str] | No | パラメータの別名 | 文字列 |
| title | Optional[str] | No | 人間可読なタイトル（OpenAPI用） | 文字列 |
| description | Optional[str] | No | パラメータの説明（OpenAPI用） | 文字列 |
| gt | Optional[float] | No | より大きい（Greater Than） | 数値のみ |
| ge | Optional[float] | No | 以上（Greater than or Equal） | 数値のみ |
| lt | Optional[float] | No | より小さい（Less Than） | 数値のみ |
| le | Optional[float] | No | 以下（Less than or Equal） | 数値のみ |
| min_length | Optional[int] | No | 最小文字列長 | 文字列のみ |
| max_length | Optional[int] | No | 最大文字列長 | 文字列のみ |
| pattern | Optional[str] | No | 正規表現パターン | 文字列のみ |
| examples | Optional[list[Any]] | No | 例示値リスト | 任意 |
| deprecated | Union[bool, str, None] | No | 非推奨フラグ | ブール値または文字列 |
| include_in_schema | bool | No | OpenAPIスキーマへの含有（デフォルト: True） | ブール値 |

### 入力データソース

- URLパス（リクエストURL）
- パスオペレーション定義（デコレータとシグネチャ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 変換後の値 | 型ヒントで指定した型 | 抽出・変換・検証済みのパラメータ値 |

### 出力先

- パスオペレーション関数の引数

## 処理フロー

### 処理シーケンス

```
1. URLパスマッチング
   └─ Starletteのパスコンバーターによるパターンマッチ
2. パラメータ値抽出
   └─ パスから対応する変数値を抽出
3. 型変換
   └─ 型ヒントに基づく変換（str→int等）
4. バリデーション実行
   └─ Pydanticによるバリデーションルール検証
5. 関数引数への受け渡し
   └─ 変換・検証済みの値をパスオペレーション関数に渡す
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[URLパスマッチング]
    B --> C[パスパラメータ抽出]
    C --> D[型変換]
    D --> E{バリデーション}
    E -->|成功| F[関数引数に設定]
    E -->|失敗| G[422エラーレスポンス]
    F --> H[パスオペレーション実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Pathは常に必須 | Pathパラメータはデフォルト値を持てない（URLの一部であるため） | 常時 |
| BR-002 | 型変換自動実行 | 型ヒントに基づいて自動的に型変換される | 型ヒント指定時 |
| BR-003 | 数値制約 | gt/ge/lt/leによる数値範囲制約が適用される | 数値型パラメータ |
| BR-004 | 文字列制約 | min_length/max_length/patternによる文字列制約が適用される | 文字列型パラメータ |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 422 | Unprocessable Entity | 型変換失敗（例：文字列→整数変換不可） | 正しい型の値をURLに含める |
| 422 | Unprocessable Entity | バリデーション失敗（範囲外、パターン不一致等） | バリデーション要件を満たす値を使用 |
| 404 | Not Found | パスパターンにマッチしない | 正しいURLパターンを使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- パス変換は非常に軽量な処理
- Pydanticバリデーションのオーバーヘッドは数マイクロ秒

## セキュリティ考慮事項

- パスパラメータは常にバリデーションされるため、インジェクション攻撃に対する基本的な防御を提供
- SQLインジェクション対策はアプリケーション層で別途必要
- パストラバーサル攻撃対策としてpatternバリデーションの活用を推奨

## 備考

- Pathクラスはparams.Paramを継承し、in_=ParamTypes.pathが設定される
- OpenAPI 3.xのpath parameter仕様に準拠

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Pathクラスの定義と継承関係を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | ParamTypes列挙型、Paramクラス、Pathクラスの定義 |

**読解のコツ**:
- **20-25行目**: ParamTypes列挙型でpath/query/header/cookieを定義
- **27-136行目**: Paramクラス（PydanticのFieldInfo継承）がバリデーションパラメータを保持
- **138-219行目**: PathクラスはParamを継承し、in_=ParamTypes.pathを設定。defaultは...（必須）

#### Step 2: 関数インターフェースを理解する

Path()関数の定義を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | param_functions.py | `fastapi/param_functions.py` | Path関数の定義（14-337行目） |

**主要処理フロー**:
1. **14-285行目**: Path関数のパラメータ定義（Annotatedドキュメント付き）
2. **286-306行目**: 使用例のドキュメント
3. **307-337行目**: params.Pathインスタンスの生成と返却

#### Step 3: 依存性解決での処理を理解する

パスパラメータがどのように依存性として解決されるかを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | dependencies/utils.py | `fastapi/dependencies/utils.py` | get_dependant関数でのパスパラメータ処理 |

### プログラム呼び出し階層図

```
@app.get("/items/{item_id}")
    │
    ├─ compile_path() [starlette/routing.py]
    │      └─ パスパターン解析とコンバーター設定
    │
    ├─ get_dependant() [dependencies/utils.py]
    │      └─ パスパラメータの依存性として登録
    │
    └─ solve_dependencies() [dependencies/utils.py]
           │
           ├─ request.path_params取得
           ├─ Pydanticバリデーション
           └─ 関数引数への値設定
```

### データフロー図

```
[入力]                     [処理]                           [出力]

/items/42 ───▶ パスマッチング ───▶ {"item_id": "42"}
                                        │
                                        ▼
                                   型変換（str→int）
                                        │
                                        ▼
                                   バリデーション
                                        │
                                        ▼
                                   item_id: int = 42 ───▶ 関数引数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| param_functions.py | `fastapi/param_functions.py` | ソース | Path関数定義 |
| params.py | `fastapi/params.py` | ソース | Pathクラス、ParamTypes定義 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | 依存性解決でのパスパラメータ処理 |
| routing.py | `fastapi/routing.py` | ソース | compile_pathによるパスパターン処理 |
