# 機能設計書 41-AsyncExitStackミドルウェア

## 概要

本ドキュメントは、FastAPIにおけるAsyncExitStackミドルウェアの設計と実装について記述する。このミドルウェアは依存性注入システムにおけるyield処理（コンテキストマネージャ形式の依存性）をサポートするための内部基盤機能である。

### 本機能の処理概要

**業務上の目的・背景**：FastAPIの依存性注入システムでは、リソースの取得と解放を自動管理するために`yield`キーワードを使った依存性が使用される。例えば、データベースセッションの取得と自動クローズ、ファイルハンドルの管理、外部サービス接続のライフサイクル管理などが該当する。AsyncExitStackミドルウェアは、これらのリソースをリクエスト処理後に確実にクリーンアップするための基盤を提供する。

**機能の利用シーン**：本機能は以下のシーンで自動的に利用される：
- データベースセッションを`yield`で提供する依存性
- ファイルアップロード時の一時ファイル管理
- 外部API接続プールの管理
- トランザクション管理など、処理完了後にクリーンアップが必要なリソース

**主要な処理内容**：
1. ASGIスコープ内にAsyncExitStackインスタンスを作成・登録
2. リクエスト処理の間、コンテキストを維持
3. リクエスト処理完了後、登録されたすべてのコンテキストマネージャを逆順でクローズ
4. 例外発生時も確実にリソースをクリーンアップ

**関連システム・外部連携**：依存性注入システム（`fastapi/dependencies/`）と密接に連携し、`yield`を使用した依存性のライフサイクルを管理する。

**権限による制御**：本機能は内部基盤機能であり、権限による制御は行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は内部基盤機能であり、直接関連する画面はない |

## 機能種別

リソース管理 / ミドルウェア処理 / クリーンアップ処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| app | ASGIApp | Yes | ラップ対象のASGIアプリケーション | なし |
| context_name | str | No | スコープ内でのスタック識別名（デフォルト: "fastapi_middleware_astack"） | なし |

### 入力データソース

ASGIプロトコル経由で受信するリクエストのscope、receive、send

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| scope[context_name] | AsyncExitStack | ASGIスコープに登録されるAsyncExitStackインスタンス |

### 出力先

ASGIスコープ辞書に直接登録される

## 処理フロー

### 処理シーケンス

```
1. ミドルウェアインスタンス化
   └─ app、context_nameをインスタンス変数に保存
2. リクエスト受信（__call__呼び出し）
   └─ async withでAsyncExitStackを作成
3. スコープへのスタック登録
   └─ scope[context_name] = stack
4. 内部アプリケーション実行
   └─ await self.app(scope, receive, send)
5. クリーンアップ処理
   └─ async with終了時に自動的にスタック内のコンテキストを解放
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B[AsyncExitStack作成]
    B --> C[スコープにスタック登録]
    C --> D[内部アプリ実行]
    D --> E{例外発生?}
    E -->|No| F[正常完了]
    E -->|Yes| G[例外捕捉]
    F --> H[AsyncExitStack自動クローズ]
    G --> H
    H --> I[クリーンアップ完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-41-01 | 自動クリーンアップ | 登録されたすべてのコンテキストマネージャは、リクエスト処理完了後に自動的にクローズされる | 常時 |
| BR-41-02 | 逆順クローズ | コンテキストマネージャは登録と逆の順序でクローズされる | 常時 |
| BR-41-03 | 例外時クリーンアップ | 例外発生時も確実にクリーンアップを実行する | 例外発生時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | クリーンアップ時例外 | コンテキストマネージャの__aexit__で例外発生 | AsyncExitStack内部で処理され、元の例外が再送出される |

### リトライ仕様

クリーンアップ処理に対するリトライは行わない

## トランザクション仕様

本機能自体はトランザクション管理を行わないが、データベース依存性で使用されるセッションのコミット/ロールバックを確実に実行するための基盤を提供する。

## パフォーマンス要件

- AsyncExitStackの作成は軽量な操作であり、パフォーマンスへの影響は最小限
- 各リクエストで1つのスタックインスタンスが作成される

## セキュリティ考慮事項

- リソースリークの防止により、セキュリティ上の問題（メモリ枯渇等）を回避
- ファイルハンドルや接続の確実なクローズにより、情報漏洩リスクを低減

## 備考

- 本ミドルウェアは主にファイルのクローズ処理に使用される
- 依存性自身のAsyncExitStackは別途管理される（コメント: "dependencies are closed in their own AsyncExitStack"）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ASGIプロトコルの基本型定義を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.py | `starlette/types.py` | ASGIApp、Receive、Scope、Send型の定義を確認 |

**読解のコツ**: StarletteのASGI型定義はFastAPIでも再利用されている。Scopeは辞書型でリクエスト情報を保持する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | asyncexitstack.py | `fastapi/middleware/asyncexitstack.py` | ミドルウェアクラスの完全な実装 |

**主要処理フロー**:
1. **8-13行目**: `__init__`メソッドでapp、context_nameを保持
2. **15-18行目**: `__call__`メソッドでAsyncExitStackを作成し、スコープに登録後、内部アプリを実行

#### Step 3: 依存性注入との連携を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | request_response関数でのAsyncExitStack使用 |

**主要処理フロー**:
- **91-100行目**: request_response関数内でfastapi_inner_astackとfastapi_function_astackが作成される

### プログラム呼び出し階層図

```
AsyncExitStackMiddleware.__call__
    │
    ├─ AsyncExitStack() # コンテキストマネージャ作成
    │      └─ scope[context_name] = stack # スコープに登録
    │
    └─ self.app(scope, receive, send) # 内部アプリ呼び出し
           │
           └─ 依存性のyield処理
                  └─ stack.enter_async_context() # 自動登録
```

### データフロー図

```
[入力]               [処理]                         [出力]

Request ────────▶ AsyncExitStackMiddleware ────▶ Response
   │                    │                            │
   └─ scope             ├─ AsyncExitStack作成        └─ クリーンアップ完了
                        ├─ スコープに登録
                        └─ 内部アプリ実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| asyncexitstack.py | `fastapi/middleware/asyncexitstack.py` | ソース | AsyncExitStackミドルウェアの実装 |
| routing.py | `fastapi/routing.py` | ソース | リクエスト処理でのAsyncExitStack使用 |
| utils.py | `fastapi/dependencies/utils.py` | ソース | 依存性解決時のコンテキスト管理 |
| __init__.py | `fastapi/middleware/__init__.py` | ソース | ミドルウェアパッケージの公開インターフェース |
