# 機能設計書 42-Jinja2Templates

## 概要

本ドキュメントは、FastAPIにおけるJinja2Templatesの設計と実装について記述する。この機能はJinja2テンプレートエンジンとの連携機能を提供し、HTMLテンプレートのレンダリングに使用される。

### 本機能の処理概要

**業務上の目的・背景**：モダンなWebアプリケーションでは、APIエンドポイントだけでなく、HTMLページを動的に生成する必要がある場合がある。管理画面、ランディングページ、メール本文の生成など、サーバーサイドレンダリングが必要なシーンにおいて、Jinja2Templatesは強力なテンプレートエンジンとしてHTMLの動的生成を実現する。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- 管理画面やダッシュボードのHTML生成
- メールテンプレートのレンダリング
- 静的サイトジェネレーターとしての使用
- サーバーサイドレンダリング（SSR）が必要なページ
- HTMLドキュメントの動的生成

**主要な処理内容**：
1. テンプレートディレクトリの設定と初期化
2. テンプレートファイルの読み込みとコンパイル
3. コンテキスト変数を使用したテンプレートのレンダリング
4. TemplateResponseオブジェクトの生成と返却
5. url_for関数による静的ファイルパス生成のサポート

**関連システム・外部連携**：Starlette Jinja2Templatesクラスをそのまま再エクスポートしており、Jinja2ライブラリに依存する。StaticFiles機能と組み合わせて使用されることが多い。

**権限による制御**：本機能自体は権限制御を行わない。テンプレート内での表示制御はコンテキスト変数を通じてアプリケーション層で実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | アイテム詳細画面 | 主機能 | Jinja2テンプレートエンジンを使用してitem.htmlをレンダリング。アイテムIDをテンプレート変数として渡す |

## 機能種別

テンプレートレンダリング / HTML生成 / ビュー処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| directory | str / PathLike | Yes | テンプレートファイルを格納するディレクトリパス | 存在するディレクトリであること |
| context_processors | list[Callable] | No | コンテキストに自動追加される関数のリスト | なし |
| env | Environment | No | カスタムJinja2環境オブジェクト | なし |

### TemplateResponseのパラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| request | Request | Yes | Starletteリクエストオブジェクト | なし |
| name | str | Yes | テンプレートファイル名 | 存在するファイルであること |
| context | dict | No | テンプレートに渡すコンテキスト辞書 | なし |
| status_code | int | No | HTTPステータスコード（デフォルト: 200） | 有効なHTTPステータスコード |
| headers | dict | No | 追加のHTTPヘッダー | なし |
| media_type | str | No | レスポンスのメディアタイプ | なし |

### 入力データソース

- テンプレートファイル：指定されたディレクトリ内の.htmlファイル等
- コンテキスト変数：エンドポイント関数から渡される辞書データ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TemplateResponse | TemplateResponse | レンダリングされたHTMLを含むレスポンスオブジェクト |
| body | bytes | レンダリングされたHTML文字列（バイト列） |
| headers | dict | Content-Type: text/html等のヘッダー |

### 出力先

HTTPレスポンスとしてクライアントに返却

## 処理フロー

### 処理シーケンス

```
1. Jinja2Templates初期化
   └─ テンプレートディレクトリの設定
2. テンプレートリクエスト受信
   └─ エンドポイント関数から呼び出し
3. コンテキスト準備
   └─ request、その他変数を辞書にまとめる
4. テンプレートレンダリング
   └─ Jinja2エンジンによるHTML生成
5. レスポンス生成
   └─ TemplateResponseオブジェクトの作成と返却
```

### フローチャート

```mermaid
flowchart TD
    A[エンドポイント呼び出し] --> B[Jinja2Templates初期化確認]
    B --> C[テンプレートファイル読み込み]
    C --> D{テンプレート存在?}
    D -->|No| E[TemplateNotFound例外]
    D -->|Yes| F[コンテキスト変数準備]
    F --> G[テンプレートレンダリング]
    G --> H[TemplateResponse生成]
    H --> I[HTMLレスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-42-01 | テンプレート継承 | extends、blockを使用してベーステンプレートを継承可能 | テンプレート設計時 |
| BR-42-02 | 自動エスケープ | HTMLタグは自動的にエスケープされ、XSS攻撃を防止 | デフォルト有効 |
| BR-42-03 | url_for関数 | テンプレート内でurl_for関数を使用してURLを生成可能 | コンテキストにrequestが含まれる場合 |
| BR-42-04 | フィルタ・マクロ | Jinja2の標準フィルタとマクロが使用可能 | 常時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | TemplateNotFound | 指定されたテンプレートファイルが存在しない | テンプレートファイルパスを確認 |
| - | TemplateSyntaxError | テンプレートの構文エラー | テンプレートファイルの構文を修正 |
| - | UndefinedError | コンテキストに存在しない変数を参照 | コンテキストに必要な変数を追加 |

### リトライ仕様

テンプレートレンダリングに対するリトライは行わない

## トランザクション仕様

該当なし

## パフォーマンス要件

- テンプレートはコンパイル後にキャッシュされ、再利用される
- 初回アクセス時のみコンパイルが実行される
- auto_reloadオプションでファイル変更の自動検出が可能

## セキュリティ考慮事項

- 自動エスケープにより、XSS攻撃を防止
- テンプレートファイルへの直接アクセスを防止（適切なディレクトリ設定）
- ユーザー入力をテンプレートとして評価しない（サーバーサイドテンプレートインジェクション対策）

## 備考

- FastAPIではStarletteのJinja2Templatesをそのまま再エクスポートしている
- Jinja2ライブラリのインストールが必要（`pip install jinja2`）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FastAPIでの再エクスポート構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | templating.py | `fastapi/templating.py` | StarletteからのJinja2Templates再エクスポート |

**読解のコツ**: FastAPIのtemplating.pyは1行のみで、Starletteの実装をそのまま使用している。実際の実装はStarletteを参照する必要がある。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | templating.py | `starlette/templating.py` | Jinja2Templatesクラスの完全な実装 |

**主要処理フロー**:
1. **1行目**: `from starlette.templating import Jinja2Templates as Jinja2Templates`

#### Step 3: 使用例を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | examples | FastAPI公式ドキュメント | テンプレートの実際の使用方法 |

**使用例**:
```python
from fastapi import FastAPI, Request
from fastapi.templating import Jinja2Templates

app = FastAPI()
templates = Jinja2Templates(directory="templates")

@app.get("/items/{id}")
async def read_item(request: Request, id: str):
    return templates.TemplateResponse(
        request=request, name="item.html", context={"id": id}
    )
```

### プログラム呼び出し階層図

```
fastapi.templating.Jinja2Templates
    │
    └─ starlette.templating.Jinja2Templates
           │
           ├─ __init__(directory, ...)
           │      └─ jinja2.Environment 初期化
           │
           └─ TemplateResponse(request, name, context)
                  ├─ get_template(name)
                  ├─ render(context)
                  └─ HTMLResponse 生成
```

### データフロー図

```
[入力]                    [処理]                           [出力]

テンプレートファイル ──▶ Jinja2Environment ──────────────▶ HTML文字列
       │                      │                              │
       └─ .html              └─ コンパイル & キャッシュ        │
                                                              ▼
コンテキスト辞書 ──────▶ Template.render() ──────────────▶ TemplateResponse
       │                      │                              │
       └─ {"id": "123"}      └─ 変数置換 & レンダリング       └─ text/html
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| templating.py | `fastapi/templating.py` | ソース | Starlette Jinja2Templatesの再エクスポート |
| templating.py | `starlette/templating.py` | ソース（外部） | Jinja2Templates実装本体 |
| staticfiles.py | `fastapi/staticfiles.py` | ソース | 静的ファイル配信（テンプレートと併用） |
| responses.py | `starlette/responses.py` | ソース（外部） | HTMLResponse基底クラス |
