# 機能設計書 43-StaticFiles

## 概要

本ドキュメントは、FastAPIにおけるStaticFilesの設計と実装について記述する。この機能は静的ファイル（CSS、JavaScript、画像等）の配信機能を提供し、Webアプリケーションのフロントエンドリソース配信に使用される。

### 本機能の処理概要

**業務上の目的・背景**：Webアプリケーションでは、HTML以外にもCSS、JavaScript、画像ファイルなどの静的リソースを配信する必要がある。StaticFilesはこれらのファイルを効率的に配信するための機能を提供する。APIサーバーと静的ファイルサーバーを分離せずに、単一のFastAPIアプリケーションで完結させたい場合に特に有用である。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- CSSスタイルシートの配信
- JavaScriptファイルの配信
- 画像ファイル（PNG、JPG、SVG等）の配信
- フォントファイルの配信
- favicon.icoの配信
- ダウンロード可能なドキュメントの配信

**主要な処理内容**：
1. 静的ファイルディレクトリのマウント
2. ファイルパスの解決とセキュリティチェック
3. ファイルの読み込みとレスポンス生成
4. 適切なContent-Typeヘッダーの設定
5. キャッシュ関連ヘッダーの管理

**関連システム・外部連携**：Starlette StaticFilesクラスをそのまま再エクスポートしており、Jinja2Templatesと組み合わせて使用されることが多い。

**権限による制御**：本機能自体は権限制御を行わない。必要に応じてミドルウェアやルーティングで制御を追加する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | アイテム詳細画面 | 補助機能 | styles.css等の静的ファイルを配信。テンプレート内でurl_for関数で参照 |

## 機能種別

ファイル配信 / 静的リソース管理 / HTTPレスポンス処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| directory | str / PathLike | Yes（packages未指定時） | 静的ファイルを格納するディレクトリパス | 存在するディレクトリであること |
| packages | list[str / tuple] | No | Pythonパッケージから静的ファイルを配信する場合のパッケージリスト | なし |
| html | bool | No | HTMLファイルをデフォルトで配信するか（デフォルト: False） | なし |
| check_dir | bool | No | ディレクトリの存在チェックを行うか（デフォルト: True） | なし |
| follow_symlink | bool | No | シンボリックリンクを追跡するか（デフォルト: False） | なし |

### マウント時のパラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| path | str | Yes | URLパスプレフィックス（例: "/static"） | なし |
| app | StaticFiles | Yes | StaticFilesインスタンス | なし |
| name | str | No | ルート名（url_for用） | なし |

### 入力データソース

- 静的ファイル：指定されたディレクトリ内のファイル
- HTTPリクエスト：ファイルパスを含むURL

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FileResponse | FileResponse | ファイルコンテンツを含むレスポンスオブジェクト |
| body | bytes | ファイルのバイナリデータ |
| headers | dict | Content-Type、Content-Length等のヘッダー |

### 出力先

HTTPレスポンスとしてクライアントに返却

## 処理フロー

### 処理シーケンス

```
1. StaticFilesマウント
   └─ app.mount("/static", StaticFiles(directory="static"), name="static")
2. リクエスト受信
   └─ GET /static/css/styles.css
3. パス解決
   └─ ディレクトリパス + リクエストパス
4. セキュリティチェック
   └─ パストラバーサル攻撃の防止
5. ファイル存在確認
   └─ ファイルが存在するか確認
6. Content-Type決定
   └─ ファイル拡張子からMIMEタイプを推定
7. レスポンス生成
   └─ FileResponseオブジェクトの作成と返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B[パス解決]
    B --> C{パストラバーサル?}
    C -->|Yes| D[403 Forbidden]
    C -->|No| E{ファイル存在?}
    E -->|No| F{ディレクトリ?}
    F -->|Yes| G{html=True?}
    G -->|Yes| H[index.html検索]
    G -->|No| I[404 Not Found]
    F -->|No| I
    H --> J{index.html存在?}
    J -->|Yes| K[ファイル配信]
    J -->|No| I
    E -->|Yes| K
    K --> L[Content-Type設定]
    L --> M[FileResponse返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-01 | パストラバーサル防止 | "../"等を使用したディレクトリ外へのアクセスを禁止 | 常時 |
| BR-43-02 | MIMEタイプ自動設定 | ファイル拡張子に基づいてContent-Typeを自動設定 | 常時 |
| BR-43-03 | ディレクトリインデックス | html=Trueの場合、ディレクトリアクセス時にindex.htmlを返却 | html=True設定時 |
| BR-43-04 | シンボリックリンク制御 | follow_symlink=Falseの場合、シンボリックリンクを追跡しない | デフォルト |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | 指定されたファイルが存在しない | ファイルパスを確認 |
| 403 | Forbidden | パストラバーサル攻撃の検出 | 正当なパスでアクセス |
| 405 | Method Not Allowed | GET/HEAD以外のメソッド | GET/HEADメソッドを使用 |

### リトライ仕様

静的ファイル配信に対するリトライは行わない

## トランザクション仕様

該当なし

## パフォーマンス要件

- ファイルはストリーミング配信され、メモリ効率が良い
- ETagヘッダーによる条件付きリクエストのサポート
- Last-Modifiedヘッダーによるキャッシュ制御

## セキュリティ考慮事項

- パストラバーサル攻撃の防止（ディレクトリ外アクセス禁止）
- シンボリックリンク追跡のデフォルト無効化
- 機密ファイルを静的ディレクトリに配置しないこと

## 備考

- FastAPIではStarletteのStaticFilesをそのまま再エクスポートしている
- aiofilesライブラリのインストールで非同期ファイル読み込みが有効になる

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FastAPIでの再エクスポート構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | staticfiles.py | `fastapi/staticfiles.py` | StarletteからのStaticFiles再エクスポート |

**読解のコツ**: FastAPIのstaticfiles.pyは1行のみで、Starletteの実装をそのまま使用している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | staticfiles.py | `starlette/staticfiles.py` | StaticFilesクラスの完全な実装 |

**主要処理フロー**:
1. **1行目**: `from starlette.staticfiles import StaticFiles as StaticFiles`

#### Step 3: 使用例を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | examples | FastAPI公式ドキュメント | 静的ファイルの実際の使用方法 |

**使用例**:
```python
from fastapi import FastAPI
from fastapi.staticfiles import StaticFiles

app = FastAPI()
app.mount("/static", StaticFiles(directory="static"), name="static")
```

### プログラム呼び出し階層図

```
fastapi.staticfiles.StaticFiles
    │
    └─ starlette.staticfiles.StaticFiles
           │
           ├─ __init__(directory, packages, html, ...)
           │      └─ ディレクトリパス設定
           │
           └─ __call__(scope, receive, send)
                  ├─ get_path(scope)
                  ├─ lookup_path(path)
                  │      ├─ パストラバーサルチェック
                  │      └─ ファイル存在確認
                  └─ FileResponse / PlainTextResponse
```

### データフロー図

```
[入力]                    [処理]                           [出力]

HTTPリクエスト ────────▶ StaticFiles ─────────────────▶ ファイルデータ
       │                      │                              │
       └─ /static/style.css  ├─ パス解決                    └─ bytes
                              ├─ セキュリティチェック
                              ├─ MIMEタイプ判定
                              └─ FileResponse生成
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| staticfiles.py | `fastapi/staticfiles.py` | ソース | Starlette StaticFilesの再エクスポート |
| staticfiles.py | `starlette/staticfiles.py` | ソース（外部） | StaticFiles実装本体 |
| templating.py | `fastapi/templating.py` | ソース | Jinja2Templates（静的ファイルと併用） |
| responses.py | `starlette/responses.py` | ソース（外部） | FileResponse基底クラス |
