# 機能設計書 44-TestClient

## 概要

本ドキュメントは、FastAPIにおけるTestClientの設計と実装について記述する。この機能はAPI統合テスト用のクライアントを提供し、FastAPIアプリケーションのエンドポイントをプログラムから呼び出してテストすることを可能にする。

### 本機能の処理概要

**業務上の目的・背景**：品質の高いAPIを提供するためには、包括的なテストが不可欠である。TestClientは、実際のHTTPサーバーを起動せずにFastAPIアプリケーションをテストするための同期的なテストクライアントを提供する。これにより、CI/CDパイプラインでの自動テスト、開発中の迅速なフィードバック、リグレッションテストの実行が効率的に行える。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- ユニットテストでのAPIエンドポイント検証
- 統合テストでの複数エンドポイント連携確認
- CI/CDパイプラインでの自動テスト実行
- 開発中のAPIレスポンス検証
- WebSocketエンドポイントのテスト
- 認証・認可フローのテスト

**主要な処理内容**：
1. ASGIアプリケーションのラップとテスト環境構築
2. HTTP/HTTPSリクエストの送信とレスポンス取得
3. WebSocket接続のテストサポート
4. セッション管理とCookie処理
5. テストコンテキストでの依存性オーバーライド

**関連システム・外部連携**：Starlette TestClientクラスをそのまま再エクスポートしている。内部的にhttpxライブラリを使用してHTTPリクエストを生成する。

**権限による制御**：本機能自体は権限制御を行わない。テスト時の認証・認可テストはテストコード内で実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はテスト用クライアントであり、直接関連する画面はない |

## 機能種別

テスト支援 / 統合テスト / HTTP/WebSocketクライアント

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| app | ASGIApp | Yes | テスト対象のFastAPIアプリケーション | なし |
| base_url | str | No | ベースURL（デフォルト: "http://testserver"） | 有効なURL形式 |
| raise_server_exceptions | bool | No | サーバー例外を発生させるか（デフォルト: True） | なし |
| root_path | str | No | ASGIルートパス | なし |
| cookies | dict | No | 初期Cookie | なし |

### HTTPリクエストメソッドパラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| url | str | Yes | リクエスト先URL | なし |
| params | dict | No | クエリパラメータ | なし |
| headers | dict | No | HTTPヘッダー | なし |
| json | Any | No | JSONボディ | なし |
| data | dict | No | フォームデータ | なし |
| files | dict | No | アップロードファイル | なし |

### 入力データソース

- テストコードからのAPI呼び出し
- pytest等のテストフレームワークからの実行

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Response | httpx.Response | HTTPレスポンスオブジェクト |
| status_code | int | HTTPステータスコード |
| headers | dict | レスポンスヘッダー |
| json() | Any | JSONパースされたレスポンスボディ |
| text | str | テキストとしてのレスポンスボディ |
| content | bytes | バイト列としてのレスポンスボディ |

### 出力先

テストコード内での検証に使用

## 処理フロー

### 処理シーケンス

```
1. TestClient初期化
   └─ FastAPIアプリケーションのラップ
2. リクエスト発行
   └─ client.get("/items/1")
3. ASGI呼び出し
   └─ アプリケーションへのリクエスト転送
4. レスポンス受信
   └─ Response オブジェクトの生成
5. 検証
   └─ assert response.status_code == 200
```

### フローチャート

```mermaid
flowchart TD
    A[テストコード] --> B[TestClient初期化]
    B --> C[HTTPメソッド呼び出し]
    C --> D[リクエスト構築]
    D --> E[ASGIアプリ呼び出し]
    E --> F[アプリケーション処理]
    F --> G[レスポンス生成]
    G --> H[Response オブジェクト返却]
    H --> I[テストコードで検証]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-44-01 | 同期実行 | TestClientは同期的に動作し、asyncコードを内部で実行 | 常時 |
| BR-44-02 | サーバー例外伝播 | raise_server_exceptions=Trueの場合、サーバー側の例外がテストコードに伝播 | デフォルト設定時 |
| BR-44-03 | セッション維持 | コンテキストマネージャ使用時、セッション（Cookie等）が維持される | with文使用時 |
| BR-44-04 | 依存性オーバーライド | app.dependency_overridesで依存性をモックに置換可能 | テスト時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない。テスト対象アプリの操作は別途考慮）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception伝播 | サーバー側で例外発生（raise_server_exceptions=True時） | テストコードでキャッチして検証 |
| - | httpx.HTTPError | HTTPクライアントエラー | リクエスト内容を確認 |
| - | AssertionError | WebSocket接続エラー | WebSocketエンドポイントを確認 |

### リトライ仕様

テストクライアントに対するリトライは行わない

## トランザクション仕様

テスト時のデータベーストランザクション管理は、テストコード側で実装する（pytest-asyncio + SQLAlchemyロールバック等）

## パフォーマンス要件

- 実際のHTTPサーバーを起動しないため、テスト実行が高速
- 各テストで新規TestClientを作成しても低オーバーヘッド

## セキュリティ考慮事項

- テスト環境でのみ使用することを推奨
- 本番環境のURLをbase_urlに設定しないこと
- 認証情報のハードコーディングを避け、環境変数等で管理

## 備考

- FastAPIではStarletteのTestClientをそのまま再エクスポートしている
- httpxライブラリに依存（Starletteの依存）
- 非同期テストにはpytest-asyncioとhttpx.AsyncClientの組み合わせも使用可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FastAPIでの再エクスポート構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | testclient.py | `fastapi/testclient.py` | StarletteからのTestClient再エクスポート |

**読解のコツ**: FastAPIのtestclient.pyは1行のみで、Starletteの実装をそのまま使用している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | testclient.py | `starlette/testclient.py` | TestClientクラスの完全な実装 |

**主要処理フロー**:
1. **1行目**: `from starlette.testclient import TestClient as TestClient`

#### Step 3: 使用例を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | テストコード | プロジェクトのtestsディレクトリ | TestClientの実際の使用方法 |

**使用例**:
```python
from fastapi import FastAPI
from fastapi.testclient import TestClient

app = FastAPI()

@app.get("/")
def read_root():
    return {"Hello": "World"}

client = TestClient(app)

def test_read_root():
    response = client.get("/")
    assert response.status_code == 200
    assert response.json() == {"Hello": "World"}
```

### プログラム呼び出し階層図

```
fastapi.testclient.TestClient
    │
    └─ starlette.testclient.TestClient
           │
           ├─ __init__(app, ...)
           │      └─ httpx.Client 初期化
           │
           ├─ get(url, ...)
           │      └─ request("GET", url, ...)
           │
           ├─ post(url, ...)
           │      └─ request("POST", url, ...)
           │
           └─ websocket_connect(url, ...)
                  └─ WebSocket接続管理
```

### データフロー図

```
[入力]                    [処理]                           [出力]

テストコード ─────────▶ TestClient ──────────────────▶ Response
       │                      │                              │
       └─ リクエストパラメータ ├─ httpx.Client構築           └─ status_code
          - URL               ├─ ASGIトランスポート             json()
          - headers           ├─ アプリ呼び出し                 text
          - json/data         └─ Response構築                   content
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| testclient.py | `fastapi/testclient.py` | ソース | Starlette TestClientの再エクスポート |
| testclient.py | `starlette/testclient.py` | ソース（外部） | TestClient実装本体 |
| applications.py | `fastapi/applications.py` | ソース | テスト対象のFastAPIアプリケーション |
| routing.py | `fastapi/routing.py` | ソース | エンドポイントルーティング |
