# 機能設計書 46-run_in_threadpool

## 概要

本ドキュメントは、FastAPIにおけるrun_in_threadpoolの設計と実装について記述する。この機能は同期関数を非同期コンテキストで実行するための機能を提供し、ブロッキング処理の非同期化に使用される。

### 本機能の処理概要

**業務上の目的・背景**：FastAPIは非同期（async/await）ベースのフレームワークであるが、既存のライブラリや処理の中には同期的にしか動作しないものが多い。データベースドライバ、ファイルI/O、CPU集約的な計算処理などは、非同期イベントループをブロックしてしまう可能性がある。run_in_threadpoolはこれらの同期処理を別スレッドで実行し、メインの非同期イベントループをブロックしないようにする。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- 同期的なデータベースドライバの使用
- 同期的なファイルI/O処理
- CPU集約的な計算処理
- 同期的な外部ライブラリの呼び出し
- レガシーコードとの統合

**主要な処理内容**：
1. 同期関数をスレッドプールで実行
2. 非同期コンテキストからの結果待機
3. 例外の適切な伝播
4. コンテキストマネージャの非同期ラップ（contextmanager_in_threadpool）

**関連システム・外部連携**：Starlette concurrencyモジュールを再エクスポート。内部的にanyioライブラリを使用してスレッドプールを管理する。

**権限による制御**：本機能自体は権限制御を行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は内部ユーティリティであり、直接関連する画面はない |

## 機能種別

並行処理 / スレッドプール管理 / 非同期変換

## 入力仕様

### 入力パラメータ（run_in_threadpool）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| func | Callable[..., T] | Yes | 実行する同期関数 | 呼び出し可能なオブジェクト |
| *args | Any | No | 関数に渡す位置引数 | なし |
| **kwargs | Any | No | 関数に渡すキーワード引数 | なし |

### 入力パラメータ（contextmanager_in_threadpool）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| cm | AbstractContextManager[T] | Yes | 同期コンテキストマネージャ | コンテキストマネージャプロトコル実装 |

### 入力データソース

アプリケーションコードからの呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | T | 同期関数の戻り値 |
| AsyncGenerator[T, None] | 非同期ジェネレータ | contextmanager_in_threadpoolの戻り値 |

### 出力先

呼び出し元の非同期コードに返却

## 処理フロー

### 処理シーケンス（run_in_threadpool）

```
1. run_in_threadpool呼び出し
   └─ 同期関数とパラメータを受け取る
2. スレッドプールへの委譲
   └─ anyio.to_thread.run_syncで実行
3. 結果待機
   └─ awaitで非同期的に結果を待つ
4. 結果返却
   └─ 同期関数の戻り値を返す
```

### 処理シーケンス（contextmanager_in_threadpool）

```
1. コンテキストマネージャ受け取り
   └─ 同期コンテキストマネージャを受け取る
2. __enter__の実行
   └─ run_in_threadpoolで__enter__を実行
3. yieldで値を返す
   └─ コンテキストマネージャの値を返す
4. __exit__の実行
   └─ CapacityLimiterを使用して__exit__を実行
```

### フローチャート

```mermaid
flowchart TD
    A[非同期コードから呼び出し] --> B[run_in_threadpool]
    B --> C[anyio.to_thread.run_sync]
    C --> D[スレッドプールで同期関数実行]
    D --> E{例外発生?}
    E -->|No| F[結果返却]
    E -->|Yes| G[例外伝播]
    F --> H[呼び出し元に返却]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | イベントループ非ブロック | メインの非同期イベントループをブロックしない | 常時 |
| BR-46-02 | 例外伝播 | 同期関数内で発生した例外は呼び出し元に伝播される | 例外発生時 |
| BR-46-03 | コンテキスト維持 | 同期関数内でも現在のコンテキスト変数が維持される | 常時 |
| BR-46-04 | 終了時リミッター | contextmanager_in_threadpoolでは__exit__時にCapacityLimiterを使用 | contextmanager使用時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Exception伝播 | 同期関数内で例外発生 | 呼び出し元でキャッチして処理 |
| - | TypeError | 呼び出し不可能なオブジェクトを渡した | Callableを渡す |

### リトライ仕様

リトライは呼び出し元のコードで実装する

## トランザクション仕様

トランザクション管理は呼び出し元のコードで実装する

## パフォーマンス要件

- スレッドプールのデフォルトサイズはシステム依存（anyioの設定）
- CapacityLimiterによりリソース競合を防止
- 頻繁な呼び出しでもスレッドの再利用により効率的

## セキュリティ考慮事項

- スレッドセーフでない処理に注意
- 共有リソースへのアクセス時は適切なロックを使用

## 備考

- FastAPIではStarletteのrun_in_threadpoolを再エクスポートしている
- contextmanager_in_threadpoolはFastAPI独自の実装
- コメント（21-26行目）にレースコンディション/デッドロック防止の説明あり

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

型定義とインポート構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | concurrency.py | `fastapi/concurrency.py` | インポート文と型定義 |

**読解のコツ**: concurrency.pyは再エクスポートと独自実装の両方を含む。Starletteからの再エクスポート（8-12行目）とFastAPI独自のcontextmanager_in_threadpool（17-41行目）を区別して読む。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | concurrency.py | `fastapi/concurrency.py` | run_in_threadpoolの再エクスポート |

**主要処理フロー**:
1. **8-12行目**: Starletteから`iterate_in_threadpool`、`run_in_threadpool`、`run_until_first_complete`を再エクスポート

#### Step 3: contextmanager_in_threadpoolを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | concurrency.py | `fastapi/concurrency.py` | FastAPI独自のコンテキストマネージャラッパー |

**主要処理フロー**:
- **17-20行目**: @asynccontextmanagerデコレータと関数シグネチャ
- **21-26行目**: レースコンディション/デッドロック防止のコメント
- **27行目**: CapacityLimiter(1)で__exit__用のリミッター作成
- **29行目**: run_in_threadpoolで__enter__を実行
- **30-37行目**: 例外処理（__exit__への例外情報渡し）
- **38-41行目**: 正常終了時の__exit__呼び出し

### プログラム呼び出し階層図

```
run_in_threadpool (Starlette再エクスポート)
    │
    └─ starlette.concurrency.run_in_threadpool
           │
           └─ anyio.to_thread.run_sync

contextmanager_in_threadpool (FastAPI独自実装)
    │
    ├─ CapacityLimiter(1) 作成
    │
    ├─ run_in_threadpool(cm.__enter__)
    │      └─ __enter__をスレッドプールで実行
    │
    └─ anyio.to_thread.run_sync(cm.__exit__, ...)
           └─ __exit__をリミッター付きで実行
```

### データフロー図

```
[入力]                    [処理]                           [出力]

同期関数 ──────────────▶ run_in_threadpool ────────────▶ 戻り値
       │                      │                              │
       └─ func, args, kwargs  └─ スレッドプール実行           └─ T

同期CM ────────────────▶ contextmanager_in_threadpool ──▶ 非同期CM
       │                      │                              │
       └─ AbstractContextManager ├─ __enter__スレッド実行    └─ AsyncGenerator[T]
                              └─ __exit__リミッター付き実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| concurrency.py | `fastapi/concurrency.py` | ソース | run_in_threadpool再エクスポートとcontextmanager_in_threadpool実装 |
| concurrency.py | `starlette/concurrency.py` | ソース（外部） | run_in_threadpool実装本体 |
| routing.py | `fastapi/routing.py` | ソース | リクエスト処理でのrun_in_threadpool使用 |
