# 機能設計書 47-generate_unique_id

## 概要

本ドキュメントは、FastAPIにおけるgenerate_unique_idの設計と実装について記述する。この機能はエンドポイントの一意なIDを生成する機能を提供し、OpenAPIのoperationIdに使用される。

### 本機能の処理概要

**業務上の目的・背景**：OpenAPI仕様では、各APIオペレーションに一意なoperationIdを付与することが推奨されている。このIDはAPIクライアントの自動生成、ドキュメントの参照、API管理ツールでの識別などに使用される。generate_unique_idはFastAPIのルート定義から自動的に一意なIDを生成し、開発者の負担を軽減する。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- OpenAPIスキーマ生成時のoperationId自動付与
- APIクライアント（TypeScript、Python等）の自動生成
- Swagger UI / ReDocでのAPI識別
- APIゲートウェイでのルーティング設定

**主要な処理内容**：
1. ルートオブジェクトから名前（関数名）を取得
2. パスフォーマットを取得
3. 非英数字を"_"に置換
4. HTTPメソッド（小文字）を末尾に付加

**関連システム・外部連携**：OpenAPIスキーマ生成機能と連携。routing.pyでデフォルトのIDジェネレータとして使用される。

**権限による制御**：本機能自体は権限制御を行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | 生成されたoperationIdがSwagger UIのAPIドキュメントで使用される |
| 2 | ReDoc | 補助機能 | 生成されたoperationIdがReDocのAPIドキュメントで使用される |

## 機能種別

ユーティリティ / ID生成 / OpenAPI連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| route | APIRoute | Yes | FastAPIのルートオブジェクト | nameとpath_formatとmethodsが存在すること |

### 入力データソース

FastAPIルーティングシステムからのAPIRouteオブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| operation_id | str | 一意なオペレーションID（例: "read_item_items__item_id__get"） |

### 出力先

OpenAPIスキーマのoperationIdフィールド

## 処理フロー

### 処理シーケンス

```
1. ルート名取得
   └─ route.name（エンドポイント関数名）
2. パスフォーマット取得
   └─ route.path_format（例: "/items/{item_id}"）
3. 文字列結合
   └─ f"{name}{path_format}"
4. 非英数字置換
   └─ re.sub(r"\W", "_", operation_id)
5. HTTPメソッド付加
   └─ f"{operation_id}_{method.lower()}"
6. ID返却
   └─ "read_item_items__item_id__get"
```

### フローチャート

```mermaid
flowchart TD
    A[APIRoute受け取り] --> B[route.name取得]
    B --> C[route.path_format取得]
    C --> D[名前とパスを結合]
    D --> E[非英数字を_に置換]
    E --> F[HTTPメソッドを取得]
    F --> G[メソッドを小文字で末尾に付加]
    G --> H[operation_id返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-47-01 | 非英数字置換 | 英数字以外の文字は"_"に置換される | 常時 |
| BR-47-02 | メソッド小文字 | HTTPメソッドは小文字で付加される | 常時 |
| BR-47-03 | 最初のメソッド使用 | route.methodsの最初の要素が使用される | 常時 |
| BR-47-04 | 一意性 | 同一パス・メソッドの組み合わせでは同一IDが生成される | 常時 |

### 計算ロジック

```python
operation_id = f"{route.name}{route.path_format}"
operation_id = re.sub(r"\W", "_", operation_id)
operation_id = f"{operation_id}_{list(route.methods)[0].lower()}"
```

例:
- 入力: name="read_item", path_format="/items/{item_id}", methods={"GET"}
- 出力: "read_item_items__item_id__get"

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | AssertionError | route.methodsがNoneまたは空 | ルート定義にHTTPメソッドを指定 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 正規表現による置換は軽量な処理
- ルート登録時に一度だけ実行される

## セキュリティ考慮事項

- operationIdは公開情報として扱われるため、機密性のある名前を使用しないこと

## 備考

- generate_operation_id_for_path関数は非推奨（deprecated）となっており、本関数が推奨
- カスタムのID生成関数を使用する場合は、ルートデコレータのoperation_idパラメータで指定可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

APIRouteクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | routing.py | `fastapi/routing.py` | APIRouteクラスの定義（name、path_format、methods属性） |

**読解のコツ**: APIRouteはStarletteのRouteを継承しており、FastAPI固有の拡張が加わっている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | utils.py | `fastapi/utils.py` | generate_unique_id関数の実装 |

**主要処理フロー**:
1. **123行目**: 関数定義 `def generate_unique_id(route: "APIRoute") -> str:`
2. **124行目**: 名前とパスの結合 `operation_id = f"{route.name}{route.path_format}"`
3. **125行目**: 非英数字の置換 `operation_id = re.sub(r"\W", "_", operation_id)`
4. **126行目**: methodsのアサーション `assert route.methods`
5. **127行目**: メソッド付加 `operation_id = f"{operation_id}_{list(route.methods)[0].lower()}"`
6. **128行目**: 返却 `return operation_id`

#### Step 3: 使用箇所を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | generate_unique_idのインポートと使用 |

**主要処理フロー**:
- **55行目**: `from fastapi.utils import generate_unique_id`
- ルート作成時にデフォルトのoperation_id生成関数として使用

### プログラム呼び出し階層図

```
APIRoute登録
    │
    └─ generate_unique_id(route)
           │
           ├─ route.name 取得
           │
           ├─ route.path_format 取得
           │
           ├─ re.sub(r"\W", "_", ...) # 非英数字置換
           │
           └─ f"..._{method.lower()}" # メソッド付加
                  │
                  └─ operation_id 返却
```

### データフロー図

```
[入力]                    [処理]                           [出力]

APIRoute ──────────────▶ generate_unique_id ────────────▶ operation_id
       │                      │                              │
       ├─ name="read_item"    ├─ 文字列結合                  └─ "read_item_items__item_id__get"
       ├─ path_format="/items/{item_id}"  ├─ 非英数字置換
       └─ methods={"GET"}     └─ メソッド付加
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| utils.py | `fastapi/utils.py` | ソース | generate_unique_id関数の実装 |
| routing.py | `fastapi/routing.py` | ソース | APIRouteクラスでのgenerate_unique_id使用 |
| openapi/utils.py | `fastapi/openapi/utils.py` | ソース | OpenAPIスキーマ生成でのoperationId使用 |
