# 機能設計書 5-Queryパラメータ

## 概要

本ドキュメントは、FastAPIフレームワークにおけるQueryパラメータ機能の設計を記述する。URLクエリ文字列から値を受け取るパラメータ定義機能であり、デフォルト値、必須/任意、型変換、バリデーション機能を提供する。

### 本機能の処理概要

Queryパラメータは、URLのクエリ文字列（例：`/items?skip=0&limit=10`）から値を抽出し、パスオペレーション関数の引数として渡す機能である。PathパラメータとDifferent、任意パラメータやデフォルト値の設定が可能で、検索やフィルタリングなどの付加的なパラメータに適している。

**業務上の目的・背景**：REST API設計において、リソースのフィルタリング、ページネーション、ソート条件などはクエリパラメータで受け取るのが一般的である。本機能により、これらのオプショナルなパラメータを型安全かつ宣言的に定義でき、バリデーションも自動化される。開発者はパラメータ抽出やエラーハンドリングのコードを書く必要がなくなる。

**機能の利用シーン**：
- リスト取得APIのページネーション（`?skip=0&limit=10`）
- 検索API（`?q=keyword&category=electronics`）
- フィルタリング（`?status=active&min_price=100`）
- ソート指定（`?sort_by=created_at&order=desc`）

**主要な処理内容**：
1. クエリ文字列からパラメータ値の抽出
2. デフォルト値の適用（値が未指定の場合）
3. 型ヒントに基づく型変換
4. バリデーションルールに基づく値検証
5. OpenAPIスキーマへのパラメータ情報追加

**関連システム・外部連携**：Pydantic（FieldInfo基底クラス、バリデーション）、OpenAPI（スキーマ生成）

**権限による制御**：本機能自体は権限制御を持たない

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連なし |

## 機能種別

パラメータ抽出 / バリデーション / 型変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| default | Any | No | デフォルト値（Undefinedで必須、値指定で任意） | 任意 |
| alias | Optional[str] | No | パラメータの別名 | 文字列 |
| title | Optional[str] | No | 人間可読なタイトル（OpenAPI用） | 文字列 |
| description | Optional[str] | No | パラメータの説明（OpenAPI用） | 文字列 |
| gt | Optional[float] | No | より大きい（Greater Than） | 数値のみ |
| ge | Optional[float] | No | 以上（Greater than or Equal） | 数値のみ |
| lt | Optional[float] | No | より小さい（Less Than） | 数値のみ |
| le | Optional[float] | No | 以下（Less than or Equal） | 数値のみ |
| min_length | Optional[int] | No | 最小文字列長 | 文字列のみ |
| max_length | Optional[int] | No | 最大文字列長 | 文字列のみ |
| pattern | Optional[str] | No | 正規表現パターン | 文字列のみ |
| examples | Optional[list[Any]] | No | 例示値リスト | 任意 |
| deprecated | Union[bool, str, None] | No | 非推奨フラグ | ブール値または文字列 |
| include_in_schema | bool | No | OpenAPIスキーマへの含有（デフォルト: True） | ブール値 |

### 入力データソース

- URLクエリ文字列
- パスオペレーション定義（関数シグネチャ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 変換後の値 | 型ヒントで指定した型 | 抽出・変換・検証済みのパラメータ値 |

### 出力先

- パスオペレーション関数の引数

## 処理フロー

### 処理シーケンス

```
1. クエリ文字列パース
   └─ URLからクエリパラメータを抽出
2. 値の存在チェック
   └─ 値がなければデフォルト値を使用、必須なら422エラー
3. 型変換
   └─ 型ヒントに基づく変換
4. バリデーション実行
   └─ Pydanticによるバリデーションルール検証
5. 関数引数への受け渡し
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[クエリ文字列パース]
    B --> C{値が存在?}
    C -->|Yes| D[型変換]
    C -->|No| E{デフォルト値あり?}
    E -->|Yes| F[デフォルト値使用]
    E -->|No| G[422エラー]
    D --> H{バリデーション}
    F --> H
    H -->|成功| I[関数引数に設定]
    H -->|失敗| G
    I --> J[パスオペレーション実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | デフォルト値による任意化 | default指定時、パラメータは任意となる | default指定時 |
| BR-002 | Undefined時は必須 | defaultがUndefinedの場合、パラメータは必須 | default未指定時 |
| BR-003 | リスト型対応 | list[str]等の型で同名パラメータを複数回受け取り可能 | リスト型ヒント時 |
| BR-004 | None許容 | Optional[T]型でNone（値なし）を許容 | Optional型ヒント時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 422 | Unprocessable Entity | 必須パラメータが未指定 | パラメータを指定する |
| 422 | Unprocessable Entity | 型変換失敗 | 正しい型の値を指定 |
| 422 | Unprocessable Entity | バリデーション失敗 | バリデーション要件を満たす値を使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- クエリ文字列パースは軽量な処理
- 同一リクエストで複数のQueryパラメータを効率的に処理

## セキュリティ考慮事項

- クエリパラメータはURLに含まれるためログに記録されやすい
- 機密情報（パスワード等）をクエリパラメータで送信しない
- max_lengthでDoS攻撃対策

## 備考

- QueryクラスはParamを継承し、in_=ParamTypes.queryが設定される
- Swagger UIでクエリパラメータ入力フォームが自動生成される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Queryクラスの定義と継承関係を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | Queryクラスの定義（222-301行目） |

**読解のコツ**:
- **222-223行目**: QueryクラスはParamを継承、in_=ParamTypes.query
- **227行目**: default: Any = Undefined（Pathと異なりデフォルト値設定可能）

#### Step 2: 関数インターフェースを理解する

Query()関数の定義を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | param_functions.py | `fastapi/param_functions.py` | Query関数の定義（340-639行目） |

**主要処理フロー**:
1. **340-348行目**: defaultパラメータ定義（Undefined可能）
2. **609-639行目**: params.Queryインスタンスの生成と返却

### プログラム呼び出し階層図

```
@app.get("/items")
async def get_items(skip: int = Query(0), limit: int = Query(10)):
    │
    ├─ get_dependant() [dependencies/utils.py]
    │      └─ クエリパラメータの依存性として登録
    │
    └─ solve_dependencies() [dependencies/utils.py]
           │
           ├─ request.query_params取得
           ├─ デフォルト値適用
           ├─ Pydanticバリデーション
           └─ 関数引数への値設定
```

### データフロー図

```
[入力]                     [処理]                           [出力]

?skip=0&limit=10 ───▶ クエリパースparse_qs ───▶ {"skip": "0", "limit": "10"}
                                                    │
                                                    ▼
                                               型変換（str→int）
                                                    │
                                                    ▼
                                               バリデーション
                                                    │
                                                    ▼
                                   skip: int = 0, limit: int = 10 ───▶ 関数引数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| param_functions.py | `fastapi/param_functions.py` | ソース | Query関数定義 |
| params.py | `fastapi/params.py` | ソース | Queryクラス定義 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | 依存性解決でのクエリパラメータ処理 |
