# 機能設計書 50-型ヒント・型定義

## 概要

本ドキュメントは、FastAPIにおける型ヒント・型定義機能の設計と実装について記述する。この機能はカスタム型定義とヒントを提供し、DecoratedCallable、IncEx等のFastAPI固有の型を定義する。

### 本機能の処理概要

**業務上の目的・背景**：型ヒントはPythonコードの品質向上、IDEのサポート強化、ドキュメント自動生成に貢献する。FastAPIは型ヒントを積極的に活用するフレームワークであり、フレームワーク固有の型を一元的に定義することで、コードの一貫性と可読性を高めている。また、静的型チェッカー（mypy、pyright等）との連携を容易にする。

**機能の利用シーン**：本機能は以下のシーンで利用される：
- デコレータ関数の型アノテーション
- レスポンスモデルのinclude/exclude指定
- 依存性キャッシュのキー定義
- モデル名マッピング
- Union型のエイリアス

**主要な処理内容**：
1. DecoratedCallable型変数の定義（デコレータ用）
2. IncEx型の定義（include/exclude指定用）
3. ModelNameMap型の定義（モデル名マッピング用）
4. DependencyCacheKey型の定義（依存性キャッシュ用）
5. UnionType型の取得（Python 3.10以降の新構文対応）

**関連システム・外部連携**：Pydantic BaseModel、Python typing、typing_extensions

**権限による制御**：本機能自体は権限制御を行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は型定義であり、直接関連する画面はない |

## 機能種別

型定義 / 型ヒント / ユーティリティ

## 入力仕様

### 定義される型

| 型名 | 定義 | 説明 |
|-----|------|------|
| DecoratedCallable | TypeVar("DecoratedCallable", bound=Callable[..., Any]) | デコレータが返す関数の型 |
| UnionType | getattr(types, "UnionType", Union) | Python 3.10+のUnion型エイリアス |
| ModelNameMap | dict[Union[type[BaseModel], type[Enum]], str] | モデル型から名前へのマッピング |
| IncEx | Union[set[int], set[str], dict[int, Any], dict[str, Any]] | include/excludeパラメータ用 |
| DependencyCacheKey | tuple[Optional[Callable[..., Any]], tuple[str, ...], str] | 依存性キャッシュのキー |

### 入力データソース

型定義はコンパイル時に評価され、実行時の入力はない

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| DecoratedCallable | TypeVar | 型変数 |
| UnionType | type | Union型またはUnionType |
| ModelNameMap | type alias | 辞書型エイリアス |
| IncEx | type alias | Union型エイリアス |
| DependencyCacheKey | type alias | タプル型エイリアス |

### 出力先

FastAPI内部のモジュールで型アノテーションとして使用

## 処理フロー

### 処理シーケンス

```
1. モジュールインポート
   └─ import types, typing, pydantic
2. TypeVar定義
   └─ DecoratedCallable = TypeVar(...)
3. UnionType取得
   └─ getattr(types, "UnionType", Union)
4. 型エイリアス定義
   └─ ModelNameMap, IncEx, DependencyCacheKey
```

### フローチャート

```mermaid
flowchart TD
    A[types.pyインポート] --> B[必要なモジュールインポート]
    B --> C[DecoratedCallable TypeVar定義]
    C --> D{Python 3.10+?}
    D -->|Yes| E[types.UnionType取得]
    D -->|No| F[Union使用]
    E --> G[ModelNameMap定義]
    F --> G
    G --> H[IncEx定義]
    H --> I[DependencyCacheKey定義]
    I --> J[型としてエクスポート]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-50-01 | TypeVar境界 | DecoratedCallableはCallableにバインドされる | 常時 |
| BR-50-02 | UnionType互換 | Python 3.10未満ではUnionを使用 | Python < 3.10 |
| BR-50-03 | 型チェック | 定義された型は静的型チェッカーで検証可能 | 型チェック実行時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能自体はデータベース操作を行わない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 型定義自体はエラーを発生させない | - |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 型定義はモジュールロード時に一度だけ評価される
- 実行時のパフォーマンスへの影響はない

## セキュリティ考慮事項

- 型定義はセキュリティに直接影響しない
- 型チェックによりバグの早期発見に貢献

## 備考

- types.pyは12行のシンプルな実装
- Python 3.10以降のパイプ構文（X | Y）対応のためUnionTypeを定義
- 型エイリアスはコードの可読性向上に貢献

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

types.pyの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.py | `fastapi/types.py` | 全ての型定義 |

**読解のコツ**: types.pyは12行のシンプルな実装。各型エイリアスの用途を理解することが重要。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | types.py | `fastapi/types.py` | 全コード |

**主要処理フロー**:
1. **1-5行目**: インポート文（types, Enum, typing, pydantic）
2. **7行目**: DecoratedCallable TypeVar定義
3. **8行目**: UnionType取得（getattr でPython 3.10+対応）
4. **9行目**: ModelNameMap型エイリアス定義
5. **10行目**: IncEx型エイリアス定義
6. **11行目**: DependencyCacheKey型エイリアス定義

#### Step 3: 使用箇所を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | routing.py | `fastapi/routing.py` | DecoratedCallable、IncExの使用 |
| 3-2 | _compat/v2.py | `fastapi/_compat/v2.py` | ModelNameMap、IncExの使用 |

**使用例**:
```python
# DecoratedCallableの使用（デコレータ定義）
def api_route(
    path: str,
    ...
) -> Callable[[DecoratedCallable], DecoratedCallable]:
    ...

# IncExの使用（レスポンスモデル設定）
response_model_include: Optional[IncEx] = None
response_model_exclude: Optional[IncEx] = None
```

### プログラム呼び出し階層図

```
fastapi/types.py
    │
    ├─ DecoratedCallable
    │      └─ routing.py (デコレータ定義)
    │      └─ applications.py (デコレータ定義)
    │
    ├─ UnionType
    │      └─ _compat/shared.py (Union型判定)
    │      └─ _compat/v2.py (Union型判定)
    │
    ├─ ModelNameMap
    │      └─ _compat/v2.py (スキーマ生成)
    │      └─ openapi/utils.py (OpenAPI生成)
    │
    ├─ IncEx
    │      └─ routing.py (レスポンス設定)
    │      └─ _compat/v2.py (シリアライゼーション)
    │
    └─ DependencyCacheKey
           └─ dependencies/utils.py (キャッシュ管理)
```

### データフロー図

```
[定義]                    [使用]                           [効果]

DecoratedCallable ────▶ @app.get() デコレータ ────────▶ 型安全なデコレータ
       │                      │                              │
       └─ TypeVar             └─ 戻り値型アノテーション       └─ IDE補完サポート

IncEx ────────────────▶ response_model_include ───────▶ 型安全なパラメータ
       │                      │                              │
       └─ Union型エイリアス   └─ APIRoute設定               └─ 静的型チェック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| types.py | `fastapi/types.py` | ソース | 型定義の一元管理 |
| routing.py | `fastapi/routing.py` | ソース | DecoratedCallable、IncExの使用 |
| _compat/shared.py | `fastapi/_compat/shared.py` | ソース | UnionTypeの使用 |
| _compat/v2.py | `fastapi/_compat/v2.py` | ソース | ModelNameMap、IncExの使用 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | DependencyCacheKeyの使用 |
