# 機能設計書 8-Bodyパラメータ

## 概要

本ドキュメントは、FastAPIフレームワークにおけるBodyパラメータ機能の設計を記述する。リクエストボディからJSON形式等のデータを受け取るパラメータ定義機能であり、Pydanticモデルとの連携によるバリデーション機能を提供する。

### 本機能の処理概要

Bodyパラメータは、HTTPリクエストのボディ（主にJSON形式）からデータを抽出し、パスオペレーション関数の引数として渡す機能である。Pydanticモデルと連携することで、複雑なネスト構造を持つデータの受け取りとバリデーションを実現する。

**業務上の目的・背景**：REST APIにおいて、POST/PUT/PATCHリクエストでクライアントからデータを受け取る際、リクエストボディを使用するのが一般的である。ユーザー登録情報、注文データ、設定更新など、構造化されたデータを安全に受け取る必要がある。本機能とPydanticの組み合わせにより、データスキーマの定義、自動バリデーション、ドキュメント生成を一体化できる。

**機能の利用シーン**：
- ユーザー登録・更新API（ユーザー情報のJSONを受け取り）
- 注文作成API（注文詳細のJSONを受け取り）
- 設定更新API（設定値のJSONを受け取り）
- バッチ処理API（複数アイテムのJSONを受け取り）

**主要な処理内容**：
1. リクエストボディの読み取り
2. Content-Typeに基づくパース（JSONの場合はjson.loads）
3. Pydanticモデルへのマッピングとバリデーション
4. OpenAPIスキーマへのリクエストボディ情報追加

**関連システム・外部連携**：Pydantic（モデル定義、バリデーション）、OpenAPI（スキーマ生成）

**権限による制御**：本機能自体は権限制御を持たない

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連なし |

## 機能種別

パラメータ抽出 / バリデーション / データマッピング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| default | Any | No | デフォルト値（Undefinedで必須） | 任意 |
| embed | Union[bool, None] | No | ボディをキー付きでネストするか（デフォルト: None） | ブール値 |
| media_type | str | No | メディアタイプ（デフォルト: "application/json"） | MIMEタイプ |
| alias | Optional[str] | No | パラメータの別名 | 文字列 |
| title | Optional[str] | No | 人間可読なタイトル（OpenAPI用） | 文字列 |
| description | Optional[str] | No | パラメータの説明（OpenAPI用） | 文字列 |
| gt | Optional[float] | No | より大きい | 数値のみ |
| ge | Optional[float] | No | 以上 | 数値のみ |
| lt | Optional[float] | No | より小さい | 数値のみ |
| le | Optional[float] | No | 以下 | 数値のみ |
| min_length | Optional[int] | No | 最小文字列長 | 文字列のみ |
| max_length | Optional[int] | No | 最大文字列長 | 文字列のみ |
| pattern | Optional[str] | No | 正規表現パターン | 文字列のみ |
| examples | Optional[list[Any]] | No | 例示値リスト | 任意 |
| deprecated | Union[bool, str, None] | No | 非推奨フラグ | ブール値または文字列 |
| include_in_schema | bool | No | OpenAPIスキーマへの含有（デフォルト: True） | ブール値 |

### 入力データソース

- HTTPリクエストボディ（JSON形式）
- パスオペレーション定義（関数シグネチャ、Pydanticモデル型ヒント）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 変換後の値 | 型ヒントで指定した型（Pydanticモデル等） | パース・変換・検証済みのボディデータ |

### 出力先

- パスオペレーション関数の引数

## 処理フロー

### 処理シーケンス

```
1. リクエストボディ読み取り
   └─ await request.body()でバイト列として取得
2. Content-Typeチェック
   └─ application/jsonの場合JSONパース実行
3. JSONパース
   └─ json.loads()でPythonオブジェクトに変換
4. Pydanticバリデーション
   └─ 型ヒントのPydanticモデルにマッピング・検証
5. 関数引数への受け渡し
```

### フローチャート

```mermaid
flowchart TD
    A[HTTPリクエスト受信] --> B[Content-Typeチェック]
    B --> C{JSON?}
    C -->|Yes| D[request.json取得]
    C -->|No| E[request.body取得]
    D --> F[Pydanticモデル変換]
    E --> F
    F --> G{バリデーション}
    G -->|成功| H[関数引数に設定]
    G -->|失敗| I[422エラー]
    H --> J[パスオペレーション実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 単一Body自動認識 | Pydanticモデル型の引数は自動的にBodyとして認識 | Pydanticモデル型ヒント時 |
| BR-002 | 複数Body時のembed | 複数のBody引数がある場合、自動的にembed=Trueとなる | 複数Body引数時 |
| BR-003 | media_type指定 | media_typeはOpenAPIに反映されるが、パースには影響しない | media_type指定時 |

### 計算ロジック

該当なし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 400 | Bad Request | JSONパースエラー | 正しいJSON形式で送信 |
| 422 | Unprocessable Entity | Pydanticバリデーションエラー | スキーマに適合するデータを送信 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- JSONパースはストリーミングではなく一括読み込み
- 大容量データの場合はメモリ使用量に注意

## セキュリティ考慮事項

- Content-Length制限でDoS対策
- 深いネスト構造による再帰的パース攻撃に注意
- 入力値の適切なサニタイズ

## 備考

- BodyクラスはPydanticのFieldInfoを継承
- Pydanticモデルの型ヒントがあれば、Body()は暗黙的に適用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Bodyクラスの定義を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | params.py | `fastapi/params.py` | Bodyクラスの定義（470-579行目） |

**読解のコツ**:
- **470行目**: BodyクラスはFieldInfoを直接継承（Paramではない）
- **477行目**: embed: Union[bool, None] = None（複数Body時に自動設定）
- **478行目**: media_type: str = "application/json"
- **518行目**: self.embed = embed
- **519行目**: self.media_type = media_type

#### Step 2: 関数インターフェースを理解する

Body()関数の定義を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | param_functions.py | `fastapi/param_functions.py` | Body関数の定義（1258-1582行目） |

**主要処理フロー**:
1. **1279-1292行目**: embedパラメータの定義と説明
2. **1293-1301行目**: media_typeパラメータの定義
3. **1550-1582行目**: params.Bodyインスタンスの生成と返却

### プログラム呼び出し階層図

```
@app.post("/items")
async def create_item(item: Item):  # ItemはPydanticモデル
    │
    ├─ get_dependant() [dependencies/utils.py]
    │      └─ Bodyパラメータとして認識・登録
    │
    ├─ get_body_field() [dependencies/utils.py]
    │      └─ ボディフィールドの生成
    │
    └─ solve_dependencies() [dependencies/utils.py]
           │
           ├─ await request.json() or request.body()
           ├─ Pydanticモデル変換・バリデーション
           └─ 関数引数への値設定
```

### データフロー図

```
[入力]                     [処理]                           [出力]

POST /items
Content-Type: application/json
{"name": "foo", "price": 100}
         │
         ▼
    request.json() ───▶ {"name": "foo", "price": 100}
                               │
                               ▼
                        Pydantic変換 ───▶ Item(name="foo", price=100)
                               │
                               ▼
                        バリデーション
                               │
                               ▼
                        item: Item ───▶ 関数引数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| param_functions.py | `fastapi/param_functions.py` | ソース | Body関数定義 |
| params.py | `fastapi/params.py` | ソース | Bodyクラス定義 |
| dependencies/utils.py | `fastapi/dependencies/utils.py` | ソース | ボディ処理、get_body_field |
| routing.py | `fastapi/routing.py` | ソース | リクエストボディ読み取り処理 |
