# 画面設計書 4-アイテム詳細画面

## 概要

本ドキュメントは、FastAPIフレームワークのドキュメント用サンプルコードとして提供されるJinja2テンプレートを使用したアイテム詳細画面の設計仕様を記載する。

### 本画面の処理概要

アイテム詳細画面は、Jinja2テンプレートエンジンを使用してHTMLレスポンスを返すサンプル実装である。URLパスからアイテムIDを取得し、テンプレートに埋め込んで表示する。FastAPIにおけるテンプレートレンダリングの基本的な使い方を示すための教育目的のコードである。

**業務上の目的・背景**：FastAPIは主にAPI開発向けのフレームワークだが、従来のサーバーサイドレンダリングによるHTMLページ生成も可能である。このサンプルは、Jinja2テンプレートエンジンとFastAPIの統合方法を開発者に示すことを目的としている。ECサイトや管理画面など、HTMLページを動的に生成する必要があるケースで参考となる実装パターンを提供する。

**画面へのアクセス方法**：`/items/{id}` でアクセス可能。例えば `http://localhost:8000/items/42` でアイテムID "42" の詳細画面が表示される。`{id}`は任意の文字列を受け付ける。

**主要な操作・処理内容**：
1. URLパスからアイテムID（`{id}`）を取得
2. Jinja2テンプレート（item.html）を読み込み
3. テンプレート変数にアイテムIDとリクエストオブジェクトを渡す
4. レンダリング結果をHTMLResponseとして返却
5. 静的ファイル（styles.css）の参照

**画面遷移**：
- 遷移元: アイテム一覧画面（想定）、直接URLアクセス
- 遷移先: 自画面（アイテムIDリンクのクリック）
- 関連画面: 静的ファイル（/static/styles.css）

**権限による表示制御**：このサンプル実装には認証・認可の仕組みは含まれていない。実際のアプリケーションでは、FastAPIの依存関係注入（Depends）を使用して認証チェックを追加することが可能。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 42 | Jinja2Templates | 主機能 | Jinja2テンプレートエンジンを使用してitem.htmlをレンダリング。アイテムIDをテンプレート変数として渡す |
| 43 | StaticFiles | 補助機能 | styles.css等の静的ファイルを配信。テンプレート内でurl_for関数で参照 |
| 4 | Pathパラメータ | 補助機能 | URLパス内の{id}パラメータからアイテムIDを取得 |
| 3 | Request/Response処理 | API連携 | HTMLResponseを使用してレンダリング結果を返却 |

## 画面種別

詳細 / サンプル

## URL/ルーティング

| パス | HTTPメソッド | 説明 |
|------|-------------|------|
| `/items/{id}` | GET | 指定されたIDのアイテム詳細を表示 |
| `/static/{path}` | GET | 静的ファイル（CSS等）を配信 |

## 入出力項目

### リクエスト（画面表示時）

| 項目名 | 型 | 必須 | 説明 |
|--------|-----|------|------|
| id | 文字列（パスパラメータ） | 必須 | 表示するアイテムのID |

### 出力（レスポンス）

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Content-Type | ヘッダー | text/html; charset=utf-8 |
| Body | HTML | レンダリングされたHTMLコンテンツ |

## 表示項目

| 項目名 | 説明 |
|--------|------|
| タイトル | "Item Details" |
| 見出し（h1） | "Item ID: {id}" - アイテムIDへのリンク付き |

## イベント仕様

### 1-画面初期表示

1. ブラウザが`/items/{id}`にGETリクエストを送信
2. FastAPIが`read_item`関数を実行
3. パスパラメータからidを取得
4. `templates.TemplateResponse`でテンプレートをレンダリング
5. HTMLResponseとしてブラウザに返却
6. ブラウザがCSSファイル（/static/styles.css）を読み込み
7. ページが表示される

### 2-アイテムIDリンククリック

1. ユーザーがh1内のリンクをクリック
2. 同じ`/items/{id}`に遷移（自画面リロード）

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| - | - | - | このサンプルはデータベースを使用しない |

※ 実際のアプリケーションでは、idをキーにデータベースからアイテム情報を取得する実装が想定される

## メッセージ仕様

| メッセージ種別 | メッセージ内容 | 表示条件 |
|---------------|---------------|----------|
| - | - | このサンプルにはメッセージ表示機能がない |

※ 実際のアプリケーションでは、アイテムが見つからない場合のエラーメッセージ等を実装する

## 例外処理

| 例外状況 | 処理内容 |
|---------|----------|
| テンプレートファイル未存在 | Jinja2のTemplateNotFoundエラー（500 Internal Server Error） |
| 静的ファイル未存在 | 404 Not Foundを返却 |

※ このサンプルではidの存在チェックは行われない。実際のアプリケーションでは、データベース検索で該当なしの場合に404を返す実装が必要

## 備考

- このコードはFastAPI公式ドキュメントのチュートリアル用サンプル
- `docs_src/templates/`ディレクトリに配置されている
- Python 3.9以降の型ヒント構文を使用（tutorial001_py39.py）
- 実運用では、テンプレートへのデータベース取得結果の受け渡しが必要
- 静的ファイルは`StaticFiles`ミドルウェアで配信

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

テンプレートに渡されるコンテキスト変数とHTMLテンプレートの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | item.html | `docs_src/templates/templates/item.html` | HTMLテンプレートの構造、Jinja2変数の使用方法 |
| 1-2 | styles.css | `docs_src/templates/static/styles.css` | CSSスタイル定義 |

**読解のコツ**: Jinja2テンプレートでは`{{ 変数名 }}`で変数を埋め込み、`{{ url_for('関数名', パラメータ) }}`でURLを動的に生成する。

#### Step 2: エントリーポイントを理解する

FastAPIアプリケーションの設定とエンドポイント定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tutorial001_py39.py | `docs_src/templates/tutorial001_py39.py` | アプリケーション全体の構成 |

**主要処理フロー**:
1. **1-4行目**: 必要なモジュールのインポート（FastAPI、Request、HTMLResponse、StaticFiles、Jinja2Templates）
2. **6行目**: FastAPIアプリケーションインスタンスの作成
3. **8行目**: `/static`パスに静的ファイルディレクトリをマウント
4. **11行目**: Jinja2Templatesインスタンスを作成（templatesディレクトリを指定）
5. **14-17行目**: `/items/{id}`エンドポイントを定義、TemplateResponseでレンダリング

#### Step 3: テンプレートレンダリング処理を理解する

`read_item`関数とテンプレートの連携を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tutorial001_py39.py | `docs_src/templates/tutorial001_py39.py` | read_item関数（14-17行目） |

**主要処理フロー**:
- **14行目**: `@app.get("/items/{id}", response_class=HTMLResponse)` - HTMLレスポンスを返すGETエンドポイント
- **15行目**: `async def read_item(request: Request, id: str):` - Requestオブジェクトとパスパラメータを受け取る
- **16-17行目**: `templates.TemplateResponse(request=request, name="item.html", context={"id": id})` - テンプレートをレンダリングして返却

### プログラム呼び出し階層図

```
FastAPI Application
    │
    ├─ app.mount("/static", ...)  [tutorial001_py39.py:8]
    │      └─ StaticFiles(directory="static")
    │             └─ styles.css の配信
    │
    └─ @app.get("/items/{id}")  [tutorial001_py39.py:14]
           └─ read_item(request, id)  [tutorial001_py39.py:15]
                  └─ templates.TemplateResponse()
                         ├─ request: Requestオブジェクト
                         ├─ name: "item.html"
                         └─ context: {"id": id}
                                │
                                ▼
                         Jinja2テンプレートレンダリング
                                │
                                └─ item.html
                                       ├─ {{ id }}
                                       ├─ {{ url_for('static', path='/styles.css') }}
                                       └─ {{ url_for('read_item', id=id) }}
```

### データフロー図

```
[入力]                    [処理]                           [出力]

HTTPリクエスト            read_item関数                    HTMLResponse
GET /items/42 ──────────▶ パスパラメータ取得 ────────────▶ レンダリング結果
                               │
                               ▼
                         TemplateResponse
                               │
                               ├─ request オブジェクト
                               ├─ テンプレート名: item.html
                               └─ コンテキスト: {id: "42"}
                                        │
                                        ▼
                                  Jinja2レンダリング
                                        │
                                        ▼
ブラウザ                  追加リクエスト                   CSSファイル
ページ表示 ─────────────▶ GET /static/styles.css ────────▶ スタイル適用


[テンプレート内のURL生成]

{{ url_for('static', path='/styles.css') }}  ──▶  /static/styles.css
{{ url_for('read_item', id=id) }}            ──▶  /items/42
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tutorial001_py39.py | `docs_src/templates/tutorial001_py39.py` | ソース | FastAPIアプリケーション、エンドポイント定義 |
| item.html | `docs_src/templates/templates/item.html` | テンプレート | アイテム詳細画面のHTMLテンプレート |
| styles.css | `docs_src/templates/static/styles.css` | 静的ファイル | ページスタイル定義（h1を緑色に） |
| __init__.py | `docs_src/templates/__init__.py` | ソース | パッケージ初期化（空ファイル） |
| __init__.py | `docs_src/templates/templates/__init__.py` | ソース | パッケージ初期化（空ファイル） |
| __init__.py | `docs_src/templates/static/__init__.py` | ソース | パッケージ初期化（空ファイル） |
