# 通知設計書 6-請求書イベント通知

## 概要

本ドキュメントは、FastAPIのOpenAPI Callbacks機能を使用して、請求書作成時に外部APIにコールバック通知を送信する機能の設計書である。支払い完了などのイベントを外部システムに通知するAPI仕様を定義する。

### 本通知の処理概要

FastAPIのOpenAPI Callbacks機能により、請求書関連イベント（支払い完了など）を外部システムに通知するためのコールバックAPI仕様を定義する。`/invoices/` エンドポイントで請求書を作成する際、呼び出し元が指定した`callback_url`に対してイベント通知が送信されることを文書化する。

**業務上の目的・背景**：決済・課金システムでは、請求書の処理状態（作成、支払い完了、失敗など）を外部システムに通知する必要がある。コールバック機能により、APIの利用者が自身のシステムで請求書イベントを受信し、会計処理、在庫管理、顧客通知などの後続処理を実行できるようになる。

**通知の送信タイミング**：請求書作成後、支払い処理が完了した際（概念的なイベント）。docstringに「Send the invoice, collect the money, send the notification (the callback)」と記載されており、課金処理完了後のコールバックを想定している。

**通知の受信者**：請求書作成APIの呼び出し時に`callback_url`パラメータで指定されたエンドポイント。外部開発者が自身のシステムで用意するコールバック受信用エンドポイント。

**通知内容の概要**：請求書イベント情報（イベント説明、支払い済みフラグ）を含むJSONペイロードがPOSTリクエストとして送信される。コールバックURLは動的に構築され、請求書IDが含まれる。

**期待されるアクション**：コールバック受信者は通知を受け取り、以下のような処理を実行する：
- 支払い完了の確認と記録
- 領収書の発行
- 商品・サービスの提供開始
- 顧客への支払い確認メール送信

## 通知種別

API Callback（HTTP POST通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（HTTP POST）※仕様定義のみ |
| 優先度 | 高（リアルタイム通知） |
| リトライ | 実装依存（仕様定義のみ） |

### 送信先決定ロジック

1. 請求書作成API呼び出し時に`callback_url`パラメータを受け取る
2. コールバックURL = `{callback_url}/invoices/{invoice.id}`
3. 動的に構築されたURLにPOSTリクエストを送信

## 通知テンプレート

### Callbackリクエストの場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| URLパターン | `{$callback_url}/invoices/{$request.body.id}` |

### リクエストボディテンプレート

```json
{
  "description": "{description}",
  "paid": {paid}
}
```

### レスポンス

```json
{
  "ok": {ok}
}
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | 添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| callback_url | コールバック先URL | APIリクエストパラメータ | No |
| invoice.id | 請求書ID | Invoiceモデル | Yes |
| description | イベント説明 | InvoiceEventモデル | Yes |
| paid | 支払い済みフラグ | InvoiceEventモデル | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API | POST /invoices/ | callback_urlが指定されている | 請求書作成・支払い処理完了時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| callback_url未指定 | callback_urlパラメータがNoneの場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[POST /invoices/ 受信] --> B[請求書データ検証]
    B --> C[請求書作成]
    C --> D[顧客へ請求書送付]
    D --> E[支払い回収処理]
    E --> F{callback_url指定あり?}
    F -->|No| G[レスポンス返却]
    F -->|Yes| H[コールバックURL構築]
    H --> I[InvoiceEventペイロード作成]
    I --> J[HTTP POST送信]
    J --> K{送信成功?}
    K -->|Yes| G
    K -->|No| L[エラー処理]
    L --> G
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| なし | - | 仕様定義のみ（実装依存） |

### Invoiceモデル

| 参照項目 | 用途 | データ型 |
|---------|------|---------|
| id | 請求書識別子 | str |
| title | 請求書タイトル | str (Optional) |
| customer | 顧客名 | str |
| total | 合計金額 | float |

### InvoiceEventモデル

| 参照項目 | 用途 | データ型 |
|---------|------|---------|
| description | イベント説明 | str |
| paid | 支払い済みフラグ | bool |

### InvoiceEventReceivedモデル（レスポンス）

| 参照項目 | 用途 | データ型 |
|---------|------|---------|
| ok | 受信確認 | bool |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | 仕様定義のみ |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 接続タイムアウト | callback_urlへの接続失敗 | 実装依存 |
| HTTPエラー | 4xx/5xxレスポンス | 実装依存 |
| URL不正 | callback_urlの形式不正 | Pydanticバリデーション（HttpUrl型） |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 実装依存 |
| リトライ間隔 | 実装依存 |
| リトライ対象エラー | 実装依存 |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 実装依存 |
| 1日あたり上限 | 実装依存 |

### 配信時間帯

制限なし（イベント駆動）

## セキュリティ考慮事項

- callback_urlはHttpUrl型でバリデーション（不正なURL形式を拒否）
- HTTPSのみを許可することを推奨
- 署名検証（HMAC等）による改ざん検知を推奨
- 請求書情報（顧客名、金額）は機密情報として取り扱う

## 備考

- 本実装例はOpenAPI Callbacksの仕様定義機能のデモンストレーション
- 実際のコールバック送信ロジックは含まれていない
- OpenAPIドキュメントのcallbacksセクションとして出力される
- APIRouterを使用してコールバック仕様を定義
- `{$callback_url}`と`{$request.body.id}`は動的パラメータ

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

請求書とイベント通知のデータモデルを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | 行7-12: Invoiceモデル定義 |
| 1-2 | tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | 行14-20: InvoiceEvent, InvoiceEventReceivedモデル定義 |

**読解のコツ**: 3つのPydanticモデルの役割を理解する。Invoiceは請求書作成リクエスト、InvoiceEventはコールバックペイロード、InvoiceEventReceivedはコールバックレスポンス。

#### Step 2: コールバックルーター定義を理解する

APIRouterを使用したコールバック仕様の定義方法を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | 行23: invoices_callback_router作成 |
| 2-2 | tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | 行26-30: コールバックエンドポイント定義 |

**主要処理フロー**:
1. **行23**: `invoices_callback_router = APIRouter()` - コールバック用ルーター作成
2. **行26-27**: デコレータ - URLパターンとレスポンスモデル指定
3. **行28-30**: `invoice_notification`関数 - コールバック仕様定義（実装なし）

#### Step 3: メインエンドポイントとの関連を理解する

請求書作成エンドポイントとコールバックの関連付けを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | 行33-51: create_invoiceエンドポイント |

**主要処理フロー**:
- **行33**: `@app.post("/invoices/", callbacks=invoices_callback_router.routes)` - callbacksパラメータでコールバック仕様を関連付け
- **行34**: `callback_url: HttpUrl | None = None` - コールバックURL受け取り
- **行36-51**: docstring - 処理フローの説明

### プログラム呼び出し階層図

```
FastAPI Application
    │
    ├─ @app.post("/invoices/", callbacks=invoices_callback_router.routes)
    │      │
    │      └─ create_invoice(invoice: Invoice, callback_url: HttpUrl | None)
    │             │
    │             └─ [概念的な処理フロー - docstringに記載]
    │                    │
    │                    ├─ Send the invoice to the client
    │                    ├─ Collect the money from the client
    │                    └─ Send a notification back (callback)
    │
    └─ invoices_callback_router
           │
           └─ @router.post("{$callback_url}/invoices/{$request.body.id}")
                  └─ invoice_notification(body: InvoiceEvent)
                         └─ [OpenAPI仕様生成のみ]
```

### データフロー図

```
[概念的なフロー - 実装例では仕様定義のみ]

[リクエスト]          [サーバー処理]              [コールバック]

POST /invoices/
{                     ───▶ 請求書作成
  "id": "INV-001",          │
  "customer": "John",       ▼
  "total": 100.0       請求書送付・回収
}                           │
callback_url=               ▼
"https://ext.com"     コールバック送信 ───▶ POST {callback_url}/invoices/INV-001
                                          {
                                            "description": "Payment successful",
                                            "paid": true
                                          }
                                                │
                                                ▼
                                          {"ok": true}
```

### OpenAPIスキーマ出力例

```yaml
paths:
  /invoices/:
    post:
      summary: Create Invoice
      callbacks:
        invoice_notification:
          '{$callback_url}/invoices/{$request.body.id}':
            post:
              summary: Invoice Notification
              requestBody:
                required: true
                content:
                  application/json:
                    schema:
                      $ref: '#/components/schemas/InvoiceEvent'
              responses:
                '200':
                  content:
                    application/json:
                      schema:
                        $ref: '#/components/schemas/InvoiceEventReceived'
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tutorial001_py310.py | `docs_src/openapi_callbacks/tutorial001_py310.py` | サンプル | Callback仕様定義の例 |
| FastAPI | `fastapi/applications.py` | フレームワーク | callbacksパラメータの提供 |
| APIRouter | `fastapi/routing.py` | フレームワーク | コールバックルーター機能 |
| OpenAPI | - | 仕様 | 生成されるOpenAPIドキュメント |
