# 非機能要件定義書

## 概要

本ドキュメントは、FastAPIフレームワークの非機能要件を定義するものである。FastAPIはPython 3.9以上で動作する高性能なWebフレームワークであり、Starlette（ASGI）とPydantic（データバリデーション）を基盤として構築されている。本定義書では、パフォーマンス、可用性、セキュリティ、拡張性、保守性、運用・監視の各観点から要件を整理する。

## 性能要件

### レスポンスタイム

| 処理種別 | 目標値 | 備考 |
| --- | --- | --- |
| API応答（同期処理） | 50ms以内 | シンプルなCRUD操作、バリデーション含む |
| API応答（非同期処理） | 100ms以内 | 外部API呼び出しやDB操作を含む場合 |
| OpenAPIスキーマ生成 | 初回500ms以内 | `/openapi.json`エンドポイント、キャッシュ後は即座に応答 |
| Swagger UI表示 | 1秒以内 | `/docs`エンドポイント |
| WebSocket接続確立 | 100ms以内 | 初回ハンドシェイク完了まで |

### スループット

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 同時接続数 | 10,000以上 | 非同期I/O（anyio/trio対応）により実現 |
| 秒間リクエスト数 | 高スループット | Uvicorn + uvloopで最適化可能 |
| 並行処理 | CapacityLimiterで制御可能 | `anyio.CapacityLimiter`でスレッドプール管理 |

### パフォーマンス最適化機能

| 機能 | 説明 | 根拠ファイル |
| --- | --- | --- |
| GZip圧縮 | `GZipMiddleware`によるレスポンス圧縮 | `fastapi/middleware/gzip.py` |
| 高速JSONレスポンス | `ORJSONResponse`、`UJSONResponse`による高速シリアライゼーション | `fastapi/responses.py` |
| スレッドプール管理 | `run_in_threadpool`による同期処理の非同期実行 | `fastapi/concurrency.py` |
| バックグラウンドタスク | `BackgroundTasks`によるレスポンス後の非同期処理 | `fastapi/background.py` |

## 可用性要件

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 稼働率 | フレームワーク依存なし | デプロイ環境・インフラに依存 |
| 計画停止時間 | 最小化可能 | Lifespan管理による graceful shutdown対応 |
| 障害復旧時間（RTO） | インフラ依存 | 自動再起動はデプロイ環境に依存 |
| 目標復旧時点（RPO） | アプリケーション依存 | ステートレス設計を推奨 |

### 耐障害性機能

| 機能 | 説明 | 根拠ファイル |
| --- | --- | --- |
| 例外ハンドリング | `HTTPException`、`RequestValidationError`等の統一的な例外処理 | `fastapi/exceptions.py` |
| カスタム例外ハンドラ | `http_exception_handler`による柔軟なエラーレスポンス | `fastapi/exception_handlers.py` |
| バリデーションエラー | `422 Unprocessable Entity`による詳細なエラー情報提供 | `fastapi/exception_handlers.py` |
| WebSocket例外 | `WebSocketException`による適切な接続終了 | `fastapi/exceptions.py` |

## セキュリティ要件

### 認証・認可機能

| 機能 | 説明 | 根拠ファイル |
| --- | --- | --- |
| OAuth2パスワードフロー | `OAuth2PasswordBearer`、`OAuth2PasswordRequestForm` | `fastapi/security/oauth2.py` |
| OAuth2認可コードフロー | `OAuth2AuthorizationCodeBearer` | `fastapi/security/oauth2.py` |
| HTTP Basic認証 | `HTTPBasic`、`HTTPBasicCredentials` | `fastapi/security/http.py` |
| HTTP Bearer認証 | `HTTPBearer`、`HTTPAuthorizationCredentials` | `fastapi/security/http.py` |
| HTTP Digest認証 | `HTTPDigest`（スタブ実装） | `fastapi/security/http.py` |
| APIキー認証（クエリ） | `APIKeyQuery` | `fastapi/security/api_key.py` |
| APIキー認証（ヘッダー） | `APIKeyHeader` | `fastapi/security/api_key.py` |
| APIキー認証（Cookie） | `APIKeyCookie` | `fastapi/security/api_key.py` |
| OpenID Connect | `OpenIdConnect` | `fastapi/security/open_id_connect_url.py` |
| スコープベース認可 | `SecurityScopes`による細粒度アクセス制御 | `fastapi/security/oauth2.py` |

### 通信セキュリティ

| 機能 | 説明 | 根拠ファイル |
| --- | --- | --- |
| HTTPSリダイレクト | `HTTPSRedirectMiddleware`による強制HTTPS | `fastapi/middleware/httpsredirect.py` |
| 信頼済みホスト制限 | `TrustedHostMiddleware`によるホスト検証 | `fastapi/middleware/trustedhost.py` |
| CORS設定 | `CORSMiddleware`によるクロスオリジン制御 | `fastapi/middleware/cors.py` |

### 入力バリデーション

| 機能 | 説明 | 根拠ファイル |
| --- | --- | --- |
| Pydanticバリデーション | 自動型検証・変換（Pydantic v2以上必須） | `pyproject.toml` |
| リクエストバリデーション | `RequestValidationError`による詳細エラー | `fastapi/exceptions.py` |
| パラメータ検証 | Path/Query/Body/Form/Cookie/Headerパラメータの自動検証 | `fastapi/params.py` |

## 拡張性要件

### モジュール構造

| 機能 | 説明 | 根拠 |
| --- | --- | --- |
| APIRouter | ルーティングの分割・モジュール化 | `fastapi/routing.py` |
| 依存性注入 | `Depends`による柔軟な依存関係管理 | `fastapi/params.py` |
| ミドルウェア拡張 | `Middleware`クラスによるカスタムミドルウェア追加 | `fastapi/middleware/__init__.py` |
| カスタムレスポンス | `Response`クラスの継承による拡張 | `fastapi/responses.py` |

### 互換性

| 項目 | 対応バージョン | 備考 |
| --- | --- | --- |
| Python | 3.9, 3.10, 3.11, 3.12, 3.13, 3.14 | `pyproject.toml`で定義 |
| Starlette | 0.40.0 - 0.51.0 | ASGI基盤 |
| Pydantic | 2.7.0以上 | データバリデーション |
| OS | Linux, macOS, Windows | CI/CDでテスト済み |

## 保守性要件

### コード品質

| 項目 | 対応内容 | 根拠ファイル |
| --- | --- | --- |
| 型ヒント | strict mypyによる静的型検査 | `pyproject.toml` |
| リンター | ruffによるコード品質チェック | `pyproject.toml` |
| フォーマッター | ruff format（blackベース） | `.pre-commit-config.yaml` |
| pre-commitフック | 自動コード品質チェック | `.pre-commit-config.yaml` |

### テスト要件

| 項目 | 対応内容 | 根拠ファイル |
| --- | --- | --- |
| テストフレームワーク | pytest | `pyproject.toml` |
| カバレッジ目標 | 100% | `.github/workflows/test.yml` |
| マルチプラットフォームテスト | Ubuntu, macOS, Windows | `.github/workflows/test.yml` |
| マルチバージョンテスト | Python 3.9 - 3.14 | `.github/workflows/test.yml` |
| ベンチマーク | CodSpeed integration | `.github/workflows/test.yml` |

### ドキュメント

| 項目 | 対応内容 | 根拠 |
| --- | --- | --- |
| APIドキュメント自動生成 | OpenAPI 3.1.0スキーマ | `fastapi/openapi/` |
| Swagger UI | `/docs`で自動提供 | `fastapi/openapi/docs.py` |
| ReDoc | `/redoc`で自動提供 | `fastapi/openapi/docs.py` |
| アノテーションドキュメント | `annotated-doc`による詳細説明 | `pyproject.toml` |

## 運用・監視要件

### ロギング

| 項目 | 対応内容 | 根拠ファイル |
| --- | --- | --- |
| ロガー | 標準logging モジュール（"fastapi"ロガー） | `fastapi/logger.py` |
| デバッグモード | `debug=True`でトレースバック表示 | `fastapi/applications.py` |

### ライフサイクル管理

| 機能 | 説明 | 根拠 |
| --- | --- | --- |
| Lifespan | アプリケーション起動・終了時の処理定義 | `fastapi/applications.py` |
| AsyncExitStack | 非同期リソースの適切な解放 | `fastapi/middleware/asyncexitstack.py` |
| Context Manager | `contextmanager_in_threadpool`による同期コンテキスト管理 | `fastapi/concurrency.py` |

### デプロイメント

| 項目 | 推奨構成 | 備考 |
| --- | --- | --- |
| ASGIサーバー | Uvicorn（uvloop付き推奨） | `pyproject.toml`のstandard依存 |
| プロセス管理 | Gunicorn + Uvicorn workers | 本番環境推奨 |
| コンテナ化 | Docker対応 | ステートレス設計 |

## 備考

### 制約事項

1. **Pydantic v1非対応**: Pydantic v1は明示的にサポート対象外（`PydanticV1NotSupportedError`）
2. **HTTPDigest認証**: スタブ実装のみ、完全な実装はアプリケーション側で必要
3. **セッション管理**: SessionMiddleware使用時は`itsdangerous`ライブラリが必要

### 推奨オプション依存

| 機能 | 必要なパッケージ | 用途 |
| --- | --- | --- |
| テストクライアント | httpx | APIテスト |
| テンプレート | jinja2 | HTMLテンプレート |
| フォーム・ファイル | python-multipart | フォームデータ処理 |
| メールバリデーション | email-validator | メールアドレス検証 |
| 高速JSON | orjson, ujson | レスポンス最適化 |
| 設定管理 | pydantic-settings | 環境変数管理 |

### 本ドキュメントのスコープ

本ドキュメントはFastAPIフレームワーク自体の非機能要件を定義しており、個別アプリケーションの非機能要件はアプリケーション設計時に別途定義する必要がある。特にパフォーマンス目標値やスケーラビリティ要件は、デプロイ環境やインフラ構成に大きく依存する。
