# API設計書

## 概要

本ドキュメントは、Fat Free CRM (Customer Relationship Management) アプリケーションのAPI設計書です。Fat Free CRMはRuby on Railsで構築されたオープンソースのCRMシステムで、顧客、見込み客、商談、キャンペーン、タスクなどの管理機能を提供します。

本APIはRESTfulな設計に基づいており、JSON/XML形式でのデータ交換をサポートしています。

## 共通仕様

### ベースURL

```
https://{your-domain}/
```

### 認証方式

**Devise（セッションベース認証）**

- Cookie/セッションベースの認証を使用
- ログインエンドポイント: `POST /users/sign_in`
- ログアウトエンドポイント: `DELETE /users/sign_out`

認証が必要なエンドポイントに未認証でアクセスした場合、HTMLリクエストはログインページにリダイレクトされ、JSON/XMLリクエストは401 Unauthorizedを返します。

### CORS設定

全てのAPIエンドポイントでCORSが有効になっています。

```
Access-Control-Allow-Origin: *
Access-Control-Allow-Methods: POST, GET, PUT, DELETE, OPTIONS
Access-Control-Allow-Headers: Origin, Content-Type, Accept, Authorization, Token
Access-Control-Max-Age: 1728000
```

### 共通ヘッダー

| ヘッダー名 | 必須 | 説明 |
| --- | --- | --- |
| Content-Type | ○ | application/json または application/xml |
| Accept | - | application/json, application/xml, text/csv, application/xls |
| Cookie | ○ | セッション情報（認証後） |

### 対応フォーマット

| フォーマット | 説明 |
| --- | --- |
| HTML | Webブラウザ向け |
| JSON | API向け |
| XML | API向け |
| CSV | データエクスポート（indexアクションのみ） |
| XLS | データエクスポート（indexアクションのみ） |
| RSS/ATOM | フィード（一部エンドポイントのみ） |

### 共通エラーレスポンス

| ステータスコード | 説明 |
| --- | --- |
| 400 | Bad Request - リクエストパラメータ不正 |
| 401 | Unauthorized - 認証エラー |
| 403 | Forbidden - 権限エラー（CanCan認可） |
| 404 | Not Found - リソース未検出 |
| 422 | Unprocessable Entity - バリデーションエラー |
| 500 | Internal Server Error - サーバーエラー |

### 共通クエリパラメータ

| パラメータ名 | 型 | 説明 |
| --- | --- | --- |
| page | integer | ページ番号（ページネーション） |
| per_page | integer | 1ページあたりの件数（最大200） |
| query | string | テキスト検索クエリ |
| q | hash | Ransack検索パラメータ（高度な検索） |

## API一覧

### 認証系

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Sessions | /users/sign_in | GET | ログインページ表示 |
| Sessions | /users/sign_in | POST | ログイン |
| Sessions | /users/sign_out | DELETE | ログアウト |
| Registrations | /users/sign_up | GET | 新規登録ページ表示 |
| Registrations | /users | POST | 新規登録 |
| Passwords | /users/password/new | GET | パスワードリセットページ |
| Passwords | /users/password | POST | パスワードリセットメール送信 |
| Confirmations | /users/confirmation | GET | メール確認 |

### メインエンティティ

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Accounts | /accounts | GET | 取引先一覧取得 |
| Accounts | /accounts/:id | GET | 取引先詳細取得 |
| Accounts | /accounts | POST | 取引先作成 |
| Accounts | /accounts/:id | PUT/PATCH | 取引先更新 |
| Accounts | /accounts/:id | DELETE | 取引先削除 |
| Contacts | /contacts | GET | 連絡先一覧取得 |
| Contacts | /contacts/:id | GET | 連絡先詳細取得 |
| Contacts | /contacts | POST | 連絡先作成 |
| Contacts | /contacts/:id | PUT/PATCH | 連絡先更新 |
| Contacts | /contacts/:id | DELETE | 連絡先削除 |
| Leads | /leads | GET | 見込み客一覧取得 |
| Leads | /leads/:id | GET | 見込み客詳細取得 |
| Leads | /leads | POST | 見込み客作成 |
| Leads | /leads/:id | PUT/PATCH | 見込み客更新 |
| Leads | /leads/:id | DELETE | 見込み客削除 |
| Leads | /leads/:id/convert | GET | 見込み客変換画面 |
| Leads | /leads/:id/promote | PUT/PATCH | 見込み客を連絡先に昇格 |
| Leads | /leads/:id/reject | PUT | 見込み客を却下 |
| Opportunities | /opportunities | GET | 商談一覧取得 |
| Opportunities | /opportunities/:id | GET | 商談詳細取得 |
| Opportunities | /opportunities | POST | 商談作成 |
| Opportunities | /opportunities/:id | PUT/PATCH | 商談更新 |
| Opportunities | /opportunities/:id | DELETE | 商談削除 |
| Campaigns | /campaigns | GET | キャンペーン一覧取得 |
| Campaigns | /campaigns/:id | GET | キャンペーン詳細取得 |
| Campaigns | /campaigns | POST | キャンペーン作成 |
| Campaigns | /campaigns/:id | PUT/PATCH | キャンペーン更新 |
| Campaigns | /campaigns/:id | DELETE | キャンペーン削除 |
| Tasks | /tasks | GET | タスク一覧取得 |
| Tasks | /tasks/:id | GET | タスク詳細取得 |
| Tasks | /tasks | POST | タスク作成 |
| Tasks | /tasks/:id | PUT/PATCH | タスク更新 |
| Tasks | /tasks/:id | DELETE | タスク削除 |
| Tasks | /tasks/:id/complete | PUT | タスク完了 |
| Tasks | /tasks/:id/uncomplete | PUT | タスク未完了に戻す |

### ユーザー管理

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Users | /users | GET | ユーザー一覧取得 |
| Users | /users/:id | GET | ユーザー詳細取得 |
| Users | /users/:id | PUT/PATCH | ユーザー更新 |
| Users | /users/:id/avatar | GET | アバター設定画面 |
| Users | /users/:id/upload_avatar | PUT/PATCH | アバターアップロード |
| Users | /users/:id/password | GET | パスワード変更画面 |
| Users | /users/:id/change_password | PATCH | パスワード変更 |
| Users | /users/opportunities_overview | GET | 商談概要 |
| Users | /users/auto_complete | GET/POST | ユーザーオートコンプリート |

### 管理者機能

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Admin Users | /admin/users | GET | ユーザー一覧（管理） |
| Admin Users | /admin/users/:id | GET | ユーザー詳細（管理） |
| Admin Users | /admin/users | POST | ユーザー作成（管理） |
| Admin Users | /admin/users/:id | PUT/PATCH | ユーザー更新（管理） |
| Admin Users | /admin/users/:id | DELETE | ユーザー削除（管理） |
| Admin Users | /admin/users/:id/suspend | PUT | ユーザー停止 |
| Admin Users | /admin/users/:id/reactivate | PUT | ユーザー再有効化 |
| Admin Groups | /admin/groups | GET | グループ一覧 |
| Admin Groups | /admin/groups/:id | GET | グループ詳細 |
| Admin Groups | /admin/groups | POST | グループ作成 |
| Admin Groups | /admin/groups/:id | PUT/PATCH | グループ更新 |
| Admin Groups | /admin/groups/:id | DELETE | グループ削除 |
| Admin Settings | /admin/settings | GET | 設定一覧 |
| Admin Settings | /admin/settings | PUT | 設定更新 |
| Admin Tags | /admin/tags | GET | タグ一覧 |
| Admin Tags | /admin/tags | POST | タグ作成 |
| Admin Tags | /admin/tags/:id | PUT/PATCH | タグ更新 |
| Admin Tags | /admin/tags/:id | DELETE | タグ削除 |
| Admin Fields | /admin/fields | GET | カスタムフィールド一覧 |
| Admin Fields | /admin/fields | POST | カスタムフィールド作成 |
| Admin Fields | /admin/fields/:id | PUT/PATCH | カスタムフィールド更新 |
| Admin Fields | /admin/fields/:id | DELETE | カスタムフィールド削除 |
| Admin Field Groups | /admin/field_groups | POST | フィールドグループ作成 |
| Admin Field Groups | /admin/field_groups/:id | PUT/PATCH | フィールドグループ更新 |
| Admin Field Groups | /admin/field_groups/:id | DELETE | フィールドグループ削除 |

### その他

| カテゴリ | エンドポイント | メソッド | 説明 |
| --- | --- | --- | --- |
| Home | / | GET | ダッシュボード |
| Home | /home/options | GET | オプション設定 |
| Home | /home/toggle | GET | UI切替 |
| Home | /home/timeline | GET/PUT/POST | タイムライン操作 |
| Home | /home/timezone | GET/PUT/POST | タイムゾーン設定 |
| Home | /home/redraw | POST | 再描画 |
| Comments | /comments | GET | コメント一覧 |
| Comments | /comments | POST | コメント作成 |
| Comments | /comments/:id | PUT/PATCH | コメント更新 |
| Comments | /comments/:id | DELETE | コメント削除 |
| Lists | /lists | POST | リスト作成 |
| Lists | /lists/:id | DELETE | リスト削除 |
| Emails | /emails/:id | DELETE | メール削除 |

---

## 各APIエンドポイント定義

### Accounts（取引先）

#### 1. 取引先一覧取得

取引先の一覧を取得します。ページネーション、検索、フィルタリングに対応しています。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /accounts` |
| 認証 | 必要 |
| 権限 | 閲覧可能な取引先のみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| page | integer | - | 1 | ページ番号 |
| per_page | integer | - | 20 | 1ページあたりの件数（最大200） |
| query | string | - | - | テキスト検索（名前、メールアドレス） |
| q | hash | - | - | Ransack検索パラメータ |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "user_id": 1,
    "assigned_to": null,
    "name": "株式会社サンプル",
    "access": "Public",
    "website": "https://example.com",
    "toll_free_phone": null,
    "phone": "03-1234-5678",
    "fax": null,
    "email": "info@example.com",
    "background_info": "重要な取引先",
    "rating": 4,
    "category": "customer",
    "created_at": "2024-01-01T00:00:00.000Z",
    "updated_at": "2024-01-15T10:30:00.000Z",
    "tag_list": ["VIP", "製造業"]
  }
]
```

---

#### 2. 取引先詳細取得

指定した取引先の詳細情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /accounts/:id` |
| 認証 | 必要 |
| 権限 | 対象取引先への閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 取引先ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "user_id": 1,
  "assigned_to": 2,
  "name": "株式会社サンプル",
  "access": "Public",
  "website": "https://example.com",
  "toll_free_phone": "0120-123-456",
  "phone": "03-1234-5678",
  "fax": "03-1234-5679",
  "email": "info@example.com",
  "background_info": "重要な取引先です。毎月の定期ミーティングあり。",
  "rating": 4,
  "category": "customer",
  "billing_address": {
    "street1": "東京都渋谷区",
    "street2": "1-2-3 サンプルビル5F",
    "city": "渋谷区",
    "state": "東京都",
    "zipcode": "150-0001",
    "country": "日本"
  },
  "shipping_address": null,
  "contacts": [
    {"id": 1, "first_name": "太郎", "last_name": "山田"}
  ],
  "opportunities": [
    {"id": 1, "name": "新規システム導入", "stage": "proposal"}
  ],
  "created_at": "2024-01-01T00:00:00.000Z",
  "updated_at": "2024-01-15T10:30:00.000Z",
  "tag_list": ["VIP", "製造業"]
}
```

**レスポンス（エラー時）**

ステータスコード: `404 Not Found`

```json
"取引先は利用できません"
```

---

#### 3. 取引先作成

新しい取引先を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /accounts` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "account": {
    "name": "株式会社新規取引先",
    "access": "Public",
    "website": "https://newclient.com",
    "phone": "03-9999-8888",
    "email": "contact@newclient.com",
    "category": "customer",
    "rating": 3,
    "assigned_to": 2,
    "background_info": "紹介経由の新規取引先",
    "billing_address_attributes": {
      "street1": "東京都港区",
      "city": "港区",
      "state": "東京都",
      "zipcode": "105-0001",
      "country": "日本",
      "address_type": "Billing"
    },
    "tag_list": "新規, IT企業"
  },
  "comment_body": "初回コンタクト完了"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| name | string | ○ | max:64、ユニーク（設定による） | 取引先名 |
| access | string | - | Public/Private/Shared | アクセス権限 |
| website | string | - | max:64 | Webサイト |
| phone | string | - | max:32 | 電話番号 |
| fax | string | - | max:32 | FAX番号 |
| toll_free_phone | string | - | max:32 | フリーダイヤル |
| email | string | - | max:64 | メールアドレス |
| category | string | - | 設定値のいずれか | カテゴリ |
| rating | integer | - | 0-5 | 評価 |
| assigned_to | integer | - | - | 担当者ID |
| background_info | string | - | max:255 | 背景情報 |

**レスポンス（成功時）**

ステータスコード: `201 Created`

```json
{
  "id": 100,
  "name": "株式会社新規取引先",
  "access": "Public",
  "created_at": "2024-01-20T09:00:00.000Z",
  "updated_at": "2024-01-20T09:00:00.000Z"
}
```

**レスポンス（エラー時）**

ステータスコード: `422 Unprocessable Entity`

```json
{
  "name": ["を入力してください"]
}
```

---

#### 4. 取引先更新

既存の取引先情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /accounts/:id` |
| 認証 | 必要 |
| 権限 | 対象取引先への編集権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 取引先ID |

**リクエストボディ**

```json
{
  "account": {
    "name": "株式会社サンプル（更新）",
    "rating": 5,
    "background_info": "契約更新済み"
  }
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "name": "株式会社サンプル（更新）",
  "rating": 5,
  "updated_at": "2024-01-20T10:00:00.000Z"
}
```

---

#### 5. 取引先削除

取引先を削除します（論理削除）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /accounts/:id` |
| 認証 | 必要 |
| 権限 | 対象取引先への削除権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 取引先ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "deleted_at": "2024-01-20T11:00:00.000Z"
}
```

---

#### 6. 取引先への関連付け

取引先に連絡先や商談を関連付けます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /accounts/:id/attach` |
| 認証 | 必要 |
| 権限 | 対象取引先への編集権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 取引先ID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| assets | string | ○ | 関連付けるリソースタイプ（contacts/opportunities） |
| asset_id | integer | ○ | 関連付けるリソースID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

---

#### 7. 取引先からの関連解除

取引先から連絡先や商談の関連を解除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /accounts/:id/discard` |
| 認証 | 必要 |
| 権限 | 対象取引先への編集権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 取引先ID |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| attachment | string | ○ | 解除するリソースタイプ |
| attachment_id | integer | ○ | 解除するリソースID |

---

#### 8. 取引先の購読/購読解除

取引先の更新通知を購読または解除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /accounts/:id/subscribe` |
| 認証 | 必要 |
| 権限 | 対象取引先への閲覧権限 |

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /accounts/:id/unsubscribe` |
| 認証 | 必要 |
| 権限 | 対象取引先への閲覧権限 |

---

### Contacts（連絡先）

#### 1. 連絡先一覧取得

連絡先の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /contacts` |
| 認証 | 必要 |
| 権限 | 閲覧可能な連絡先のみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| page | integer | - | 1 | ページ番号 |
| per_page | integer | - | 20 | 1ページあたりの件数 |
| query | string | - | - | テキスト検索（名前、メール、電話） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "user_id": 1,
    "lead_id": null,
    "assigned_to": 2,
    "reports_to": null,
    "first_name": "太郎",
    "last_name": "山田",
    "access": "Public",
    "title": "部長",
    "department": "営業部",
    "source": "web",
    "email": "yamada@example.com",
    "alt_email": null,
    "phone": "03-1111-2222",
    "mobile": "090-1111-2222",
    "fax": null,
    "blog": null,
    "linkedin": "https://linkedin.com/in/yamada",
    "facebook": null,
    "twitter": null,
    "born_on": "1980-05-15",
    "do_not_call": false,
    "background_info": "主要連絡先",
    "account": {
      "id": 1,
      "name": "株式会社サンプル"
    },
    "created_at": "2024-01-01T00:00:00.000Z",
    "updated_at": "2024-01-10T08:00:00.000Z"
  }
]
```

---

#### 2. 連絡先詳細取得

指定した連絡先の詳細情報を取得します。VCF形式でのエクスポートにも対応しています。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /contacts/:id` |
| 認証 | 必要 |
| 権限 | 対象連絡先への閲覧権限 |

**パスパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| id | integer | ○ | 連絡先ID |

**レスポンス（成功時）**

ステータスコード: `200 OK`

VCF形式（`Accept: text/x-vcard`）でリクエストした場合、vCard形式でダウンロードできます。

---

#### 3. 連絡先作成

新しい連絡先を作成します。取引先との関連付けも同時に行えます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /contacts` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "contact": {
    "first_name": "花子",
    "last_name": "鈴木",
    "access": "Public",
    "title": "課長",
    "department": "開発部",
    "email": "suzuki@example.com",
    "phone": "03-3333-4444",
    "mobile": "090-3333-4444",
    "assigned_to": 2,
    "business_address_attributes": {
      "street1": "東京都新宿区",
      "city": "新宿区",
      "state": "東京都",
      "zipcode": "160-0001",
      "country": "日本",
      "address_type": "Business"
    }
  },
  "account": {
    "id": 1
  },
  "opportunity": 5,
  "comment_body": "初回面談実施"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| first_name | string | 設定による | max:64 | 名 |
| last_name | string | 設定による | max:64 | 姓 |
| access | string | - | Public/Private/Shared | アクセス権限 |
| title | string | - | max:64 | 役職 |
| department | string | - | max:64 | 部署 |
| email | string | - | max:254 | メールアドレス |
| alt_email | string | - | max:254 | 副メールアドレス |
| phone | string | - | max:32 | 電話番号 |
| mobile | string | - | max:32 | 携帯電話 |
| fax | string | - | max:32 | FAX |
| blog | string | - | max:128 | ブログURL |
| linkedin | string | - | max:128 | LinkedIn URL |
| facebook | string | - | max:128 | Facebook URL |
| twitter | string | - | max:128 | Twitter URL |
| born_on | date | - | - | 生年月日 |
| do_not_call | boolean | - | - | 電話不可フラグ |

---

#### 4. 連絡先更新

既存の連絡先情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /contacts/:id` |
| 認証 | 必要 |
| 権限 | 対象連絡先への編集権限 |

---

#### 5. 連絡先削除

連絡先を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /contacts/:id` |
| 認証 | 必要 |
| 権限 | 対象連絡先への削除権限 |

---

### Leads（見込み客）

#### 1. 見込み客一覧取得

見込み客の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /leads` |
| 認証 | 必要 |
| 権限 | 閲覧可能な見込み客のみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| page | integer | - | 1 | ページ番号 |
| query | string | - | - | テキスト検索（名前、会社名、メール） |
| status | string | - | - | ステータスフィルター |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "user_id": 1,
    "campaign_id": 1,
    "assigned_to": null,
    "first_name": "次郎",
    "last_name": "田中",
    "access": "Public",
    "title": "代表取締役",
    "company": "株式会社新規見込み",
    "source": "advertisement",
    "status": "new",
    "referred_by": null,
    "email": "tanaka@prospect.com",
    "phone": "03-5555-6666",
    "mobile": null,
    "rating": 3,
    "do_not_call": false,
    "background_info": "展示会で名刺交換",
    "created_at": "2024-01-05T00:00:00.000Z",
    "updated_at": "2024-01-05T00:00:00.000Z"
  }
]
```

---

#### 2. 見込み客詳細取得

指定した見込み客の詳細情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /leads/:id` |
| 認証 | 必要 |
| 権限 | 対象見込み客への閲覧権限 |

---

#### 3. 見込み客作成

新しい見込み客を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /leads` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "lead": {
    "first_name": "三郎",
    "last_name": "佐藤",
    "access": "Public",
    "title": "マネージャー",
    "company": "株式会社興味あり",
    "source": "web",
    "status": "new",
    "email": "sato@interested.com",
    "phone": "03-7777-8888",
    "rating": 2
  },
  "campaign": 1,
  "comment_body": "Webフォームからの問い合わせ"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| first_name | string | 設定による | - | 名 |
| last_name | string | 設定による | - | 姓 |
| company | string | - | max:64 | 会社名 |
| title | string | - | max:64 | 役職 |
| source | string | - | 設定値のいずれか | リードソース |
| status | string | - | 設定値のいずれか | ステータス |
| referred_by | string | - | max:64 | 紹介元 |
| email | string | - | max:64 | メールアドレス |
| phone | string | - | max:32 | 電話番号 |
| mobile | string | - | max:32 | 携帯電話 |
| rating | integer | - | 0-5 | 評価 |
| do_not_call | boolean | - | - | 電話不可フラグ |

---

#### 4. 見込み客更新

既存の見込み客情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /leads/:id` |
| 認証 | 必要 |
| 権限 | 対象見込み客への編集権限 |

---

#### 5. 見込み客削除

見込み客を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /leads/:id` |
| 認証 | 必要 |
| 権限 | 対象見込み客への削除権限 |

---

#### 6. 見込み客変換画面

見込み客を連絡先に変換するための画面データを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /leads/:id/convert` |
| 認証 | 必要 |
| 権限 | 対象見込み客への編集権限 |

---

#### 7. 見込み客昇格（Promote）

見込み客を連絡先に昇格させます。同時に取引先と商談を作成できます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /leads/:id/promote` |
| 認証 | 必要 |
| 権限 | 対象見込み客への編集権限 |

**リクエストボディ**

```json
{
  "account": {
    "name": "株式会社新規取引先",
    "access": "Public"
  },
  "opportunity": {
    "name": "新規案件",
    "stage": "prospecting",
    "amount": 1000000,
    "probability": 20
  },
  "access": "Public"
}
```

**レスポンス（成功時）**

ステータスコード: `200 OK`

見込み客のステータスが「converted」に変更され、連絡先、取引先、商談が作成されます。

**レスポンス（エラー時）**

ステータスコード: `422 Unprocessable Entity`

```json
[
  {"name": ["を入力してください"]}
]
```

---

#### 8. 見込み客却下（Reject）

見込み客を却下します。ステータスが「rejected」に変更されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /leads/:id/reject` |
| 認証 | 必要 |
| 権限 | 対象見込み客への編集権限 |

---

### Opportunities（商談）

#### 1. 商談一覧取得

商談の一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /opportunities` |
| 認証 | 必要 |
| 権限 | 閲覧可能な商談のみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| page | integer | - | 1 | ページ番号 |
| per_page | integer | - | 20 | 1ページあたりの件数 |
| query | string | - | - | テキスト検索（名前、ID） |
| stage | string | - | - | ステージフィルター |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "user_id": 1,
    "campaign_id": 1,
    "assigned_to": 2,
    "name": "新規システム導入案件",
    "access": "Public",
    "source": "web",
    "stage": "proposal",
    "probability": 50,
    "amount": 5000000.00,
    "discount": 0.00,
    "closes_on": "2024-03-31",
    "background_info": "予算承認済み",
    "account": {
      "id": 1,
      "name": "株式会社サンプル"
    },
    "created_at": "2024-01-01T00:00:00.000Z",
    "updated_at": "2024-01-15T00:00:00.000Z"
  }
]
```

---

#### 2. 商談詳細取得

指定した商談の詳細情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /opportunities/:id` |
| 認証 | 必要 |
| 権限 | 対象商談への閲覧権限 |

---

#### 3. 商談作成

新しい商談を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /opportunities` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "opportunity": {
    "name": "クラウド移行案件",
    "access": "Public",
    "stage": "prospecting",
    "probability": 20,
    "amount": 3000000,
    "discount": 100000,
    "closes_on": "2024-06-30",
    "source": "referral",
    "assigned_to": 2,
    "background_info": "競合他社からの乗り換え検討中"
  },
  "account": {
    "id": 1
  },
  "contact": 5,
  "campaign": 2,
  "comment_body": "初回提案完了"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| name | string | ○ | max:64 | 商談名 |
| access | string | - | Public/Private/Shared | アクセス権限 |
| stage | string | - | 設定値のいずれか | ステージ |
| probability | integer | - | 0-100 | 確度（%） |
| amount | decimal | - | - | 金額 |
| discount | decimal | - | - | 割引額 |
| closes_on | date | - | - | クローズ予定日 |
| source | string | - | 設定値のいずれか | ソース |

---

#### 4. 商談更新

既存の商談情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /opportunities/:id` |
| 認証 | 必要 |
| 権限 | 対象商談への編集権限 |

---

#### 5. 商談削除

商談を削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /opportunities/:id` |
| 認証 | 必要 |
| 権限 | 対象商談への削除権限 |

---

### Campaigns（キャンペーン）

#### 1. キャンペーン一覧取得

キャンペーンの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /campaigns` |
| 認証 | 必要 |
| 権限 | 閲覧可能なキャンペーンのみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| page | integer | - | 1 | ページ番号 |
| per_page | integer | - | 20 | 1ページあたりの件数 |
| query | string | - | - | テキスト検索（名前） |
| status | string | - | - | ステータスフィルター |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
[
  {
    "id": 1,
    "user_id": 1,
    "assigned_to": null,
    "name": "2024年春キャンペーン",
    "access": "Public",
    "status": "started",
    "budget": 1000000.00,
    "target_leads": 100,
    "target_conversion": 0.1,
    "target_revenue": 10000000.00,
    "leads_count": 45,
    "opportunities_count": 5,
    "revenue": 2500000.00,
    "starts_on": "2024-03-01",
    "ends_on": "2024-05-31",
    "objectives": "新規顧客獲得",
    "background_info": "春の販促キャンペーン",
    "created_at": "2024-02-01T00:00:00.000Z",
    "updated_at": "2024-03-15T00:00:00.000Z"
  }
]
```

---

#### 2. キャンペーン詳細取得

指定したキャンペーンの詳細情報を取得します。RSS/ATOM形式でのフィード出力も可能です。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /campaigns/:id` |
| 認証 | 必要 |
| 権限 | 対象キャンペーンへの閲覧権限 |

---

#### 3. キャンペーン作成

新しいキャンペーンを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /campaigns` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "campaign": {
    "name": "夏季セールキャンペーン",
    "access": "Public",
    "status": "planned",
    "budget": 500000,
    "target_leads": 50,
    "target_conversion": 0.15,
    "target_revenue": 5000000,
    "starts_on": "2024-07-01",
    "ends_on": "2024-08-31",
    "objectives": "夏季限定商品の販促",
    "assigned_to": 2
  },
  "comment_body": "企画書承認済み"
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| name | string | ○ | max:64、ユニーク（ユーザー単位） | キャンペーン名 |
| access | string | - | Public/Private/Shared | アクセス権限 |
| status | string | - | 設定値のいずれか | ステータス |
| budget | decimal | - | - | 予算 |
| target_leads | integer | - | - | 目標見込み客数 |
| target_conversion | float | - | - | 目標コンバージョン率 |
| target_revenue | decimal | - | - | 目標売上 |
| starts_on | date | - | - | 開始日 |
| ends_on | date | - | starts_on以降 | 終了日 |
| objectives | text | - | - | 目標・目的 |

---

#### 4. キャンペーン更新

既存のキャンペーン情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /campaigns/:id` |
| 認証 | 必要 |
| 権限 | 対象キャンペーンへの編集権限 |

---

#### 5. キャンペーン削除

キャンペーンを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /campaigns/:id` |
| 認証 | 必要 |
| 権限 | 対象キャンペーンへの削除権限 |

---

### Tasks（タスク）

#### 1. タスク一覧取得

タスクの一覧を取得します。タスクはビュー（pending/assigned/completed）によってグループ化されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /tasks` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスクのみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| view | string | - | pending | ビュータイプ（pending/assigned/completed） |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "due_asap": [
    {
      "id": 1,
      "user_id": 1,
      "assigned_to": null,
      "completed_by": null,
      "name": "緊急：見積書作成",
      "asset_id": 1,
      "asset_type": "Opportunity",
      "priority": "high",
      "category": "call",
      "bucket": "due_asap",
      "due_at": null,
      "completed_at": null,
      "background_info": "本日中に送付必要"
    }
  ],
  "due_today": [],
  "due_tomorrow": [],
  "due_this_week": [],
  "due_next_week": [],
  "due_later": []
}
```

---

#### 2. タスク詳細取得

指定したタスクの詳細情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /tasks/:id` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスク |

---

#### 3. タスク作成

新しいタスクを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /tasks` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

```json
{
  "task": {
    "name": "フォローアップコール",
    "asset_id": 1,
    "asset_type": "Contact",
    "priority": "normal",
    "category": "call",
    "bucket": "due_tomorrow",
    "assigned_to": 2,
    "background_info": "提案後のフォロー"
  }
}
```

| フィールド名 | 型 | 必須 | バリデーション | 説明 |
| --- | --- | --- | --- | --- |
| name | string | ○ | max:255 | タスク名 |
| asset_id | integer | - | - | 関連エンティティID |
| asset_type | string | - | - | 関連エンティティタイプ |
| priority | string | - | - | 優先度 |
| category | string | - | 設定値のいずれか | カテゴリ |
| bucket | string | - | 設定値のいずれか | 期限バケット |
| due_at | datetime | - | - | 期限 |
| assigned_to | integer | - | - | 担当者ID |
| calendar | string | - | bucket=specific_timeの場合必須 | 特定日時 |

---

#### 4. タスク更新

既存のタスク情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /tasks/:id` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスク |

---

#### 5. タスク削除

タスクを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /tasks/:id` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスク |

---

#### 6. タスク完了

タスクを完了状態にします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /tasks/:id/complete` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスク |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "completed_at": "2024-01-20T14:00:00.000Z",
  "completed_by": 1
}
```

---

#### 7. タスク未完了に戻す

完了したタスクを未完了状態に戻します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /tasks/:id/uncomplete` |
| 認証 | 必要 |
| 権限 | 自分に関連するタスク |

---

### Comments（コメント）

#### 1. コメント一覧取得

指定したエンティティに紐づくコメント一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /comments` |
| 認証 | 必要 |
| 権限 | 対象エンティティへの閲覧権限 |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| {entity}_id | integer | ○ | エンティティID（account_id, contact_id等） |

---

#### 2. コメント作成

エンティティにコメントを追加します。コメント内で `@username` 形式でユーザーをメンションすると、そのユーザーに通知が送られます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /comments` |
| 認証 | 必要 |
| 権限 | 対象エンティティへの閲覧権限 |

**リクエストボディ**

```json
{
  "comment": {
    "commentable_type": "Account",
    "commentable_id": 1,
    "comment": "重要な更新があります。@yamada 確認をお願いします。",
    "private": false
  }
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| commentable_type | string | ○ | コメント対象エンティティタイプ |
| commentable_id | integer | ○ | コメント対象エンティティID |
| comment | text | ○ | コメント本文 |
| private | boolean | - | プライベートフラグ |
| title | string | - | タイトル |

---

#### 3. コメント更新

既存のコメントを更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /comments/:id` |
| 認証 | 必要 |
| 権限 | 対象エンティティへの閲覧権限 |

---

#### 4. コメント削除

コメントを削除します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `DELETE /comments/:id` |
| 認証 | 必要 |
| 権限 | 対象エンティティへの閲覧権限 |

---

### Users（ユーザー）

#### 1. ユーザー一覧取得

ユーザーの一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /users` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

---

#### 2. ユーザー詳細取得

指定したユーザーの詳細情報を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /users/:id` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**レスポンス（成功時）**

ステータスコード: `200 OK`

```json
{
  "id": 1,
  "username": "admin",
  "email": "admin@example.com",
  "first_name": "管理者",
  "last_name": "ユーザー",
  "title": "システム管理者",
  "company": "株式会社サンプル",
  "alt_email": null,
  "phone": "03-1111-0000",
  "mobile": "090-1111-0000",
  "admin": true,
  "created_at": "2024-01-01T00:00:00.000Z",
  "last_sign_in_at": "2024-01-20T09:00:00.000Z"
}
```

---

#### 3. ユーザー更新

自分のユーザー情報を更新します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /users/:id` |
| 認証 | 必要 |
| 権限 | 自分自身または管理者 |

**リクエストボディ**

```json
{
  "user": {
    "first_name": "更新後名",
    "last_name": "更新後姓",
    "email": "updated@example.com",
    "phone": "03-2222-3333"
  }
}
```

---

#### 4. パスワード変更

自分のパスワードを変更します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PATCH /users/:id/change_password` |
| 認証 | 必要 |
| 権限 | 自分自身 |

**リクエストボディ**

```json
{
  "current_password": "current_password",
  "user": {
    "password": "new_password",
    "password_confirmation": "new_password"
  }
}
```

---

#### 5. アバターアップロード

ユーザーアバター画像をアップロードします。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /users/:id/upload_avatar` |
| 認証 | 必要 |
| 権限 | 自分自身 |

**リクエスト**

multipart/form-dataでアバター画像を送信します。

```
avatar[image]: (ファイル)
```

Gravatarを使用する場合:

```
gravatar: true
```

---

### Admin（管理者機能）

#### Admin Users（ユーザー管理）

管理者によるユーザー管理機能です。

**1. ユーザー一覧取得**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /admin/users` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**2. ユーザー作成**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /admin/users` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**リクエストボディ**

```json
{
  "user": {
    "username": "newuser",
    "email": "newuser@example.com",
    "first_name": "新規",
    "last_name": "ユーザー",
    "password": "password123",
    "password_confirmation": "password123",
    "admin": false,
    "group_ids": [1, 2]
  }
}
```

**3. ユーザー停止**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /admin/users/:id/suspend` |
| 認証 | 必要 |
| 権限 | 管理者のみ（自分自身以外） |

**4. ユーザー再有効化**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /admin/users/:id/reactivate` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

---

#### Admin Groups（グループ管理）

ユーザーグループの管理機能です。

**1. グループ一覧取得**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /admin/groups` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**2. グループ作成**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /admin/groups` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**リクエストボディ**

```json
{
  "group": {
    "name": "営業チーム",
    "user_ids": [1, 2, 3]
  }
}
```

---

#### Admin Settings（設定管理）

システム設定の管理機能です。

**1. 設定一覧取得**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /admin/settings` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**2. 設定更新**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `PUT /admin/settings` |
| 認証 | 必要 |
| 権限 | 管理者のみ |

**リクエストボディ**

```json
{
  "settings": {
    "host": "crm.example.com",
    "base_url": "https://crm.example.com",
    "locale": "ja",
    "per_user_locale": "1",
    "default_access": "Public",
    "user_signup": "allowed",
    "require_first_names": "1",
    "require_last_names": "1",
    "account_category": "customer\npartner\nvendor",
    "lead_status": "new\ncontacted\nqualified",
    "opportunity_stage": "prospecting\nproposal\nnegotiation\nwon\nlost"
  }
}
```

---

### 共通エンドポイント機能

#### オートコンプリート

各エンティティでオートコンプリート検索が利用可能です。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET/POST /{entity}/auto_complete` |
| 認証 | 必要 |
| 権限 | 閲覧可能なエンティティのみ |

**クエリパラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| term | string | ○ | 検索文字列 |
| related | string | - | 除外する関連エンティティ（形式: "entity/id"） |

**レスポンス（JSON形式）**

```json
{
  "results": [
    {"id": 1, "text": "株式会社サンプル"},
    {"id": 2, "text": "株式会社テスト"}
  ]
}
```

---

#### フィルター

一覧表示でのフィルタリング機能です。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `POST /{entity}/filter` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**リクエストボディ**

エンティティによって異なりますが、主にステータスやカテゴリでフィルタリングします。

```
category=customer  # Accounts
status=new         # Leads, Campaigns
stage=proposal     # Opportunities
```

---

#### 再描画（Redraw）

表示設定を変更して一覧を再描画します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /{entity}/redraw` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**クエリパラメータ**

| パラメータ名 | 型 | 説明 |
| --- | --- | --- |
| per_page | integer | 1ページあたりの表示件数 |
| sort_by | string | ソート順 |

---

#### 高度な検索

Ransackを使用した高度な検索機能です。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /{entity}/advanced_search` |
| 認証 | 必要 |
| 権限 | ログインユーザー |

**クエリパラメータ**

Ransack形式の検索パラメータを使用します。

```
q[name_cont]=サンプル
q[created_at_gteq]=2024-01-01
q[s]=name+asc
```

---

#### バージョン履歴

エンティティの変更履歴を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| エンドポイント | `GET /{entity}/versions` |
| 認証 | 必要 |
| 権限 | 対象エンティティへの閲覧権限 |

---

## 備考

### アクセス権限について

Fat Free CRMでは、各エンティティに対して以下のアクセス権限が設定できます。

| 権限 | 説明 |
| --- | --- |
| Public | 全ユーザーがアクセス可能 |
| Private | 作成者のみアクセス可能 |
| Shared | 指定したユーザー/グループのみアクセス可能 |

### ページネーション

一覧取得APIは`will_paginate`によるページネーションを使用しています。

- デフォルトの1ページあたり件数: 20件
- 最大件数: 200件
- CSV/XLSエクスポート時はページネーション無効

### データエクスポート

一覧取得API（index）では以下の形式でのエクスポートが可能です。

- CSV: `Accept: text/csv` または `.csv` 拡張子
- XLS: `Accept: application/xls` または `.xls` 拡張子

### タグ機能

エンティティにはタグを付与できます。検索時に `#タグ名` 形式で検索するとタグでフィルタリングされます。

### Webhook/通知

コメント追加時、購読しているユーザーにメール通知が送信されます。通知を受け取るには `subscribe_to_comment_replies` 設定が必要です。

### API認証の拡張

標準のセッション認証に加え、APIトークン認証の実装も可能です。`User`モデルには`authentication_token`カラムが存在します。

### Rate Limiting

本システムには標準でRate Limitingは実装されていません。本番環境ではnginxやRackミドルウェアでの制限を推奨します。
