# Fat Free CRM コードリーディングガイドライン

## はじめに

このガイドラインは、Fat Free CRM（オープンソースのCRMプラットフォーム）のコードベースを効率的に理解するための手引きです。
Ruby on Railsに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、Rubyの基本構文と概念を解説します。

### 1.1 プログラム構造

Rubyはオブジェクト指向スクリプト言語です。Fat Free CRMでは、Ruby on Railsフレームワークを使用してMVCアーキテクチャを実装しています。

**クラス定義の基本:**
```ruby
# ファイル: app/models/entities/lead.rb:40-44
class Lead < ActiveRecord::Base
  belongs_to :user, optional: true
  belongs_to :campaign, optional: true
  belongs_to :assignee, class_name: "User", foreign_key: :assigned_to, optional: true
  has_one :contact, dependent: :nullify
```

**解説:**
- `class Lead < ActiveRecord::Base` - LeadクラスがActiveRecord::Baseを継承
- `belongs_to`, `has_one`, `has_many` - Railsの関連付けマクロ

### 1.2 データ型と変数

**インスタンス変数（@で始まる）:**
```ruby
# ファイル: app/controllers/home_controller.rb:14-17
def index
  @activities = get_activities
  @my_tasks = Task.visible_on_dashboard(current_user).includes(:user, :asset).by_due_at
  @my_opportunities = Opportunity.visible_on_dashboard(current_user).includes(:account, :user, :tags).by_closes_on.by_amount
  @my_accounts = Account.visible_on_dashboard(current_user).includes(:user, :tags).by_name
```

**シンボル（:で始まる）:**
```ruby
# ファイル: app/controllers/application_controller.rb:24-27
respond_to :html, only: %i[index show auto_complete]
respond_to :js
respond_to :json, :xml, except: :edit
respond_to :atom, :csv, :rss, :xls, only: :index
```

### 1.3 制御構造

**条件分岐:**
```ruby
# ファイル: app/models/entities/lead.rb:158-164
def full_name(format = nil)
  if format.nil? || format == "before"
    "#{first_name} #{last_name}"
  else
    "#{last_name}, #{first_name}"
  end
end
```

**スコープとラムダ:**
```ruby
# ファイル: app/models/entities/lead.rb:54-60
scope :state, lambda { |filters|
  where(['status IN (?)' + (filters.delete('other') ? ' OR status IS NULL' : ''), filters])
}
scope :converted,    ->       { where(status: 'converted') }
scope :for_campaign, ->(id)   { where(campaign_id: id) }
scope :created_by,   ->(user) { where(user_id: user.id) }
```

### 1.4 関数/メソッド定義

**インスタンスメソッド:**
```ruby
# ファイル: app/models/entities/lead.rb:95-104
def save_with_permissions(params)
  self.campaign = Campaign.find(params[:campaign]) unless params[:campaign].blank?
  if params[:lead][:access] == "Campaign" && campaign
    save_with_model_permissions(Campaign.find(campaign_id))
  else
    self.attributes = params[:lead]
    save
  end
end
```

**クラスメソッド:**
```ruby
# ファイル: app/models/entities/lead.rb:86-88
def self.per_page
  20
end
```

### 1.5 モジュール/インポート

Rubyではモジュールを使用して名前空間を管理します。

**モジュールの定義とインクルード:**
```ruby
# ファイル: lib/fat_free_crm/permissions.rb:8-13
module FatFreeCRM
  module Permissions
    def self.included(base)
      base.extend(ClassMethods)
    end
```

**require/require_relativeによる読み込み:**
```ruby
# ファイル: lib/fat_free_crm.rb:33-54
require 'fat_free_crm/load_settings'
require "fat_free_crm/gem_dependencies"
require "fat_free_crm/gem_ext"
require "fat_free_crm/custom_fields"
require "fat_free_crm/version"
```

---

## 2. プロジェクト固有の概念

> このセクションでは、Fat Free CRM特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### Ruby on Rails基本概念

**ActiveRecordモデル:**
ActiveRecordはRailsのORM（Object-Relational Mapping）で、データベーステーブルをRubyクラスにマッピングします。

```ruby
# ファイル: app/models/entities/account.rb:30-42
class Account < ActiveRecord::Base
  belongs_to :user, optional: true
  belongs_to :assignee, class_name: "User", foreign_key: :assigned_to, optional: true
  has_many :account_contacts, dependent: :destroy
  has_many :contacts, -> { distinct }, through: :account_contacts
  has_many :account_opportunities, dependent: :destroy
  has_many :opportunities, -> { order("opportunities.id DESC").distinct }, through: :account_opportunities
```

**コントローラーとアクション:**
```ruby
# ファイル: app/controllers/entities/leads_controller.rb:8-21
class LeadsController < EntitiesController
  before_action :get_data_for_sidebar, only: :index
  autocomplete :account, :name, full: true

  def index
    @leads = get_leads(page: page_param)
    respond_with @leads do |format|
      format.xls { render layout: 'header' }
      format.csv { render csv: @leads }
    end
  end
```

#### Deviseによる認証

Fat Free CRMはDeviseを使用してユーザー認証を実装しています。

```ruby
# ファイル: config/routes.rb:17-20
devise_for :users, controllers: { registrations: 'registrations',
                                  sessions: 'sessions',
                                  passwords: 'passwords',
                                  confirmations: 'confirmations' }
```

### 2.2 プロジェクト独自のパターン

#### Entitiesコントローラー継承パターン

すべてのエンティティコントローラー（Leads, Contacts, Accounts等）は`EntitiesController`を継承しています。

```ruby
# ファイル: app/controllers/entities_controller.rb:8-19
class EntitiesController < ApplicationController
  before_action :set_current_tab, only: %i[index show]
  before_action :set_view, only: %i[index show redraw]
  before_action :set_options, only: :index
  before_action :load_ransack_search, only: :index

  load_and_authorize_resource

  after_action :update_recently_viewed, only: :show

  helper_method :entity, :entities
```

#### パーミッションシステム

Fat Free CRMは独自のパーミッションシステムを実装しています。

```ruby
# ファイル: lib/fat_free_crm/permissions.rb:14-29
module ClassMethods
  def uses_user_permissions
    unless included_modules.include?(InstanceMethods)
      has_many :permissions, as: :asset
      scope :my, ->(current_user) { accessible_by(current_user.ability) }
      include FatFreeCRM::Permissions::InstanceMethods
      extend FatFreeCRM::Permissions::SingletonMethods
    end
  end
end
```

#### カスタムフィールドシステム

動的にフィールドを追加できるカスタムフィールド機能があります。

```ruby
# ファイル: app/models/entities/lead.rb:69
has_fields
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `snake_case.rb` | Rubyファイル名 | `leads_controller.rb`, `account.rb` |
| `app/models/entities/` | エンティティモデル | `lead.rb`, `account.rb`, `contact.rb` |
| `app/controllers/entities/` | エンティティコントローラー | `leads_controller.rb` |
| `app/views/{リソース名}/` | ビューファイル | `app/views/leads/` |
| `lib/fat_free_crm/` | コアライブラリ | `permissions.rb`, `fields.rb` |
| `spec/` | テストファイル | `spec/models/`, `spec/controllers/` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Controller` | コントローラークラス | `LeadsController`, `HomeController` |
| `Helper` | ヘルパーモジュール | `LeadsHelper`, `ApplicationHelper` |
| `get_` | データ取得メソッド | `get_leads`, `get_data_for_sidebar` |
| `respond_to_` | レスポンス処理メソッド | `respond_to_destroy`, `respond_to_not_found` |
| `save_with_` | 複合保存メソッド | `save_with_permissions`, `save_with_account_and_permissions` |
| `update_with_` | 複合更新メソッド | `update_with_lead_counters` |
| `@` | インスタンス変数 | `@leads`, `@account` |
| `@@` | クラス変数 | 使用頻度低 |
| `:` | シンボル | `:index`, `:show`, `:create` |

### 3.3 プログラム分類一覧

| 分類 | 場所 | 説明 |
|------|------|------|
| エンティティモデル | `app/models/entities/` | CRMの主要ビジネスオブジェクト |
| 中間テーブルモデル | `app/models/entities/` | `account_contact.rb`, `account_opportunity.rb` |
| ユーザー関連モデル | `app/models/users/` | ユーザー、グループ、権限 |
| フィールド関連モデル | `app/models/fields/` | カスタムフィールド定義 |
| オブザーバー | `app/models/observers/` | モデル変更時の自動処理 |
| ポリモーフィックモデル | `app/models/polymorphic/` | 複数モデルに関連付く汎用モデル |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
fat_free_crm/
├── app/
│   ├── assets/          # JavaScript, CSS, 画像
│   ├── controllers/     # コントローラー
│   │   ├── admin/       # 管理画面コントローラー
│   │   └── entities/    # エンティティコントローラー
│   ├── helpers/         # ビューヘルパー
│   ├── inputs/          # カスタムフォーム入力
│   ├── mailers/         # メール送信
│   ├── models/          # モデル
│   │   ├── entities/    # エンティティモデル
│   │   ├── fields/      # カスタムフィールド
│   │   ├── observers/   # オブザーバー
│   │   ├── polymorphic/ # ポリモーフィックモデル
│   │   └── users/       # ユーザー関連
│   └── views/           # ビューテンプレート
├── config/
│   ├── environments/    # 環境別設定
│   ├── initializers/    # 初期化処理
│   ├── locales/         # 国際化ファイル
│   ├── routes.rb        # ルーティング定義
│   └── settings.default.yml  # デフォルト設定
├── db/
│   ├── migrate/         # マイグレーション
│   ├── schema.rb        # データベーススキーマ
│   └── seeds/           # 初期データ
├── lib/
│   └── fat_free_crm/    # コアライブラリ
├── spec/                # テストファイル
└── vendor/              # サードパーティライブラリ
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `app/controllers/` | リクエスト処理、レスポンス生成 | `application_controller.rb`, `entities_controller.rb` |
| `app/models/entities/` | CRMエンティティのビジネスロジック | `lead.rb`, `account.rb`, `contact.rb`, `opportunity.rb` |
| `app/views/` | HTMLテンプレート | `*.html.erb`, `*.js.erb` |
| `config/` | アプリケーション設定 | `routes.rb`, `settings.default.yml` |
| `db/migrate/` | データベース変更履歴 | タイムスタンプ付きマイグレーションファイル |
| `lib/fat_free_crm/` | 再利用可能なコア機能 | `permissions.rb`, `fields.rb`, `exportable.rb` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

Fat Free CRMは典型的なRails MVCアーキテクチャを採用しています。

```
┌─────────────────────────────────────────────────────────────┐
│                      クライアント                            │
│                    (Webブラウザ)                             │
└─────────────────────────────────────────────────────────────┘
                           │
                           ▼
┌─────────────────────────────────────────────────────────────┐
│                       Routes                                │
│                   (config/routes.rb)                        │
└─────────────────────────────────────────────────────────────┘
                           │
                           ▼
┌─────────────────────────────────────────────────────────────┐
│                     Controllers                             │
│  ┌─────────────────────────────────────────────────────┐   │
│  │           ApplicationController                      │   │
│  │  ┌───────────────────────────────────────────────┐  │   │
│  │  │         EntitiesController                    │  │   │
│  │  │  ┌─────────────────────────────────────────┐  │  │   │
│  │  │  │ LeadsController, ContactsController    │  │  │   │
│  │  │  │ AccountsController, OpportunitiesController │ │   │
│  │  │  └─────────────────────────────────────────┘  │  │   │
│  │  └───────────────────────────────────────────────┘  │   │
│  └─────────────────────────────────────────────────────┘   │
└─────────────────────────────────────────────────────────────┘
                           │
                           ▼
┌─────────────────────────────────────────────────────────────┐
│                       Models                                │
│  ┌────────────┐ ┌────────────┐ ┌────────────┐ ┌──────────┐ │
│  │   Lead     │ │  Contact   │ │  Account   │ │Opportunity│ │
│  └────────────┘ └────────────┘ └────────────┘ └──────────┘ │
│  ┌────────────┐ ┌────────────┐ ┌────────────┐ ┌──────────┐ │
│  │  Campaign  │ │   Task     │ │   User     │ │  Comment │ │
│  └────────────┘ └────────────┘ └────────────┘ └──────────┘ │
└─────────────────────────────────────────────────────────────┘
                           │
                           ▼
┌─────────────────────────────────────────────────────────────┐
│                      Database                               │
│              (PostgreSQL/MySQL/SQLite)                      │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| プレゼンテーション | ユーザーインターフェース、HTMLレンダリング | `app/views/`, `app/helpers/` |
| コントローラー | リクエスト処理、ビジネスロジック呼び出し | `app/controllers/` |
| モデル | ビジネスロジック、データ永続化 | `app/models/` |
| ライブラリ | 共通機能、拡張機能 | `lib/fat_free_crm/` |
| データベース | データ格納 | `db/schema.rb`, `db/migrate/` |

### 5.3 データフロー

**リクエスト処理の流れ:**

1. **ルーティング** - `config/routes.rb`でURLをコントローラーアクションにマッピング
2. **認証** - Deviseによるユーザー認証（`before_action :authenticate_user!`）
3. **認可** - CanCanCanによるアクセス制御（`load_and_authorize_resource`）
4. **アクション実行** - コントローラーメソッドでビジネスロジック実行
5. **レスポンス生成** - ビューテンプレートでHTML/JSON/XMLを生成

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

**アプリケーション起動:**
```ruby
# ファイル: config/application.rb:31-46
module FatFreeCRM
  class Application < Rails::Application
    config.load_defaults 7.0

    # Models are organized in sub-directories
    config.autoload_paths += Dir[Rails.root.join("app/models/**")] +
                             Dir[Rails.root.join("app/controllers/entities")]

    # Activate observers
    config.active_record.observers = :lead_observer, :opportunity_observer, :task_observer, :entity_observer
```

**ルーティング定義:**
```ruby
# ファイル: config/routes.rb:8-11
Rails.application.routes.draw do
  resources :lists

  root to: 'home#index'
```

### 6.2 ビジネスロジック

**リード（見込み客）のビジネスロジック:**
```ruby
# ファイル: app/models/entities/lead.rb:124-133
def promote(params)
  account_params = params[:account] || {}
  opportunity_params = params[:opportunity] || {}

  account     = Account.create_or_select_for(self, account_params)
  opportunity = Opportunity.create_for(self, account, opportunity_params)
  contact     = Contact.create_for(self, account, opportunity, params)

  [account, opportunity, contact]
end
```

**解説:** `promote`メソッドは、リードを顧客（Contact）に変換する際に、関連するAccount（取引先）とOpportunity（商談）も同時に作成します。

### 6.3 データアクセス

**スコープによるデータ取得:**
```ruby
# ファイル: app/models/entities/opportunity.rb:43-53
scope :state, lambda { |filters|
  where('stage IN (?)' + (filters.delete('other') ? ' OR stage IS NULL' : ''), filters)
}
scope :created_by,  ->(user) { where('user_id = ?', user.id) }
scope :assigned_to, ->(user) { where('assigned_to = ?', user.id) }
scope :won,         -> { where("opportunities.stage = 'won'") }
scope :lost,        -> { where("opportunities.stage = 'lost'") }
scope :not_lost,    -> { where("opportunities.stage <> 'lost'") }
scope :pipeline,    -> { where("opportunities.stage IS NULL OR (opportunities.stage != 'won' AND opportunities.stage != 'lost')") }
```

### 6.4 ユーティリティ/共通機能

**パーミッション機能:**
```ruby
# ファイル: lib/fat_free_crm/permissions.rb:61-64
def access=(value)
  remove_permissions unless value == 'Shared'
  super(value)
end
```

**エクスポート機能:**
```ruby
# ファイル: lib/fat_free_crm/exportable.rb
# CSVおよびXLSエクスポート機能を提供
exportable
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| 継承コントローラー | EntitiesControllerを継承 | 高 | `leads_controller.rb` |
| スコープチェーン | 複数スコープの連結 | 高 | 全エンティティモデル |
| ポリモーフィック関連 | 複数モデルへの関連付け | 中 | `comments`, `addresses`, `tasks` |
| before_actionフィルター | アクション前処理 | 高 | 全コントローラー |
| respond_with | 複数フォーマット対応 | 高 | 全コントローラー |
| カスタムバリデーション | ビジネスルール検証 | 中 | エンティティモデル |

### 各パターンの詳細

#### パターン1: 継承コントローラー

**目的:** 共通のCRUD操作を基底クラスに集約し、コードの重複を削減

**実装例:**
```ruby
# ファイル: app/controllers/entities/leads_controller.rb:8
class LeadsController < EntitiesController
```

**解説:** すべてのエンティティコントローラーは`EntitiesController`を継承し、`attach`, `discard`, `subscribe`, `unsubscribe`などの共通アクションを利用可能にしています。

#### パターン2: スコープチェーン

**目的:** 複雑なクエリを読みやすいメソッドチェーンで表現

**実装例:**
```ruby
# ファイル: app/controllers/home_controller.rb:16-17
@my_opportunities = Opportunity.visible_on_dashboard(current_user).includes(:account, :user, :tags).by_closes_on.by_amount
@my_accounts = Account.visible_on_dashboard(current_user).includes(:user, :tags).by_name
```

**解説:** `visible_on_dashboard`, `includes`, `by_closes_on`, `by_amount`などのスコープを連結して、必要なデータを効率的に取得しています。

#### パターン3: before_actionフィルター

**目的:** アクション実行前の共通処理を定義

**実装例:**
```ruby
# ファイル: app/controllers/application_controller.rb:11-18
before_action :configure_devise_parameters, if: :devise_controller?
before_action :authenticate_user!
before_action :set_paper_trail_whodunnit
before_action :set_context
before_action :clear_setting_cache
before_action :cors_preflight_check
before_action { hook(:app_before_filter, self) }
after_action { hook(:app_after_filter, self) }
```

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（ルーティング）
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: リードをコンタクトに変換（Promote）

**概要:** 見込み客（Lead）を顧客（Contact）に変換するビジネスフロー

**処理フロー:**
```
LeadsController#convert → LeadsController#promote → Lead#promote → Contact.create_for
```

**詳細な追跡:**

1. **ルーティング** (`config/routes.rb:115-120`)
   ```ruby
   member do
     get :convert
     post :discard
     post :subscribe
     post :unsubscribe
     put :attach
     match :promote, via: %i[patch put]
   ```

2. **コントローラーアクション（変換フォーム表示）** (`app/controllers/entities/leads_controller.rb:108-116`)
   ```ruby
   def convert
     @account = Account.new(user: current_user, name: @lead.company, access: "Lead")
     @accounts = Account.my(current_user).order('name')
     @opportunity = Opportunity.new(user: current_user, access: "Lead", stage: "prospecting", campaign: @lead.campaign, source: @lead.source)

     @previous = Lead.my(current_user).find_by_id(detect_previous_id) || detect_previous_id if detect_previous_id

     respond_with(@lead)
   end
   ```

3. **コントローラーアクション（変換実行）** (`app/controllers/entities/leads_controller.rb:120-134`)
   ```ruby
   def promote
     @account, @opportunity, @contact = @lead.promote(params.permit!)
     @accounts = Account.my(current_user).order('name')
     @stage = Setting.unroll(:opportunity_stage)

     respond_with(@lead) do |format|
       if @account.errors.empty? && @opportunity.errors.empty? && @contact.errors.empty?
         @lead.convert
         update_sidebar
   ```

4. **モデルのビジネスロジック** (`app/models/entities/lead.rb:124-133`)
   ```ruby
   def promote(params)
     account_params = params[:account] || {}
     opportunity_params = params[:opportunity] || {}

     account     = Account.create_or_select_for(self, account_params)
     opportunity = Opportunity.create_for(self, account, opportunity_params)
     contact     = Contact.create_for(self, account, opportunity, params)

     [account, opportunity, contact]
   end
   ```

5. **Contact作成** (`app/models/entities/contact.rb:171-205`)
   ```ruby
   def self.create_for(model, account, opportunity, params)
     attributes = {
       lead_id:     model.id,
       user_id:     params[:account][:user_id] || account.user_id,
       assigned_to: params[:account][:assigned_to],
       access:      params[:access]
     }
     # ...リードの属性をコンタクトにコピー
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか（routes.rb）
- [ ] 呼び出し関係を把握したか（コントローラー -> モデル）
- [ ] データの変換ポイントを確認したか（パラメータ -> モデル属性）
- [ ] エラーハンドリングを確認したか（errors.empty?のチェック）
- [ ] 最終的な出力を確認したか（respond_withのレスポンス）

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. README.md - プロジェクト概要
2. `docs/code-to-docs/機能一覧/機能一覧.csv` - 機能の全体像
3. `docs/code-to-docs/画面一覧/画面一覧.csv` - UI構成

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/機能一覧.csv` - 機能ID特定
2. `docs/code-to-docs/画面設計書/` - 該当画面の詳細
3. 関連するコントローラー・モデルのソースコード

#### 改修作業を行う場合
1. 対象機能の画面設計書
2. `config/routes.rb` - URL・アクション対応
3. 該当コントローラー・モデル
4. `spec/` - 既存テストケース

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、セットアップ手順 | 最初に読む |
| `config/routes.rb` | URLとアクションのマッピング | 機能調査時 |
| `db/schema.rb` | データベース構造 | データ構造調査時 |
| `config/settings.default.yml` | アプリケーション設定 | 設定変更時 |
| `docs/code-to-docs/データベース設計書/` | DB設計詳細 | データモデル理解時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `scope :my`とは何ですか？
A: ユーザーがアクセス可能なレコードのみを返すスコープです。`lib/fat_free_crm/permissions.rb:25`で定義されています。
```ruby
scope :my, ->(current_user) { accessible_by(current_user.ability) }
```

#### Q: `has_fields`とは何ですか？
A: カスタムフィールド機能を有効化するマクロです。`lib/fat_free_crm/fields.rb`で定義されており、動的にフィールドを追加できます。

#### Q: `uses_user_permissions`とは何ですか？
A: パーミッションシステムをモデルに追加するマクロです。`lib/fat_free_crm/permissions.rb`で定義されています。

#### Q: before_actionが多すぎて追跡が難しい
A: `ApplicationController` -> `EntitiesController` -> 各コントローラーの順で継承されているため、親クラスのbefore_actionも確認してください。

### デバッグのヒント

1. **ログ確認:** `log/development.log`でSQLクエリやエラーを確認
2. **Railsコンソール:** `rails console`でモデルの動作を対話的にテスト
3. **byebug:** ソースコードに`byebug`を挿入してブレークポイントを設定

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Lead | 見込み客（リード）。まだ顧客化していない潜在顧客 |
| Contact | 連絡先。顧客化した人物 |
| Account | 取引先。会社・組織 |
| Opportunity | 商談。ビジネスチャンス |
| Campaign | キャンペーン。マーケティング活動 |
| Task | タスク。やるべき作業 |
| Promote | リードをコンタクトに変換する操作 |
| Scope | ActiveRecordの検索条件を再利用可能にする機能 |
| before_action | コントローラーアクション実行前に呼ばれるフィルター |
| Devise | Rails用認証ライブラリ |
| CanCanCan | Rails用認可ライブラリ |
| PaperTrail | モデル変更履歴を記録するライブラリ |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `app/controllers/application_controller.rb` | 基底コントローラー | 認証、CORS、共通処理 |
| `app/controllers/entities_controller.rb` | エンティティ共通コントローラー | CRUD共通処理 |
| `app/models/entities/lead.rb` | リードモデル | 見込み客のビジネスロジック |
| `app/models/entities/contact.rb` | コンタクトモデル | 連絡先のビジネスロジック |
| `app/models/entities/account.rb` | アカウントモデル | 取引先のビジネスロジック |
| `app/models/entities/opportunity.rb` | オポチュニティモデル | 商談のビジネスロジック |
| `lib/fat_free_crm/permissions.rb` | パーミッション機能 | アクセス制御ロジック |
| `lib/fat_free_crm/fields.rb` | カスタムフィールド機能 | 動的フィールド追加 |
| `config/routes.rb` | ルーティング定義 | URL・アクションマッピング |
| `config/settings.default.yml` | デフォルト設定 | アプリケーション設定値 |

### C. 参考資料

- [Ruby on Rails ガイド（日本語）](https://railsguides.jp/)
- [Ruby公式ドキュメント](https://www.ruby-lang.org/ja/documentation/)
- [Fat Free CRM公式サイト](http://www.fatfreecrm.com/)
- [Fat Free CRM GitHub Wiki](https://github.com/fatfreecrm/fat_free_crm/wiki)
- [Devise公式ドキュメント](https://github.com/heartcombo/devise)
- [CanCanCan公式ドキュメント](https://github.com/CanCanCommunity/cancancan)
