# セキュリティ設計書

## 概要

本ドキュメントは、Fat Free CRM（Ruby on Rails製CRMアプリケーション）のセキュリティ設計について記載する。認証・認可、データ保護、通信セキュリティ、入出力対策、監査ログなどのセキュリティ機能の実装方式を定義する。

## 認証設計

### 認証方式

| 項目 | 内容 |
| --- | --- |
| 認証フレームワーク | Devise 4.6+ |
| 認証方式 | データベース認証（メールアドレス/ユーザー名 + パスワード） |
| パスワードハッシュ | authlogic_sha512（devise-encryptableによる互換性維持） |
| ハッシュストレッチ回数 | 本番環境: 20回、テスト環境: 1回 |
| パスワード長 | 8〜128文字 |

### Deviseモジュール構成

Userモデルで使用しているDeviseモジュール:
- `database_authenticatable`: データベース認証
- `registerable`: ユーザー登録機能
- `confirmable`: メールアドレス確認
- `encryptable`: カスタム暗号化アルゴリズム対応
- `recoverable`: パスワードリセット
- `rememberable`: ログイン状態の記憶
- `trackable`: ログイン履歴の追跡

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | Deviseデフォルト（2週間） | config.remember_for設定で変更可能 |
| セッション固定化対策 | 有効 | Devise標準機能 |
| ログアウト時記憶トークン無効化 | 有効 | expire_all_remember_me_on_sign_out = true |
| HTTPセッションスキップ | http_auth | skip_session_storage = [:http_auth] |
| パスワードリセットトークン有効期限 | 6時間 | reset_password_within = 6.hours |
| メール確認の再確認 | 有効 | reconfirmable = true |

### ユーザーアカウント状態管理

| 状態 | 説明 |
| --- | --- |
| 未確認 | メールアドレス確認が未完了 |
| 承認待ち | user_signup設定が:needs_approvalの場合、管理者承認待ち |
| 停止中 | 管理者によりsuspended_atが設定された状態 |
| 有効 | confirmed && !awaits_approval && !suspended |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 管理者（admin） | 全リソースへのフルアクセス | admin属性がtrueのユーザー |
| 一般ユーザー | 制限付きアクセス | 以下のルールに基づく |

### アクセス制御

CanCanCan（CanCan）による認可制御を実装。Abilityクラスで権限を定義。

#### エンティティアクセスルール

対象エンティティ: Account, Campaign, Contact, Lead, Opportunity

| 条件 | 権限 |
| --- | --- |
| access = 'Public' | 全ユーザーがアクセス可能 |
| user_id = current_user.id | 所有者としてフルアクセス |
| assigned_to = current_user.id | 担当者としてフルアクセス |
| Permissionテーブルに登録 | 共有ユーザー/グループとしてアクセス可能 |

#### タスクアクセスルール

| 条件 | 権限 |
| --- | --- |
| 作成 | 認証済みユーザーは作成可能 |
| user_id = current_user.id | 所有者としてフルアクセス |
| assigned_to = current_user.id | 担当者としてフルアクセス |
| completed_by = current_user.id | 完了者としてフルアクセス |

#### 管理者機能

Admin::ApplicationControllerにより、管理者権限チェックを実装:
- `require_admin_user`: admin属性がtrueのユーザーのみアクセス許可
- 非管理者はルートパスへリダイレクト

### 共有アクセス（Shared Access）

| アクセスレベル | 説明 |
| --- | --- |
| Private | 所有者と担当者のみアクセス可能 |
| Public | 全ユーザーがアクセス可能 |
| Shared | 明示的に許可されたユーザー/グループのみアクセス可能 |

Permissionモデルによる共有管理:
- user_idまたはgroup_idによるアクセス許可
- asset_id, asset_typeによるポリモーフィック関連

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | config.force_ssl設定（本番環境で有効化推奨、デフォルト無効） |
| HSTS | force_ssl有効化時に自動設定 |
| CORS | 全オリジン許可（Access-Control-Allow-Origin: *） |

### CORS設定詳細

ApplicationControllerでの実装:
- 許可メソッド: POST, GET, PUT, DELETE, OPTIONS
- 許可ヘッダー: Origin, Content-Type, Accept, Authorization, Token
- プリフライトキャッシュ: 1728000秒（20日）

**注意**: 現在のCORS設定は全オリジンを許可しており、本番環境では特定ドメインに制限することを推奨。

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS（HTTPS有効化時） |
| パスワード | authlogic_sha512（SHA-512ベースハッシュ、20回ストレッチ） |
| セッション | Rails標準のcookie暗号化 |
| 認証トークン | SecureRandom生成 |

### 機密情報管理

#### パラメータフィルタリング
config/application.rbでのフィルタ設定:
- password
- encrypted_password
- password_salt
- password_confirmation

これらのパラメータはログ出力時にマスクされる。

#### シークレットキー管理
- Devise secret_key: Rails.application.secret_key_baseを使用
- credentials.yml.encまたは環境変数による管理推奨

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | ERB::Util（h, j メソッド）による出力エスケープ、sanitizeヘルパーの使用 |
| SQLインジェクション | Active Record ORM使用、Ransack検索による安全なクエリ生成 |
| CSRF | protect_from_forgery with: :exception（例外発生による防御） |
| マスアサインメント | Strong Parameters（permit）による許可属性制限 |

### CSRF対策詳細

ApplicationControllerでの実装:
- `protect_from_forgery with: :exception`: 不正なトークンで例外発生
- `handle_unverified_request`: CODESPACE開発環境以外では例外発生

### Strong Parameters実装

各コントローラーでpermitメソッドにより許可属性を明示:

| コントローラー | 許可属性 |
| --- | --- |
| UsersController | username, email, first_name, last_name等（adminを除く） |
| Admin::UsersController | admin, username, email, password, group_ids等 |
| 各エンティティコントローラー | params.permit!（全属性許可）※要注意 |

**注意**: EntitiesControllerのresource_paramsでは`permit!`が使用されており、セキュリティリスクとなる可能性がある。

### 入力バリデーション

#### Userモデル
- メールアドレス: 形式検証、一意性検証、長さ制限（3〜254文字）
- ユーザー名: 形式検証（英数字、アンダースコア、ハイフンのみ）、一意性検証
- パスワード: 確認入力との一致検証、長さ制限（8〜128文字）

#### エンティティモデル
- 各フィールドの長さ制限（例: first_name 64文字）
- 数値フィールドの範囲検証（例: rating 0〜5）
- カテゴリ/ステータスの許可値リスト検証

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| バージョン履歴（PaperTrail） | エンティティの作成/更新/削除/閲覧履歴 | 無期限（データベース保存） |
| ログインアクティビティ | sign_in_count, last_sign_in_at, current_sign_in_at, IPアドレス | ユーザーレコード存在期間 |

### PaperTrail監査設定

対象モデル: User, Account, Campaign, Contact, Lead, Opportunity

| 設定 | 値 |
| --- | --- |
| バージョンクラス | Version |
| 記録イベント | create, update, destroy, view |
| 除外属性 | subscribed_users, last_sign_in_at等 |
| 実行者追跡 | whodunnit（set_paper_trail_whodunnit） |

### ログイン追跡（Devise Trackable）

| 属性 | 内容 |
| --- | --- |
| sign_in_count | ログイン回数 |
| current_sign_in_at | 現在のログイン日時 |
| last_sign_in_at | 前回のログイン日時 |
| current_sign_in_ip | 現在のログインIPアドレス |
| last_sign_in_ip | 前回のログインIPアドレス |

## 備考

### セキュリティ静的解析

Gemfileにbrakeman（セキュリティスキャナー）が開発用に含まれており、定期的なセキュリティ診断の実施を推奨。

### 推奨セキュリティ強化項目

1. **force_ssl有効化**: 本番環境でconfig.force_ssl = trueを設定
2. **CORS制限**: 本番環境では特定ドメインのみ許可するよう設定変更
3. **permit!の除去**: EntitiesControllerでの全属性許可を個別属性指定に変更
4. **アカウントロックアウト**: Devise lockableモジュールの有効化を検討
5. **二要素認証**: devise-two-factorなどによる追加認証層の検討
6. **CSP（Content Security Policy）**: HTTPヘッダーによるXSS対策強化
7. **レート制限**: rack-attackなどによるブルートフォース対策
