# 機能設計書 36-タスク作成

## 概要

本ドキュメントは、Fat Free CRMシステムにおける「タスク作成」機能の設計を定義する。新規タスクを作成するための機能である。

### 本機能の処理概要

タスク作成機能は、営業活動における ToDo 項目を新規登録するための機能である。

**業務上の目的・背景**：営業担当者が顧客対応や社内業務で発生するタスク（電話フォローアップ、メール送信、ミーティング準備、資料作成など）を記録し、期限管理を行うために必要な機能。タスクを取引先、連絡先、商談などのエンティティに紐付けることで、営業活動の文脈を維持しながらタスク管理が可能。

**機能の利用シーン**：顧客との商談後にフォローアップタスクを作成、上司からの指示をタスクとして記録、他のメンバーへのタスク割り当て、ダッシュボードからのクイックタスク作成、エンティティ詳細画面からの関連タスク作成。

**主要な処理内容**：
1. 新規タスクフォームの表示（new アクション）
2. タスクの期限バケット・カテゴリ選択肢の取得
3. 関連エンティティの設定（relatedパラメータ指定時）
4. タスクレコードの保存（create アクション）
5. 期限日時の自動計算（バケットに基づく）
6. サイドバー情報の更新

**関連システム・外部連携**：タスクは取引先、連絡先、リード、商談、キャンペーンなどのエンティティにポリモーフィック関連で紐付け可能。

**権限による制御**：ログインユーザーのみがタスクを作成可能。タスクは作成者または割り当て先ユーザーのみが操作可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 29 | タスク新規作成フォーム | 主画面 | タスク情報入力と作成実行 |
| 28 | タスク一覧画面 | 参照画面 | 一覧からの新規作成呼び出し |
| 6 | ホーム画面（ダッシュボード） | 参照画面 | ダッシュボードからのクイックタスク作成 |

## 機能種別

CRUD操作（Create） / データ登録

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| task[name] | String | Yes | タスク名 | 必須、最大255文字 |
| task[user_id] | Integer | Yes | 作成者ID | 必須（hidden fieldで設定） |
| task[assigned_to] | Integer | No | 割り当て先ユーザーID | ユーザー存在チェック |
| task[bucket] | String | No | 期限バケット | 設定された選択肢のいずれか |
| task[due_at] | DateTime | No | 期限日時 | 日時形式 |
| task[calendar] | String | No | 特定日時（specific_time用） | 日時形式 |
| task[category] | String | No | カテゴリ | 設定された選択肢 |
| task[priority] | String | No | 優先度 | - |
| task[asset_id] | Integer | No | 関連エンティティID | - |
| task[asset_type] | String | No | 関連エンティティ種別 | Account/Contact/Lead/Opportunity/Campaign |
| task[background_info] | String | No | 背景情報 | 最大255文字 |
| related | String | No | 関連エンティティ指定 | "account_123" 形式 |

### 入力データソース

- 画面入力（タスク作成フォーム）
- セッション情報（current_user）
- URLパラメータ（related - 関連エンティティ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| @task | Task | 作成されたタスクオブジェクト |
| @view | String | 現在の表示ビュー |
| @bucket | Array | 期限バケット選択肢 |
| @category | Array | カテゴリ選択肢 |
| @asset | Object | 関連エンティティ |

### 出力先

- 画面表示（AJAX応答）
- データベース（tasksテーブル）

## 処理フロー

### 処理シーケンス

```
1. newアクション（フォーム表示）
   └─ GET /tasks/new
2. フォーム用データの準備
   └─ バケット・カテゴリ選択肢、関連エンティティ
3. 関連エンティティの取得（relatedパラメータ指定時）
   └─ model.classify.constantize.my(current_user).find_by_id(id)
4. createアクション（作成処理）
   └─ POST /tasks
5. タスクレコードの保存
   └─ before_create: set_due_date（期限自動計算）
6. サイドバーの更新（一覧画面からの呼び出し時）
   └─ update_sidebar
7. 応答の生成
   └─ JS形式での応答
```

### フローチャート

```mermaid
flowchart TD
    A[新規作成リクエスト] --> B[newアクション]
    B --> C{relatedパラメータあり?}
    C -->|Yes| D[関連エンティティ取得]
    C -->|No| E[フォーム表示]
    D --> F{エンティティ存在?}
    F -->|No| G[not foundエラー]
    F -->|Yes| H[assetとして設定]
    H --> E
    E --> I[ユーザー入力]
    I --> J[createアクション]
    J --> K{バリデーション}
    K -->|失敗| L[エラー表示]
    L --> I
    K -->|成功| M[set_due_date]
    M --> N[タスク保存]
    N --> O{一覧からの呼び出し?}
    O -->|Yes| P[サイドバー更新]
    O -->|No| Q[応答生成]
    P --> Q
    Q --> R[完了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-36-01 | タスク名必須 | タスク名は必須入力 | 常時 |
| BR-36-02 | 作成者設定 | user_idは現在のログインユーザー | 常時 |
| BR-36-03 | 期限自動計算 | バケットに基づいてdue_atを自動設定 | bucket指定時 |
| BR-36-04 | 特定日時必須 | bucket=specific_timeの場合はcalendar必須 | specific_time選択時 |
| BR-36-05 | 関連エンティティ検証 | 関連エンティティはmy()スコープで取得可能なもの | related指定時 |

### 計算ロジック

**期限自動計算（set_due_date）**:
- due_asap: nil（期限なし）
- due_today: 今日0時
- due_tomorrow: 明日0時
- due_this_week: 今週末
- due_next_week: 来週末
- due_later: 100年後（実質無期限）
- specific_time: calendarパラメータをパース

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| タスク作成 | tasks | INSERT | 新規タスクの作成 |
| 履歴記録 | versions | INSERT | PaperTrailによる履歴記録 |

### テーブル別操作詳細

#### tasks

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | user_id | current_user.id | 作成者 |
| INSERT | name | フォーム入力値 | 必須 |
| INSERT | assigned_to | フォーム入力値 | 任意 |
| INSERT | bucket | フォーム入力値 | 期限バケット |
| INSERT | due_at | 計算値（set_due_date） | 自動計算 |
| INSERT | category | フォーム入力値 | カテゴリ |
| INSERT | priority | フォーム入力値 | 優先度 |
| INSERT | asset_id | フォーム入力値 | 関連エンティティID |
| INSERT | asset_type | フォーム入力値 | 関連エンティティ種別 |
| INSERT | background_info | フォーム入力値 | 背景情報 |
| INSERT | created_at | 現在日時 | 自動設定 |
| INSERT | updated_at | 現在日時 | 自動設定 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 404 | Not Found | 関連エンティティが存在しない | respond_to_related_not_found |
| 422 | Validation Error | バリデーション失敗 | エラーメッセージ表示 |
| - | missing_task_name | タスク名未入力 | エラーメッセージ表示 |
| - | invalid_date | specific_time時のcalendar形式不正 | エラーメッセージ表示 |

### リトライ仕様

バリデーションエラー時はフォームを再表示し、ユーザーが修正して再送信可能。

## トランザクション仕様

- タスクの作成は暗黙的なトランザクション内で実行
- PaperTrailによる履歴記録も同一トランザクション内で実行

## パフォーマンス要件

- フォーム表示: 500ミリ秒以内
- 作成処理: 1秒以内

## セキュリティ考慮事項

- ログインユーザーのみがタスクを作成可能
- Strong Parametersによる入力パラメータのホワイトリスト制御
- 関連エンティティはmy()スコープでアクセス可能なもののみ設定可能
- CSRF対策（Railsデフォルト）
- XSS対策（出力時のエスケープ）

## 備考

- バケット選択肢はSetting.unroll(:task_bucket)から取得（先頭を除く）
- カテゴリ選択肢はSetting.unroll(:task_category)から取得
- "On Specific Date..." オプションが最後に追加される
- assigneeへの通知機能（notify_assignee）は実装予定だが現在は空実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

タスクモデルのバリデーションと期限計算ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | task.rb | `app/models/polymorphic/task.rb` | バリデーションとコールバック |

**読解のコツ**:
- **116-119行目**: バリデーション定義
  - `validates_presence_of :user` - 作成者必須
  - `validates_presence_of :name` - タスク名必須
  - `validates_presence_of :calendar` - specific_time時のcalendar必須
- **121-123行目**: before_create/before_updateコールバック
  - `set_due_date` - 期限の自動計算
  - `notify_assignee` - 担当者通知（未実装）
- **217-234行目**: `set_due_date` メソッドの実装
- **279-283行目**: `specific_time` バリデーション
- **286-289行目**: `parse_calendar_date` - 日時パース

#### Step 2: エントリーポイントを理解する

TasksControllerのnewとcreateアクションが処理の起点。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | tasks_controller.rb | `app/controllers/tasks_controller.rb` | newとcreateアクション |

**主要処理フロー**:
1. **32-50行目**: newアクション全体
2. **35行目**: `@task = Task.new` - 新規タスクインスタンス
3. **37行目**: `Setting.unroll(:task_bucket)[1..-1]` - バケット選択肢取得
4. **38行目**: `Setting.unroll(:task_category)` - カテゴリ選択肢取得
5. **40-47行目**: relatedパラメータによる関連エンティティ取得
6. **66-77行目**: createアクション全体
7. **70行目**: `Task.new(task_params)` - パラメータでタスク作成
8. **73行目**: `@task.save` - 保存処理（コールバック実行）
9. **74行目**: `update_sidebar if called_from_index_page?` - サイドバー更新

#### Step 3: ビューの構造を理解する

作成フォームのビューファイルを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | _new.html.haml | `app/views/tasks/_new.html.haml` | 作成フォーム |
| 3-2 | new.js.haml | `app/views/tasks/new.js.haml` | AJAX応答 |
| 3-3 | create.js.haml | `app/views/tasks/create.js.haml` | 作成後JS処理 |

**主要処理フロー**:
- **1行目**: `simple_form_for(@task, ...)` - フォーム生成
- **4-6行目**: hidden_field_tagで作成者、関連エンティティを設定
- **8-9行目**: トップセクションとカスタムフィールドのレンダリング
- **12行目**: サブミットボタン

### プログラム呼び出し階層図

```
TasksController#new
    │
    ├─ Task.new
    │
    ├─ Setting.unroll(:task_bucket)
    │      └─ [[:due_asap, "due_asap"], [:due_today, "due_today"], ...]
    │
    ├─ Setting.unroll(:task_category)
    │      └─ [[:call, "call"], [:email, "email"], ...]
    │
    ├─ params[:related] (optional)
    │      └─ model.classify.constantize.my(current_user).find_by_id(id)
    │             └─ @asset = related
    │
    └─ respond_with(@task)
           └─ render "new.js.haml"
                  └─ render "_new.html.haml"

TasksController#create
    │
    ├─ Task.new(task_params)
    │      └─ Strong Parameters filtering
    │
    ├─ @task.save
    │      ├─ validates_presence_of :user, :name
    │      ├─ validate :specific_time
    │      ├─ before_create :set_due_date
    │      │      └─ due_at = calculated_value
    │      └─ INSERT INTO tasks
    │
    ├─ update_sidebar (if called_from_index_page?)
    │      └─ Task.totals(current_user, @view)
    │
    └─ respond_with(@task)
           └─ render "create.js.haml"
```

### データフロー図

```
[入力]                      [処理]                           [出力]

GET /tasks/new         TasksController#new
                               │
                               ▼
                       Setting.unroll取得
                       (@bucket, @category)
                               │
                               ▼
                       関連エンティティ取得
                       (params[:related])
                               │
                               ▼
                       フォーム表示
                               │
                               ▼
POST /tasks            TasksController#create
                               │
                               ▼
                       Task.new(task_params)
                               │
                               ▼
                       バリデーション
                               │
                    ┌──────────┴──────────┐
                    ▼                     ▼
              失敗                    成功
              エラー表示              set_due_date
                                          │
                                          ▼
                                    INSERT tasks
                                          │
                                          ▼
                                    update_sidebar
                                          │
                                          ▼
                                    UI更新（AJAX）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tasks_controller.rb | `app/controllers/tasks_controller.rb` | コントローラ | new/createアクション |
| task.rb | `app/models/polymorphic/task.rb` | モデル | バリデーション、期限計算 |
| _new.html.haml | `app/views/tasks/_new.html.haml` | ビュー | 作成フォームテンプレート |
| new.js.haml | `app/views/tasks/new.js.haml` | ビュー | フォーム表示AJAX |
| create.js.haml | `app/views/tasks/create.js.haml` | ビュー | 作成後AJAX処理 |
| _top_section.html.haml | `app/views/tasks/_top_section.html.haml` | ビュー | フォーム上部セクション |
| routes.rb | `config/routes.rb` | 設定 | POST /tasks ルーティング |
