# 機能設計書 50-パスワード変更

## 概要

本ドキュメントは、Fat Free CRMのパスワード変更機能（UsersController#password, #change_password）の設計仕様を定義するものである。

### 本機能の処理概要

パスワード変更機能は、ログイン済みユーザーが自分のパスワードを変更する機能である。現在のパスワードを確認した上で、新しいパスワードに変更する。

**業務上の目的・背景**：セキュリティ上の理由から、ユーザーは定期的にパスワードを変更する必要がある。また、パスワードが漏洩した可能性がある場合は速やかに変更する必要がある。このパスワード変更機能により、ユーザーは自分自身でパスワードを安全に更新できる。

**機能の利用シーン**：ユーザーが自分のプロファイルページで「パスワード変更」リンクをクリックし、現在のパスワードを入力した上で新しいパスワードを設定する場面で利用される。定期的なパスワード更新、セキュリティインシデント対応などに使用される。

**主要な処理内容**：
1. 【password】パスワード変更フォームを表示
2. 【change_password】現在のパスワードを検証
3. 【change_password】新しいパスワードを設定して保存

**関連システム・外部連携**：なし（内部処理のみ）

**権限による制御**：CanCanCanで権限管理。自分のパスワードまたは管理者のみ変更可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 32 | ユーザー詳細画面（プロフィール） | 主画面 | パスワードの変更 |

## 機能種別

認証情報更新処理 / UPDATE操作

## 入力仕様

### 入力パラメータ

#### passwordアクション

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | Integer | Yes | 対象ユーザーのID | 存在するユーザーであること |
| cancel | String | No | キャンセルフラグ | "true"でキャンセル処理 |

#### change_passwordアクション

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| id | Integer | Yes | 対象ユーザーのID | 存在するユーザーであること |
| current_password | String | Yes | 現在のパスワード | 正しいパスワードであること |
| user[password] | String | No | 新しいパスワード | 空でない場合のみ更新 |
| user[password_confirmation] | String | No | 新しいパスワード（確認） | password と一致すること |

### 入力データソース

- URLパラメータ：ユーザーID
- 画面入力：パスワード変更フォーム

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| @user | User | 対象ユーザーオブジェクト |
| flash[:notice] | String | パスワード変更結果メッセージ |

### 出力先

- データベース：usersテーブルへUPDATE（encrypted_password）
- 画面表示：AJAXレスポンスでフォーム表示更新

## 処理フロー

### 処理シーケンス

```
【passwordアクション】
1. ユーザー取得
   └─ load_and_authorize_resourceで自動取得・権限チェック

2. キャンセル処理判定
   ├─ cancel=true: パスワード変更フォームを閉じる
   └─ cancel=false: パスワード変更フォームを表示

【change_passwordアクション】
1. ユーザー取得
   └─ load_and_authorize_resourceで自動取得・権限チェック

2. 現在のパスワード検証
   └─ @user.valid_password?(params[:current_password])

3. パスワード検証結果による分岐
   ├─ 検証成功
   │      ├─ 新パスワードが空の場合：変更なしメッセージ
   │      └─ 新パスワードあり：パスワード更新
   └─ 検証失敗
          └─ エラーメッセージを追加

4. レスポンス生成
   └─ change_password.js.hamlでJavaScript応答
```

### フローチャート

```mermaid
flowchart TD
    subgraph passwordアクション
        A1[フォーム表示リクエスト] --> B1[権限チェック・ユーザー取得]
        B1 --> C1{権限あり?}
        C1 -->|No| D1[アクセス拒否]
        C1 -->|Yes| E1{cancel=true?}
        E1 -->|Yes| F1[フォームを閉じる]
        E1 -->|No| G1[パスワード変更フォーム表示]
    end

    subgraph change_passwordアクション
        A2[パスワード変更リクエスト] --> B2[権限チェック・ユーザー取得]
        B2 --> C2{権限あり?}
        C2 -->|No| D2[アクセス拒否]
        C2 -->|Yes| E2[現在のパスワード検証]
        E2 --> F2{検証成功?}
        F2 -->|No| G2[エラーメッセージ追加]
        F2 -->|Yes| H2{新パスワードあり?}
        H2 -->|No| I2[変更なしメッセージ]
        H2 -->|Yes| J2[パスワード更新]
        J2 --> K2{保存成功?}
        K2 -->|Yes| L2[成功メッセージ・フォーム閉じる]
        K2 -->|No| M2[バリデーションエラー表示]
        G2 --> N2[フォーム再表示]
        I2 --> L2
        M2 --> N2
    end
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-50-01 | 現在パスワード必須 | パスワード変更には現在のパスワード入力が必須 | 常時 |
| BR-50-02 | パスワード確認 | 新パスワードと確認入力が一致する必要がある | 新パスワード入力時 |
| BR-50-03 | 空パスワード許容 | 新パスワードが空の場合は変更しない（変更なしメッセージ表示） | 新パスワードが空の場合 |
| BR-50-04 | 権限制御 | 自分のパスワードまたは管理者のみ変更可能 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| ユーザー取得 | users | SELECT | 対象ユーザー取得 |
| パスワード更新 | users | UPDATE | encrypted_passwordの更新 |
| バージョン履歴 | versions | INSERT | 変更履歴レコード作成（paper_trail） |

### テーブル別操作詳細

#### users

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| SELECT | 全カラム | id = :user_id | 対象ユーザー取得 |
| UPDATE | encrypted_password | Devise暗号化処理 | パスワードハッシュ |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | RecordNotFound | 対象ユーザーが存在しない | 404エラー応答 |
| - | CanCan::AccessDenied | アクセス権限なし | 403エラー応答 |
| - | 現在パスワード不正 | 現在のパスワードが間違っている | エラーメッセージ表示 |
| - | パスワード確認不一致 | 新パスワードと確認が一致しない | エラーメッセージ表示 |

### リトライ仕様

リトライ不要（ユーザーが再入力して再送信）

## トランザクション仕様

パスワード更新はActiveRecordのデフォルトトランザクションで実行

## パフォーマンス要件

- パスワード変更処理は1秒以内に応答（パスワードハッシュ計算含む）

## セキュリティ考慮事項

- 認証必須：ApplicationControllerで認証チェック
- 権限管理：CanCanCanで詳細な権限制御
- 現在パスワード確認：なりすまし防止のため現在のパスワード入力必須
- パスワードハッシュ：Deviseによるbcryptハッシュ化
- XSS対策：ビューでのエスケープ処理
- パスワードログ除外：paper_trailでパスワード関連フィールドは除外

## 備考

- Deviseのvalid_password?メソッドでパスワード検証
- パスワードハッシュはDeviseが自動的に処理
- パスワードリセット（忘れた場合）は別機能（No.55）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ユーザーモデルのパスワード関連を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | user.rb | `app/models/users/user.rb` | Devise設定（49-50行目）、パスワードバリデーション（96-98行目） |

**読解のコツ**: UserモデルはDeviseのdatabase_authenticatableを使用し、パスワードのハッシュ化と検証を行う。password_required?メソッドでパスワード必須条件を定義。

#### Step 2: エントリーポイントを理解する

password/change_passwordアクションの処理を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | users_controller.rb | `app/controllers/users_controller.rb` | passwordアクション（74-76行目） |
| 2-2 | users_controller.rb | `app/controllers/users_controller.rb` | change_passwordアクション（81-96行目） |

**主要処理フロー**:
1. **82行目**: valid_password?で現在のパスワード検証
2. **83-84行目**: 新パスワードが空の場合、変更なしメッセージ
3. **86-88行目**: 新パスワードをセットして保存
4. **89行目**: 成功メッセージ設定
5. **92行目**: 検証失敗時にエラー追加

#### Step 3: ビューレスポンスを理解する

JavaScript応答の生成を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | password.js.haml | `app/views/users/password.js.haml` | フォーム表示/キャンセル処理（1-10行目） |
| 3-2 | change_password.js.haml | `app/views/users/change_password.js.haml` | パスワード変更後の表示処理（1-12行目） |
| 3-3 | _password.html.haml | `app/views/users/_password.html.haml` | パスワード変更フォームパーシャル |

**主要処理フロー**:
- **password.js.haml 1-3行目**: cancel=trueでフォームを閉じる
- **password.js.haml 5-10行目**: フォーム表示、現在のパスワード欄にフォーカス
- **change_password.js.haml 1-6行目**: 成功時にフォームを閉じてメッセージ表示
- **change_password.js.haml 7-12行目**: エラー時にフォーム再表示、該当フィールドにフォーカス

### プログラム呼び出し階層図

```
UsersController#password
    │
    ├─ load_and_authorize_resource (CanCanCan)
    │      ├─ User.find
    │      └─ authorize! :password, @user
    │
    └─ respond_with(@user)
           └─ password.js.haml
                  └─ _password.html.haml

UsersController#change_password
    │
    ├─ load_and_authorize_resource (CanCanCan)
    │      ├─ User.find
    │      └─ authorize! :change_password, @user
    │
    ├─ @user.valid_password?(params[:current_password])
    │      └─ Devise: パスワード検証
    │
    ├─ 検証成功 & 新パスワードあり
    │      ├─ @user.password = params[:user][:password]
    │      ├─ @user.password_confirmation = ...
    │      └─ @user.save
    │              └─ Devise: パスワードハッシュ化
    │
    └─ respond_with(@user)
           └─ change_password.js.haml
```

### データフロー図

```
[入力]                        [処理]                           [出力]

【password】
ユーザーID ───▶ UsersController#password ───▶ JavaScript応答
                    │                            └─ フォーム表示
                    └─ 権限チェック

【change_password】
パスワード ───▶ UsersController#change_password ───▶ データベース
  ├─ current_password    │                              └─ UPDATE users
  ├─ password            ├─ 権限チェック                     (encrypted_password)
  └─ confirmation        ├─ 現在パスワード検証
                          └─ 新パスワード保存          ───▶ JavaScript応答
                                                           ├─ 成功: フォーム閉じる
                                                           └─ 失敗: エラー表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| users_controller.rb | `app/controllers/users_controller.rb` | コントローラー | password/change_passwordアクション |
| user.rb | `app/models/users/user.rb` | モデル | Userモデル、Devise認証 |
| password.js.haml | `app/views/users/password.js.haml` | ビュー | フォーム表示JavaScript |
| change_password.js.haml | `app/views/users/change_password.js.haml` | ビュー | パスワード変更後JavaScript |
| _password.html.haml | `app/views/users/_password.html.haml` | ビュー | パスワード変更フォームパーシャル |
| routes.rb | `config/routes.rb` | 設定 | ルーティング設定（158, 160行目） |
