# Apache Flink

## 概要

Apache Flinkは、オープンソースのストリーム処理フレームワークであり、強力なストリーム処理とバッチ処理の両方の機能を提供します。高スループットと低レイテンシを同時に実現し、イベント時間処理や exactly-once 保証など、分散ストリーム処理に必要な機能を包括的にサポートしています。

## 主な機能

| 機能カテゴリ | 説明 |
| --- | --- |
| ストリームファースト | ストリーム処理とバッチ処理の両方をサポートするランタイム |
| イベント時間処理 | Dataflowモデルに基づくイベント時間と順序外処理のサポート |
| フレキシブルなウィンドウ処理 | 時間、カウント、セッション、カスタムトリガーによる柔軟なウィンドウ定義 |
| 耐障害性 | exactly-once 処理保証を持つフォールトトレランス |
| バックプレッシャー | ストリーミングプログラムにおける自然なバックプレッシャー機構 |
| Table API / SQL | 宣言的なテーブル操作とSQL対応 |
| Python API (PyFlink) | Pythonによるストリーム/バッチ処理のサポート |
| メモリ管理 | インメモリとディスクベースのデータ処理を効率的に切り替えるカスタムメモリ管理 |

## 技術スタック

### コア

- フレームワーク: Apache Flink (v2.3-SNAPSHOT)
- 言語: Java 11/17/21, Scala 2.12
- ビルドツール: Maven 3.8.6
- ログ: SLF4J 1.7.36, Log4j 2.24.3
- 分散調整: Apache ZooKeeper 3.7.2, Apache Curator 5.4.0
- シリアライゼーション: Apache Avro 1.11.4, Protocol Buffers 4.32.1

### 主要ライブラリ

- Jackson 2.20.1 (JSON処理)
- Apache Beam 2.54.0
- Apache Hive 2.3.10
- Apache ORC 1.5.6
- Py4J 0.10.9.7 (Python連携)
- Testcontainers 1.20.2 (テスト)
- JUnit 5.11.4, Mockito 5.19.0, AssertJ 3.27.3 (テスト)

## ディレクトリ構成

```
flink-master/
├── flink-annotations/          # カスタムアノテーション定義
├── flink-architecture-tests/   # アーキテクチャテスト
├── flink-clients/              # クライアントAPI
├── flink-connectors/           # コネクタ実装
│   ├── flink-connector-base/
│   ├── flink-connector-datagen/
│   ├── flink-connector-files/
│   └── flink-hadoop-compatibility/
├── flink-container/            # コンテナサポート
├── flink-core/                 # コアAPI・ユーティリティ
├── flink-core-api/             # コアAPI定義
├── flink-datastream/           # DataStream実装
├── flink-datastream-api/       # DataStream API定義
├── flink-dist/                 # ディストリビューション
├── flink-docs/                 # ドキュメント
├── flink-end-to-end-tests/     # E2Eテスト
├── flink-examples/             # サンプルコード
├── flink-filesystems/          # ファイルシステム実装
├── flink-formats/              # データフォーマット
│   ├── flink-avro/
│   ├── flink-csv/
│   ├── flink-json/
│   ├── flink-orc/
│   ├── flink-parquet/
│   └── flink-protobuf/
├── flink-kubernetes/           # Kubernetesサポート
├── flink-libraries/            # ライブラリ
├── flink-metrics/              # メトリクス
├── flink-python/               # Python API (PyFlink)
├── flink-queryable-state/      # クエリ可能状態
├── flink-rpc/                  # RPCフレームワーク
├── flink-runtime/              # ランタイム実装
├── flink-runtime-web/          # WebUI
├── flink-state-backends/       # 状態バックエンド
│   ├── flink-statebackend-changelog/
│   ├── flink-statebackend-forst/
│   └── flink-statebackend-rocksdb/
├── flink-streaming-java/       # ストリーミングAPI (Java)
├── flink-table/                # Table API / SQL
│   ├── flink-sql-client/
│   ├── flink-sql-gateway/
│   ├── flink-sql-parser/
│   ├── flink-table-api-java/
│   ├── flink-table-api-scala/
│   ├── flink-table-planner/
│   └── flink-table-runtime/
├── flink-tests/                # テスト
├── flink-yarn/                 # YARNサポート
├── tools/                      # ビルドツール
├── docs/                       # プロジェクトドキュメント
├── pom.xml                     # Mavenプロジェクト設定
├── mvnw / mvnw.cmd             # Maven Wrapper
└── LICENSE / NOTICE            # ライセンス
```

## セットアップ

### 前提条件

- Unix系環境 (Linux, Mac OS X, Cygwin, WSL)
- Git
- Maven 3.8.6
- Java (バージョン 11, 17, または 21)

### インストール手順

1. リポジトリのクローン

```bash
git clone https://github.com/apache/flink.git
cd flink
```

2. ビルド (Javaバージョンに応じて選択)

**Java 11の場合**
```bash
./mvnw clean package -DskipTests -Djdk11 -Pjava11-target
```

**Java 17の場合 (デフォルト)**
```bash
./mvnw clean package -DskipTests -Djdk17 -Pjava17-target
```

**Java 21の場合**
```bash
./mvnw clean package -DskipTests -Djdk21 -Pjava21-target
```

ビルドは約10分で完了し、`build-target`ディレクトリにインストールされます。

## 開発

### 推奨IDE

- **IntelliJ IDEA** (推奨): JavaとScalaの混合プロジェクトに最適
  - Scalaプラグイン: https://plugins.jetbrains.com/plugin/?id=1347
  - セットアップガイド: https://nightlies.apache.org/flink/flink-docs-master/flinkDev/ide_setup.html#intellij-idea

### コードスタイル

- Checkstyle 10.18.2 によるコードチェック
- Spotless 2.43.0 によるフォーマット
- Scalafmt 3.4.3 (Scalaコード)

### テスト

```bash
# 全テストの実行
./mvnw test

# 特定モジュールのテスト
./mvnw test -pl flink-core

# 統合テスト
./mvnw verify -pl flink-tests
```

### API互換性チェック

- japicmp による API 互換性チェック (参照バージョン: 2.2.0)

## 使用例

### ストリーム処理の例

```java
StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
DataStreamSource<String> text = env.socketTextStream(host, port);
DataStream<WordWithCount> windowCounts = text
    .flatMap((FlatMapFunction<String, String>) (line, collector)
        -> Arrays.stream(line.split("\\s")).forEach(collector::collect))
    .returns(String.class)
    .map(word -> new WordWithCount(word, 1))
    .returns(TypeInformation.of(WordWithCount.class))
    .keyBy(wordWithCnt -> wordWithCnt.word)
    .window(TumblingProcessingTimeWindows.of(Duration.ofSeconds(5)))
    .sum("count")
    .returns(TypeInformation.of(WordWithCount.class));

windowCounts.print();
env.execute();
```

### バッチ処理の例

```java
StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
env.setRuntimeMode(RuntimeExecutionMode.BATCH);
FileSource<String> source = FileSource
    .forRecordStreamFormat(new TextLineInputFormat(), new Path("MyInput.txt"))
    .build();
DataStreamSource<String> text = env.fromSource(source, WatermarkStrategy.noWatermarks(), "MySource");
// ... (ストリーム処理と同様のロジック)
```

## 外部コネクタ

多くのFlinkコネクタは個別のリポジトリに分離されています:

- [flink-connector-kafka](https://github.com/apache/flink-connector-kafka)
- [flink-connector-jdbc](https://github.com/apache/flink-connector-jdbc)
- [flink-connector-elasticsearch](https://github.com/apache/flink-connector-elasticsearch)
- [flink-connector-aws](https://github.com/apache/flink-connector-aws)
- [flink-connector-cassandra](https://github.com/apache/flink-connector-cassandra)
- [flink-connector-hbase](https://github.com/apache/flink-connector-hbase)
- [flink-connector-hive](https://github.com/apache/flink-connector-hive)
- [flink-connector-mongodb](https://github.com/apache/flink-connector-mongodb)
- [flink-connector-pulsar](https://github.com/apache/flink-connector-pulsar)
- [flink-connector-rabbitmq](https://github.com/apache/flink-connector-rabbitmq)

## サポート・ドキュメント

- **公式ドキュメント**: https://flink.apache.org
- **メーリングリスト**: https://flink.apache.org/community.html#mailing-lists
- **課題追跡**: https://issues.apache.org/jira/browse/FLINK
- **貢献ガイド**: https://flink.apache.org/contributing/how-to-contribute.html

## ライセンス

Apache License, Version 2.0
