# 機能設計書：組み込み関数

## 1. 機能概要

### 1.1 処理概要

組み込み関数（Built-in Functions）は、Apache Flink Table API/SQLで標準提供されるシステム関数群である。`BuiltInFunctionDefinitions`クラスで定義され、SQLクエリやTable API式内で直接使用できる。

主要な関数カテゴリ：
1. **スカラー関数**: 単一値を返す関数（文字列、数値、日時操作等）
2. **集計関数**: グループ化されたデータを集計する関数（SUM, COUNT, AVG等）
3. **テーブル関数**: 複数行を返す関数（UNNEST等）
4. **その他の関数**: 型変換、NULL処理、条件分岐等

### 1.2 業務上の目的・役割

| 項目 | 内容 |
|------|------|
| 目的 | 標準的なデータ操作・変換機能を提供 |
| 役割 | ユーザーがUDFを定義せずに一般的な処理を実行可能にする |
| 主な利用場面 | データ変換、集計、文字列操作、日時計算、条件分岐 |

### 1.3 利用シーン・ユースケース

1. **文字列処理**
   - 文字列の結合、分割、トリミング
   - パターンマッチング（LIKE, REGEXP）

2. **数値計算**
   - 四則演算、数学関数
   - 統計関数（AVG, STDDEV等）

3. **日時操作**
   - 日付・時刻の抽出、変換
   - 時間間隔の計算

4. **コレクション操作**
   - 配列・マップの操作
   - JSON処理

## 2. 入出力仕様

### 2.1 関数カテゴリ別一覧

#### 2.1.1 スカラー関数

| カテゴリ | 関数例 | 説明 |
|---------|-------|------|
| NULL処理 | IFNULL, COALESCE | NULL値の代替 |
| 型変換 | CAST, TYPEOF | データ型変換、型情報取得 |
| 文字列 | CONCAT, SUBSTRING, TRIM | 文字列操作 |
| 数値 | ABS, ROUND, FLOOR, CEIL | 数値計算 |
| 日時 | NOW, CURRENT_DATE, EXTRACT | 日時操作 |

#### 2.1.2 配列・マップ関数

| 関数名 | 入力 | 出力 | 説明 |
|--------|------|------|------|
| MAP_KEYS | MAP | ARRAY | マップのキー一覧 |
| MAP_VALUES | MAP | ARRAY | マップの値一覧 |
| MAP_ENTRIES | MAP | ARRAY<ROW> | マップのエントリ一覧 |
| MAP_FROM_ARRAYS | ARRAY, ARRAY | MAP | 配列からマップ作成 |
| MAP_UNION | MAP... | MAP | マップの結合 |
| ARRAY_APPEND | ARRAY, element | ARRAY | 要素の追加（末尾） |
| ARRAY_PREPEND | ARRAY, element | ARRAY | 要素の追加（先頭） |
| ARRAY_CONTAINS | ARRAY, element | BOOLEAN | 要素の存在確認 |
| ARRAY_DISTINCT | ARRAY | ARRAY | 重複除去 |
| ARRAY_POSITION | ARRAY, element | INT | 要素の位置取得 |
| ARRAY_REMOVE | ARRAY, element | ARRAY | 要素の削除 |
| ARRAY_REVERSE | ARRAY | ARRAY | 配列の反転 |
| ARRAY_SORT | ARRAY | ARRAY | 配列のソート |

#### 2.1.3 集計関数

| 関数名 | 説明 |
|--------|------|
| COUNT | 行数カウント |
| SUM | 合計 |
| AVG | 平均 |
| MIN | 最小値 |
| MAX | 最大値 |
| COLLECT | リスト収集 |
| DISTINCT | 重複除去 |

## 3. 処理フロー

### 3.1 関数定義構造

```mermaid
classDiagram
    class BuiltInFunctionDefinition {
        +String name
        +FunctionKind kind
        +InputTypeStrategy inputTypeStrategy
        +OutputTypeStrategy outputTypeStrategy
        +String runtimeClass
    }

    class FunctionKind {
        <<enumeration>>
        SCALAR
        AGGREGATE
        TABLE
        PROCESS_TABLE
        OTHER
    }

    class InputTypeStrategy {
        +validateArguments()
        +inferInputTypes()
    }

    class OutputTypeStrategy {
        +inferType()
    }

    BuiltInFunctionDefinition --> FunctionKind
    BuiltInFunctionDefinition --> InputTypeStrategy
    BuiltInFunctionDefinition --> OutputTypeStrategy
```

### 3.2 関数呼び出しフロー

```mermaid
sequenceDiagram
    participant User as ユーザーSQL/API
    participant Parser as Parser
    participant Resolver as ExpressionResolver
    participant BIFD as BuiltInFunctionDefinitions
    participant Runtime as RuntimeFunction

    User->>Parser: 関数呼び出し
    Parser->>Resolver: 式解決
    Resolver->>BIFD: 関数定義検索
    BIFD-->>Resolver: FunctionDefinition
    Resolver->>Resolver: 入力型検証
    Resolver->>Resolver: 出力型推論
    Resolver->>Runtime: コード生成
    Runtime-->>User: 結果
```

## 4. 業務ルール・条件

### 4.1 関数定義パターン

```java
// 基本的な関数定義
public static final BuiltInFunctionDefinition FUNCTION_NAME =
    BuiltInFunctionDefinition.newBuilder()
        .name("FUNCTION_NAME")                    // SQL名
        .kind(SCALAR)                             // 関数種別
        .inputTypeStrategy(...)                   // 入力型戦略
        .outputTypeStrategy(...)                  // 出力型戦略
        .runtimeClass("org.apache.flink...")      // 実行時クラス
        .build();
```

### 4.2 入力型戦略

| 戦略 | 説明 |
|------|------|
| sequence() | 固定引数リスト |
| varyingSequence() | 可変長引数 |
| or() | 複数パターン対応 |
| and() | 複合条件 |
| logical() | 論理型指定 |
| LITERAL | リテラル必須 |
| ANY | 任意の型 |

### 4.3 出力型戦略

| 戦略 | 説明 |
|------|------|
| explicit() | 明示的な型指定 |
| argument(n) | n番目の引数と同じ型 |
| COMMON | 共通の上位型 |
| nullableIfArgs() | 引数がnullableなら結果もnullable |
| nullableIfAllArgs() | 全引数がnullableなら結果もnullable |

### 4.4 NULL処理ルール

```java
// IFNULL: NULLの場合に代替値を返す
IFNULL(value, replacement)

// COALESCE: 最初のNON-NULL値を返す
COALESCE(value1, value2, ...)
```

## 5. コードリーディングガイド

### 5.1 プログラム呼び出し階層図

```
BuiltInFunctionDefinitions (定義クラス)
├── スカラー関数
│   ├── TYPE_OF          → TypeOfFunction
│   ├── IF_NULL          → IfNullFunction
│   ├── COALESCE         → CoalesceFunction
│   └── ...
├── 配列関数
│   ├── ARRAY_APPEND     → ArrayAppendFunction
│   ├── ARRAY_CONTAINS   → ArrayContainsFunction
│   ├── ARRAY_SORT       → ArraySortFunction
│   └── ...
├── マップ関数
│   ├── MAP_KEYS         → MapKeysFunction
│   ├── MAP_VALUES       → MapValuesFunction
│   ├── MAP_ENTRIES      → MapEntriesFunction
│   └── ...
├── 文字列関数
│   ├── CONCAT           → ...
│   ├── SUBSTRING        → ...
│   └── ...
├── 日時関数
│   ├── NOW              → ...
│   ├── CURRENT_DATE     → ...
│   └── ...
└── 集計関数
    ├── COUNT            → ...
    ├── SUM              → ...
    └── ...
```

### 5.2 データフロー図

```
[SQL/Table API]
      │
      ▼
[ExpressionParser]
      │
      ▼
[BuiltInFunctionDefinitions]
      │
      ├── name: 関数名
      ├── kind: 関数種別
      ├── inputTypeStrategy: 入力検証
      ├── outputTypeStrategy: 出力型推論
      └── runtimeClass: 実行時クラス
            │
            ▼
      [CodeGeneration]
            │
            ▼
      [RuntimeFunction]
            │
            ▼
      [Result]
```

### 5.3 関連ファイル一覧

| ファイルパス | 役割 |
|-------------|------|
| `flink-table-common/.../BuiltInFunctionDefinitions.java` | 組み込み関数定義の辞書 |
| `flink-table-common/.../BuiltInFunctionDefinition.java` | 関数定義のBuilder |
| `flink-table-common/.../FunctionDefinition.java` | 関数定義インターフェース |
| `flink-table-common/.../FunctionKind.java` | 関数種別列挙型 |
| `flink-table-runtime/.../scalar/*.java` | スカラー関数の実装クラス群 |

### 5.4 コードリーディング手順

#### Step 1: BuiltInFunctionDefinitionsクラスの概要

```java
// BuiltInFunctionDefinitions.java (119-121行目)
/** Dictionary of function definitions for all built-in functions. */
@PublicEvolving
public final class BuiltInFunctionDefinitions {
```

#### Step 2: 具体的な関数定義例

```java
// MAP_KEYS関数の定義 (157-167行目)
public static final BuiltInFunctionDefinition MAP_KEYS =
    BuiltInFunctionDefinition.newBuilder()
        .name("MAP_KEYS")
        .kind(SCALAR)
        .inputTypeStrategy(
            sequence(
                new String[] {"input"},
                new ArgumentTypeStrategy[] {logical(LogicalTypeRoot.MAP)}))
        .outputTypeStrategy(nullableIfArgs(SpecificTypeStrategies.MAP_KEYS))
        .runtimeClass("org.apache.flink.table.runtime.functions.scalar.MapKeysFunction")
        .build();
```

#### Step 3: ARRAY関数の定義例

```java
// ARRAY_APPEND関数 (241-253行目)
public static final BuiltInFunctionDefinition ARRAY_APPEND =
    BuiltInFunctionDefinition.newBuilder()
        .name("ARRAY_APPEND")
        .kind(SCALAR)
        .inputTypeStrategy(
            sequence(
                Arrays.asList("array", "element"),
                Arrays.asList(logical(LogicalTypeRoot.ARRAY), ARRAY_ELEMENT_ARG)))
        .outputTypeStrategy(nullableIfArgs(nullableIfArgs(ARRAY_APPEND_PREPEND)))
        .runtimeClass("org.apache.flink.table.runtime.functions.scalar.ArrayAppendFunction")
        .build();
```

## 6. 関連機能・API

### 6.1 関連機能

| 機能名 | 関連種別 | 説明 |
|--------|---------|------|
| Table API Java（No.13） | 利用元 | Table APIからの関数呼び出し |
| UDF（No.17） | 拡張機能 | ユーザー定義関数 |
| SQLパーサー（No.18） | 連携機能 | SQL文の解析 |
| SQLプランナー（No.19） | 連携機能 | 関数の実行計画生成 |

### 6.2 使用例

```sql
-- NULL処理
SELECT IFNULL(name, 'Unknown') FROM users;
SELECT COALESCE(phone, mobile, 'N/A') FROM contacts;

-- 配列操作
SELECT ARRAY_APPEND(tags, 'new') FROM items;
SELECT ARRAY_CONTAINS(categories, 'electronics') FROM products;

-- マップ操作
SELECT MAP_KEYS(attributes) FROM items;
SELECT MAP_FROM_ARRAYS(ARRAY['a', 'b'], ARRAY[1, 2]) AS map_col;

-- 日時関数
SELECT EXTRACT(HOUR FROM event_time) FROM events;
SELECT DATE_FORMAT(created_at, 'yyyy-MM-dd') FROM logs;
```

## 7. 設計上の考慮事項

### 7.1 拡張性

- `BuiltInFunctionDefinition.newBuilder()`パターンで新規関数を追加可能
- 入力/出力型戦略の組み合わせで柔軟な型推論を実現

### 7.2 型安全性

- `InputTypeStrategy`で入力型を厳密に検証
- `OutputTypeStrategy`で出力型を正確に推論

### 7.3 パフォーマンス考慮事項

| 考慮点 | 説明 |
|--------|------|
| コード生成 | 関数呼び出しはJITコンパイルされる |
| NULL伝播 | NULL処理は自動的に最適化される |
| 式評価 | 定数折りたたみ最適化が適用される |

## 8. 用語集

| 用語 | 説明 |
|------|------|
| BuiltInFunctionDefinition | 組み込み関数の定義を表すクラス |
| FunctionKind | 関数の種別（SCALAR, AGGREGATE, TABLE等） |
| InputTypeStrategy | 関数の入力型を検証・推論する戦略 |
| OutputTypeStrategy | 関数の出力型を推論する戦略 |
| RuntimeClass | 関数の実行時実装クラス |
| SCALAR | 単一値を返す関数種別 |
| AGGREGATE | 集計を行う関数種別 |
| TABLE | 複数行を返す関数種別 |
