# 通知設計書 13-notifyNewBlockedNodes

## 概要

本ドキュメントは、Apache Flinkの新規ブロックノード通知（notifyNewBlockedNodes）の設計仕様を記述したものである。

### 本通知の処理概要

notifyNewBlockedNodesは、ブロックリストに新しいノードが追加された際にリスナーに通知するメカニズムである。ブロックされたノードとは、障害やエラーによりジョブ実行から除外されるべきノードを指す。

**業務上の目的・背景**：分散処理システムでは、特定のノードで障害が繰り返し発生する場合、そのノードを一時的または永続的にジョブスケジューリングから除外する必要がある。ブロックリスト機能は、障害ノードへのタスク割り当てを防止し、システム全体の安定性を向上させる。

**通知の送信タイミング**：DefaultBlocklistHandlerのaddNewBlockedNodesメソッドが呼び出され、BlocklistTrackerに新しいブロックノードが追加された時点で、登録済みの全BlocklistListenerに対して通知が発行される。

**通知の受信者**：BlocklistListenerインターフェースを実装するクラスが受信者となる。主にJobMasterやResourceManagerがリスナーとして登録される。

**通知内容の概要**：新しくブロックされたノードのコレクション（Collection<BlockedNode>）が通知される。各BlockedNodeには、ノードID、ブロック理由、タイムアウト時刻などの情報が含まれる。

**期待されるアクション**：受信者は、ブロックされたノード上で実行中のタスクの再スケジューリング、スロットの回収、および今後のタスク割り当てからの除外を行う。CompletableFuture<Acknowledge>を返すことで、非同期に処理完了を通知する。

## 通知種別

内部コールバック（Javaメソッド呼び出し、非同期応答対応）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期呼び出し（非同期応答） |
| 優先度 | 高（障害対応に直結） |
| リトライ | なし（呼び出し側で管理） |

### 送信先決定ロジック

DefaultBlocklistHandlerに登録された全てのBlocklistListenerに対して通知される。リスナーはregisterBlocklistListenerメソッドで登録される。

## 通知テンプレート

### コールバックシグネチャ

```java
CompletableFuture<Acknowledge> notifyNewBlockedNodes(Collection<BlockedNode> newNodes);
```

### パラメータ詳細

| パラメータ名 | 型 | 説明 |
|------------|-----|------|
| newNodes | Collection<BlockedNode> | 新しくブロックされたノードのコレクション |

### 戻り値

| 型 | 説明 |
|-----|------|
| CompletableFuture<Acknowledge> | 通知処理完了を示すFuture |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| newNodes | ブロックノードのコレクション | BlocklistTrackerから取得 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | addNewBlockedNodes | 新規ノードが追加された場合 | 既存ノードとのマージも考慮 |
| 登録時 | registerBlocklistListener | 既存のブロックノードがある場合 | 初回登録時に現在のブロックリストを通知 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 空のコレクション | newNodesが空の場合は通知されない |
| 重複ノード | 既にブロック済みのノードのみの場合（ただしマージ結果は通知） |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[addNewBlockedNodes呼び出し] --> B{newNodesが空?}
    B -->|Yes| C[処理終了]
    B -->|No| D[BlocklistTrackerに追加]
    D --> E[追加結果取得]
    E --> F{新規追加あり?}
    F -->|Yes| G[ログ出力]
    G --> H[全リスナーにnotifyNewBlockedNodes]
    H --> I[blockResources呼び出し]
    F -->|No| J{マージ結果あり?}
    J -->|Yes| H
    J -->|No| C
    I --> C
```

## データベース参照・更新仕様

### 参照テーブル一覧

データベースアクセスは発生しない（インメモリ処理）。

### 更新テーブル一覧

データベース更新は発生しない。

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | notifyNewBlockedNodes内で例外発生 | CompletableFutureがexceptionallyで完了 |
| 状態不整合 | メインスレッド外からの呼び出し | assertRunningInMainThreadで検証 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（呼び出し側で管理） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

時間帯の制限なし（24時間稼働）

## セキュリティ考慮事項

- ノードIDは内部識別子であり、機密情報は含まれない
- 通知はJVM内部のメソッド呼び出しで完結

## 備考

- ブロックノードは一定時間経過後に自動的に削除される（タイムアウト機能）
- タイムアウトチェックはscheduleTimeoutCheck()で定期実行

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BlockedNode.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/BlockedNode.java` | ブロックノードの構造 |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BlocklistListener.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/BlocklistListener.java` | インターフェース定義（27-36行目） |

**主要処理フロー**:
1. **行27**: インターフェースの目的説明
2. **行29-35**: notifyNewBlockedNodesメソッドのJavadoc
3. **行35**: CompletableFuture<Acknowledge> notifyNewBlockedNodes(Collection<BlockedNode> newNodes)

#### Step 3: 実装クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DefaultBlocklistHandler.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/DefaultBlocklistHandler.java` | ブロックリスト管理の実装（113-147行目） |

**主要処理フロー**:
- **行113-147**: addNewBlockedNodesメソッド
- **行142**: blocklistListeners.forEach(listener -> listener.notifyNewBlockedNodes(allNodes))
- **行164-174**: registerBlocklistListenerメソッド（初回通知）

### プログラム呼び出し階層図

```
障害検知
    │
    └─ DefaultBlocklistHandler.addNewBlockedNodes(newNodes)
           │
           ├─ assertRunningInMainThread()
           │
           ├─ blocklistTracker.addNewBlockedNodes(newNodes)
           │
           └─ blocklistListeners.forEach()
                  │
                  └─ BlocklistListener.notifyNewBlockedNodes(allNodes)
                         │
                         ├─ JobMaster (ジョブ管理)
                         └─ ResourceManager (リソース管理)
```

### データフロー図

```
[入力]               [処理]                    [出力]

障害検知 ───▶ DefaultBlocklistHandler ───▶ BlocklistListener
                       │                          │
                       ├─ BlocklistTracker        └─ タスク再配置
                       │  (ノード追跡)               スロット回収
                       │
                       └─ BlocklistContext
                          (リソースブロック)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| BlocklistListener.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/BlocklistListener.java` | インターフェース | リスナーインターフェース定義 |
| DefaultBlocklistHandler.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/DefaultBlocklistHandler.java` | 実装クラス | ブロックリスト管理 |
| BlocklistTracker.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/BlocklistTracker.java` | インターフェース | ノード追跡インターフェース |
| BlockedNode.java | `flink-runtime/src/main/java/org/apache/flink/runtime/blocklist/BlockedNode.java` | データ構造 | ブロックノード情報 |
| JobMaster.java | `flink-runtime/src/main/java/org/apache/flink/runtime/jobmaster/JobMaster.java` | リスナー実装 | ジョブ側の処理 |
| ResourceManager.java | `flink-runtime/src/main/java/org/apache/flink/runtime/resourcemanager/ResourceManager.java` | リスナー実装 | リソース管理側の処理 |
