# 機能設計書 90-libc++（C++標準ライブラリ）

## 概要

本ドキュメントは、FreeBSDのC++標準ライブラリ（libc++）の機能設計を記述する。libc++はLLVMプロジェクトのC++標準ライブラリ実装であり、C++11/14/17/20/23の各標準規格に準拠したSTLコンテナ、アルゴリズム、I/O、スレッド、ファイルシステム等の機能を提供する。FreeBSDのデフォルトC++標準ライブラリとして採用されている。

### 本機能の処理概要

**業務上の目的・背景**：libc++はC++プログラムの標準ライブラリ基盤を提供する。STLコンテナ（vector/map/string等）、アルゴリズム（sort/find/transform等）、I/Oストリーム、スレッドサポート、ファイルシステム操作等、C++標準規格に準拠した豊富なAPIを実装する。LLVMプロジェクトの一部としてBSDライセンスで提供され、FreeBSD 10.0以降のデフォルトC++ライブラリとなっている。

**機能の利用シーン**：FreeBSD上のすべてのC++プログラムのランタイム基盤。STLコンテナを使用したデータ構造操作、iostream/fstreamによるI/O、std::threadによる並行処理、std::filesystemによるファイル操作、正規表現処理等。

**主要な処理内容**：
1. コンテナ: vector, string, map, unordered_map, list, deque等のSTLコンテナ
2. アルゴリズム: sort, find, transform, accumulate等の汎用アルゴリズム
3. I/Oストリーム: iostream, fstream, stringstream等のストリームI/O
4. スレッドサポート: std::thread, std::mutex, std::condition_variable, std::future
5. ファイルシステム: std::filesystem（ディレクトリ走査、パス操作、ファイル操作）
6. メモリ管理: shared_ptr, unique_ptr, allocator等のスマートポインタ・アロケータ
7. 文字列処理: std::string, std::string_view, charconv（数値変換）
8. 例外処理: exception, stdexcept, system_error
9. 乱数: random（メルセンヌ・ツイスタ等の乱数エンジン・分布）
10. 正規表現: std::regex

**関連システム・外部連携**：libcxxrt（C++ランタイムサポート: RTTI、例外処理、デマングル）、libc（標準Cライブラリ）、libthr（POSIXスレッド）。

**権限による制御**：ライブラリ自体に権限制御はない。個別API（ファイルシステム等）はOS側の権限チェックに依存。

## 関連画面

該当なし（ライブラリのため画面は存在しない）

## 機能種別

ランタイムライブラリ

## 入力仕様

### 入力パラメータ

libc++は数千のAPI（テンプレート関数・クラス）を提供するため、主要な機能カテゴリを示す。

| 機能カテゴリ | 代表的API | 説明 |
|-------------|---------|------|
| コンテナ | std::vector<T>, std::map<K,V> | 型安全な動的コンテナ |
| アルゴリズム | std::sort(), std::find() | 汎用アルゴリズム |
| 文字列 | std::string, std::to_string() | 文字列操作 |
| I/O | std::cout, std::ifstream | ストリームI/O |
| スレッド | std::thread, std::mutex | マルチスレッド |
| ファイルシステム | std::filesystem::path | ファイル操作 |
| メモリ | std::shared_ptr<T>, std::unique_ptr<T> | スマートポインタ |

### 入力データソース

C++プログラムからのAPI呼出。テンプレートインスタンス化はコンパイル時に解決。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| テンプレートインスタンス | コンパイル時 | ヘッダオンリー部分はコンパイル時にインスタンス化 |
| ランタイム結果 | 各種 | 各APIの仕様に従った戻り値 |
| 例外 | std::exception派生 | エラー時のC++例外 |

### 出力先

呼出側のメモリ空間、ストリーム出力先。

## 処理フロー

### 処理シーケンス

```
1. ライブラリロード
   └─ libc++.so.1が動的リンクされる（libc++.ldscript経由）
2. 静的初期化
   └─ std::cout/std::cin/std::cerr等のグローバルストリームオブジェクト初期化
3. アプリケーション実行
   └─ 各APIの呼出に応じたSTL処理
4. 例外処理
   └─ libcxxrt連携: C++例外のthrow/catch、RTTI、デマングル
5. 終了処理
   └─ グローバルオブジェクトのデストラクタ実行
```

### フローチャート

```mermaid
flowchart TD
    A[プログラム起動] --> B[libc++.so動的ロード]
    B --> C[静的初期化 iostream.cpp]
    C --> D[main呼出]
    D --> E[C++ API使用]
    E --> F{例外発生?}
    F -->|Yes| G[libcxxrt: 例外処理]
    F -->|No| H[正常処理続行]
    G --> I{catch?}
    I -->|Yes| H
    I -->|No| J[std::terminate]
    H --> E
    E --> K[main終了]
    K --> L[グローバルデストラクタ]
    L --> M[プロセス終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-90-01 | C++23準拠 | CXXSTD=c++23でビルド（gcc 12の場合はc++20にフォールバック） | ビルド時 |
| BR-90-02 | libcxxrt連携 | C++ランタイムサポート（RTTI、例外、guard）はlibcxxrtが提供 | 常時 |
| BR-90-03 | ABIバージョン | SHLIB_MAJOR=1。libc++.so.1として提供 | 常時 |
| BR-90-04 | ヘッダオンリー | テンプレートクラス/関数の大部分はヘッダオンリー実装 | コンパイル時 |
| BR-90-05 | visibility hidden | -fvisibility=hiddenでシンボルのデフォルト非公開 | ビルド時 |

### 計算ロジック

ryu/: 浮動小数点数から文字列への高速変換アルゴリズム（d2s: double-to-string、f2s: float-to-string、d2fixed: fixed-point表記）。charconvの基盤。

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | std::bad_alloc | メモリ確保失敗 | C++例外throw |
| - | std::out_of_range | vector/stringの範囲外アクセス(.at()) | C++例外throw |
| - | std::runtime_error | ランタイムエラー | C++例外throw |
| - | std::system_error | OS操作エラー（filesystem等） | C++例外throw |
| - | std::regex_error | 不正な正規表現パターン | C++例外throw |

### リトライ仕様

リトライはライブラリ内部では行わない。呼出側のtry-catch処理に委ねる。

## トランザクション仕様

該当なし（各API呼出は独立。STLコンテナ操作の強い例外安全性保証は規格準拠）

## パフォーマンス要件

- STLアルゴリズムは計算量の上界が規格で保証（sort: O(N log N)、find: O(N)等）
- -fdata-sections/-ffunction-sectionsによるリンク時のデッドコード除去
- ryu算法による高速浮動小数点変換

## セキュリティ考慮事項

- verbose_abort.cppによるセキュアなアサーション失敗処理
- バッファオーバーフロー防止（.at()メソッドの範囲チェック）
- _LIBCPP_REMOVE_TRANSITIVE_INCLUDESによる不要なヘッダ依存排除

## 備考

- libc++のソースはcontrib/llvm-project/libcxx/から取り込み
- C++ランタイム（libcxxrt）はcontrib/libcxxrt/から取り込み（別ライブラリ）
- ソースファイルは約55個のC++ファイル + 9個のlibcxxrtファイルで構成
- ヘッダファイルはcontrib/llvm-project/libcxx/include/に配置
- libc++.ldscriptでリンカスクリプトを管理
- __config_siteでFreeBSD固有の設定を定義

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: ビルド構成を理解する

libc++のFreeBSD向けビルド構成を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Makefile | `lib/libc++/Makefile` | ソースファイル一覧、ビルドフラグ、libcxxrt連携 |
| 1-2 | __config_site | `lib/libc++/__config_site` | FreeBSD固有のlibc++設定 |
| 1-3 | libc++.ldscript | `lib/libc++/libc++.ldscript` | リンカスクリプト |

**読解のコツ**: Makefile内のSRCS変数（17-72行目）がlibc++の非ヘッダ部分のソースファイル一覧。CXXRT_SRCS（74-82行目）がlibcxxrtのソースファイル。SRCDIR=contrib/llvm-project/libcxx/srcがソース本体の場所。CXXSTD=c++23がデフォルトのC++標準。

#### Step 2: ソースファイル構成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | iostream.cpp | `contrib/llvm-project/libcxx/src/iostream.cpp` | グローバルストリームオブジェクト初期化 |
| 2-2 | string.cpp | `contrib/llvm-project/libcxx/src/string.cpp` | std::string非テンプレート部分 |
| 2-3 | thread.cpp | `contrib/llvm-project/libcxx/src/thread.cpp` | std::thread実装 |
| 2-4 | mutex.cpp | `contrib/llvm-project/libcxx/src/mutex.cpp` | std::mutex実装 |

#### Step 3: ファイルシステムサブシステム

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | filesystem/operations.cpp | `contrib/llvm-project/libcxx/src/filesystem/operations.cpp` | ファイル操作 |
| 3-2 | filesystem/path.cpp | `contrib/llvm-project/libcxx/src/filesystem/path.cpp` | パス操作 |
| 3-3 | filesystem/directory_iterator.cpp | `contrib/llvm-project/libcxx/src/filesystem/directory_iterator.cpp` | ディレクトリ走査 |

#### Step 4: libcxxrt（C++ランタイム）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | exception.cc | `contrib/libcxxrt/exception.cc` | C++例外処理ランタイム |
| 4-2 | typeinfo.cc | `contrib/libcxxrt/typeinfo.cc` | RTTI（実行時型情報） |
| 4-3 | dynamic_cast.cc | `contrib/libcxxrt/dynamic_cast.cc` | dynamic_cast実装 |
| 4-4 | guard.cc | `contrib/libcxxrt/guard.cc` | static変数のスレッドセーフ初期化 |
| 4-5 | libelftc_dem_gnu3.c | `contrib/libcxxrt/libelftc_dem_gnu3.c` | C++シンボルデマングル |

#### Step 5: ryu浮動小数点変換

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | ryu/d2s.cpp | `contrib/llvm-project/libcxx/src/ryu/d2s.cpp` | double→string変換 |
| 5-2 | ryu/f2s.cpp | `contrib/llvm-project/libcxx/src/ryu/f2s.cpp` | float→string変換 |
| 5-3 | ryu/d2fixed.cpp | `contrib/llvm-project/libcxx/src/ryu/d2fixed.cpp` | fixed-point表記変換 |

### プログラム呼び出し階層図

```
C++アプリケーション
    |
    +-- STLコンテナ (vector/map/string)
    |       +-- ヘッダオンリー実装（コンパイル時）
    |       +-- new/delete → libc malloc/free
    |
    +-- iostream (cout/cin/cerr)
    |       +-- iostream.cpp: 静的初期化
    |       +-- libc stdio連携
    |
    +-- std::thread / std::mutex
    |       +-- thread.cpp / mutex.cpp
    |       +-- libthr pthread連携
    |
    +-- std::filesystem
    |       +-- filesystem/*.cpp
    |       +-- libc sys/ 連携 (stat/opendir等)
    |
    +-- C++例外処理
    |       +-- libcxxrt: exception.cc
    |       +-- libcxxrt: dynamic_cast.cc
    |       +-- libcxxrt: typeinfo.cc
    |
    +-- charconv (数値変換)
            +-- ryu/: 浮動小数点変換
```

### データフロー図

```
[入力]                          [処理]                          [出力]

C++ソースコード ──> ヘッダ: テンプレートインスタンス化 ──> オブジェクトコード
                    （コンパイル時）

アプリ実行時呼出 ──> libc++ランタイム ──> 処理結果
                     STL操作
                     I/O処理
                     スレッド管理

例外throw ──> libcxxrt: 例外処理 ──> catch節 / terminate
              スタック巻き戻し
              RTTI照合
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Makefile | `lib/libc++/Makefile` | ビルド設定 | FreeBSD向けlibc++ビルド |
| __config_site | `lib/libc++/__config_site` | 設定 | FreeBSD固有libc++設定 |
| libc++.ldscript | `lib/libc++/libc++.ldscript` | リンカスクリプト | libc++リンカ設定 |
| libcxx.imp | `lib/libc++/libcxx.imp` | インポート定義 | シンボルインポート定義 |
| __assertion_handler | `lib/libc++/__assertion_handler` | ヘッダ | アサーション処理 |
| libcxx/src/ | `contrib/llvm-project/libcxx/src/` | ソースディレクトリ | libc++ソース本体（55+ファイル） |
| libcxx/include/ | `contrib/llvm-project/libcxx/include/` | ヘッダディレクトリ | libc++ヘッダ群 |
| libcxxrt/ | `contrib/libcxxrt/` | ソースディレクトリ | C++ランタイムサポート |
| exception.cc | `contrib/libcxxrt/exception.cc` | ソース | C++例外処理 |
| typeinfo.cc | `contrib/libcxxrt/typeinfo.cc` | ソース | RTTI |
| dynamic_cast.cc | `contrib/libcxxrt/dynamic_cast.cc` | ソース | dynamic_cast |
| guard.cc | `contrib/libcxxrt/guard.cc` | ソース | static初期化ガード |
