# インフラ設計書

## 概要

本ドキュメントは、Ghost（オープンソースパブリッシングプラットフォーム）のインフラストラクチャ構成を記載したものです。Ghostは Node.js ベースの CMS であり、Docker コンテナを活用した開発環境と、MySQL/Redis を使用したデータ永続化・キャッシュ機構を備えています。

## システム構成

### 全体構成

Ghost のインフラストラクチャは以下の主要コンポーネントで構成されます：

1. **アプリケーション層**: Node.js ベースの Ghost Core（Express.js）
2. **リバースプロキシ層**: Caddy による SSL 終端・ルーティング
3. **データベース層**: MySQL 8.4（メインストレージ）
4. **キャッシュ層**: Redis 7.0（セッション・キャッシュ）
5. **メール配信**: Mailpit（開発環境）/ Mailgun（本番環境想定）
6. **アナリティクス**: Tinybird（リアルタイム分析）
7. **決済連携**: Stripe
8. **監視**: Prometheus + Grafana

### サーバー構成

| サーバー種別 | 役割 | スペック | 台数（開発環境） |
| --- | --- | --- | --- |
| Ghost Core | メインアプリケーションサーバー（API、フロントエンド配信） | Node.js 22.18.0, Express.js | 1 |
| Caddy Gateway | リバースプロキシ、SSL終端、ルーティング | Caddy 2（Alpine） | 1 |
| MySQL | リレーショナルデータベース | MySQL 8.4.5, InnoDB | 1 |
| Redis | インメモリキャッシュ、セッション管理 | Redis 7.0 | 1 |
| Mailpit | 開発用メールサーバー | Mailpit (linux/amd64) | 1 |
| Analytics | トラフィック分析プロキシ | ghost/traffic-analytics:1.0.42 | 1 |
| Tinybird Local | アナリティクスデータ処理 | tinybirdco/tinybird-local | 1 |

### ネットワーク構成

#### Docker ネットワーク

開発環境では `ghost_dev` という名前の Docker ネットワークを使用：

| ネットワーク名 | タイプ | 用途 |
| --- | --- | --- |
| ghost_dev | bridge | 開発環境のコンテナ間通信 |
| ghost_e2e | bridge | E2Eテスト環境のコンテナ間通信 |

#### ポートマッピング

| サービス | コンテナポート | ホストポート | 用途 |
| --- | --- | --- | --- |
| Ghost | 2368 | 2368 | メインアプリケーション |
| Caddy | 80 | 80, 2368 | HTTP リバースプロキシ |
| Caddy | 443 | 443 | HTTPS（開発時） |
| MySQL | 3306 | 3306 | データベース接続 |
| Redis | 6379 | 6379 | キャッシュ接続 |
| Mailpit SMTP | 1025 | 1025 | メール送信 |
| Mailpit Web | 8025 | 8025 | メールUI |
| Prometheus | 9090 | 9090 | メトリクス収集 |
| Grafana | 3000 | 3000 | 監視ダッシュボード |
| Tinybird | 7181 | 7181 | アナリティクスAPI |
| MinIO S3 | 9000 | 9000 | オブジェクトストレージAPI |
| MinIO Console | 9001 | 9001 | オブジェクトストレージUI |
| Admin Dev | 4200, 4201 | 4200, 4201 | 管理画面開発サーバー |
| Portal Dev | 4175, 4176 | 4175, 4176 | Portal開発サーバー |
| Comments Dev | 7173, 7174 | 7173, 7174 | コメントUI開発サーバー |
| Signup Form Dev | 6174 | 6174 | サインアップフォーム開発サーバー |
| Search Dev | 4178 | 4178 | 検索開発サーバー |
| Announcement Dev | 4177 | 4177 | アナウンスメント開発サーバー |

## ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Node.js | 22.18.0 | アプリケーションランタイム |
| MySQL | 8.4.5 | リレーショナルデータベース |
| Redis | 7.0 | キャッシュ・セッション管理 |
| Caddy | 2 (Alpine) | リバースプロキシ、SSL終端 |
| Express.js | 4.21.2 | Webフレームワーク |
| Knex | 2.4.2 | SQLクエリビルダー |
| Bookshelf | 1.2.0 | ORM |
| Stripe | 8.222.0 | 決済処理SDK |
| Mailgun.js | 10.4.0 | メール配信SDK |

## 外部サービス連携

| サービス名 | 用途 | 接続方式 |
| --- | --- | --- |
| Stripe | 決済処理（メンバーシップ、サブスクリプション） | REST API / Webhook |
| Mailgun | トランザクションメール配信 | REST API |
| Tinybird | リアルタイムアナリティクス | REST API |
| AWS S3 | メディアファイルストレージ（オプション） | S3 API |
| MinIO | S3互換オブジェクトストレージ（開発環境） | S3 API |
| Unsplash | 画像検索・挿入 | REST API |
| Sentry | エラー監視（オプション） | SDK |
| Slack | Webhook通知 | Webhook |

## コンテナ/Docker構成

### Docker Compose 構成ファイル

| ファイル | 用途 |
| --- | --- |
| compose.yml | メイン開発環境（フルDockerモード） |
| compose.dev.yaml | ハイブリッド開発環境（Docker + ホスト） |
| compose.dev.analytics.yaml | アナリティクスサービス追加 |
| compose.dev.storage.yaml | オブジェクトストレージ（MinIO）追加 |
| compose.object-storage.yml | オブジェクトストレージ構成 |
| e2e/compose.yml | E2Eテスト環境 |

### Dockerイメージ構成

#### メイン開発イメージ (Dockerfile)

```
Base: node:22.18.0-bullseye-slim
ビルドステージ:
  - development-base: 依存関係インストール
  - shade-builder: UIコンポーネントライブラリ
  - admin-x-design-system-builder: デザインシステム
  - admin-x-framework-builder: 管理画面フレームワーク
  - stats-builder: 統計アプリ
  - posts-builder: 投稿管理アプリ
  - portal-builder: Portalアプリ
  - admin-x-settings-builder: 設定アプリ
  - activitypub-builder: ActivityPubアプリ
  - admin-ember-builder: Ember管理画面
  - admin-react-builder: React管理画面
  - ghost-assets-builder: フロントエンドアセット
  - development: 最終開発イメージ
```

#### Ghost Dev イメージ (docker/ghost-dev/Dockerfile)

```
Base: node:22.18.0-bullseye-slim
用途: ホットリロード対応の軽量開発イメージ
特徴: ソースコードはランタイム時にマウント
```

#### Gateway イメージ (docker/dev-gateway/Dockerfile)

```
Base: caddy:2-alpine
用途: リバースプロキシ、開発サーバールーティング
拡張: transform-encoder プラグイン
```

### ボリューム構成

| ボリューム | 用途 |
| --- | --- |
| mysql-data | MySQLデータ永続化 |
| redis-data | Redisデータ永続化 |
| ghost-dev-data | Ghostコンテンツデータ |
| ghost-dev-images | 画像ファイル |
| ghost-dev-media | メディアファイル |
| ghost-dev-files | アップロードファイル |
| ghost-dev-logs | ログファイル |
| minio-data | MinIOオブジェクトストレージ |
| node_modules_* | 各パッケージのnode_modules（複数） |

## データベース構成

### MySQL 設定

```
バージョン: 8.4.5
ストレージエンジン: InnoDB
Buffer Pool Size: 1G
Log Buffer Size: 500M
Change Buffer Max Size: 50%
Flush Log at Trx Commit: 0（パフォーマンス優先）
Flush Method: O_DIRECT
```

### データベース環境

| 環境 | データベース名 | ユーザー |
| --- | --- | --- |
| 開発 | ghost_dev / ghost | root / ghost |
| テスト | ghost_testing | ghost |
| 本番 | ghost | 設定による |

### 主要テーブル（70以上）

- **コンテンツ管理**: posts, posts_meta, tags, posts_tags, newsletters
- **ユーザー管理**: users, roles, permissions, sessions
- **メンバー管理**: members, subscriptions, members_products, members_stripe_customers
- **決済**: stripe_products, stripe_prices, offers, offer_redemptions
- **メール**: emails, email_batches, email_recipients
- **コメント**: comments, comment_likes, comment_reports
- **アナリティクス**: members_created_events, members_click_events
- **その他**: settings, integrations, webhooks, api_keys

### SQLite サポート

開発環境では SQLite3 も使用可能：
```
ファイルパス: content/data/ghost-dev.db
```

## キャッシュ構成

### Redis 設定

```
バージョン: 7.0
ホスト: redis (Docker) / 127.0.0.1 (ローカル)
ポート: 6379
用途:
  - セッション管理
  - アプリケーションキャッシュ
  - レートリミッティング
```

### アプリケーションキャッシュ

```javascript
// 設定例
adapters__cache__Redis__host: redis
adapters__cache__Redis__port: 6379
```

### テーマ/管理画面キャッシュ

```json
{
  "caching": {
    "theme": { "maxAge": 0 },  // 開発環境
    "admin": { "maxAge": 0 }   // 開発環境
  }
}
```

## 監視・ログ構成

### Prometheus

```yaml
scrape_interval: 15s（デフォルト）
ターゲット:
  - prometheus:9090（自己監視、5秒間隔）
  - pushgateway:9091（メトリクスプッシュ、1秒間隔）
remote_write: grafana:3000/api/prom/push
```

### Grafana

```
バージョン: 8.5.27
データソース: Prometheus (http://prometheus:9090)
認証: 匿名アクセス有効（Admin権限）
ダッシュボード: /var/lib/grafana/dashboards
```

### Pushgateway

```
バージョン: v1.11.1
ポート: 9091
用途: バッチジョブメトリクスの収集
```

### ロギング設定

```json
{
  "logging": {
    "level": "info",
    "rotation": { "enabled": true },
    "transports": ["file"]
  }
}
```

### Caddy ログ

```
出力: stdout
フォーマット: JSON（本番）/ transform (common_log)（開発）
```

## セキュリティ構成

### 認証・認可

- **セッション管理**: cookie-session, express-session
- **JWT認証**: express-jwt, jsonwebtoken
- **API認証**: Content API Key, Admin API Key
- **OAuth**: Staff用のセッションベース認証

### ブルートフォース対策

- **express-brute**: ログイン試行回数制限
- **brute-knex**: データベースベースのレート制限
- **leaky-bucket**: APIレートリミッティング

### SSL/TLS

```
開発環境:
  - Caddy による自動自己署名証明書
  - local_certs ディレクティブ
  - トラストスクリプト: docker/caddy/trust_caddy_ca.sh

本番環境:
  - Caddy による自動HTTPS（Let's Encrypt）
  - または外部SSL終端（ロードバランサー等）
```

### データ保護

- **パスワード**: bcrypt ハッシュ化
- **メールアドレス検証**: @tryghost/validator
- **入力サニタイズ**: sanitize-html, DOMPurify
- **CORS**: cors ミドルウェア

## スケーリング設計

### 水平スケーリング

Ghost は以下の方法で水平スケーリング可能：

1. **ステートレスアプリケーション**: セッションをRedisに外部化
2. **データベース**: MySQL レプリケーション
3. **キャッシュ**: Redis クラスタリング
4. **ストレージ**: S3/MinIO による共有ストレージ

### Docker Compose プロファイル

| プロファイル | 含まれるサービス |
| --- | --- |
| ghost | ghost, mysql, redis, mailpit |
| split | server, admin, admin-apps, caddy, mysql, redis, mailpit |
| analytics | analytics, tinybird-local, tb-cli |
| monitoring | prometheus, grafana, pushgateway |
| stripe | stripe (CLI) |
| object-storage | minio, minio-setup |
| all | 全サービス |

### 環境分離

```bash
# 開発環境
yarn dev

# アナリティクス付き
yarn dev:analytics

# ストレージ付き
yarn dev:storage

# 全サービス
yarn dev:all
```

## 可用性設計

### ヘルスチェック

各サービスにヘルスチェックを設定：

| サービス | チェック方法 | 間隔 | リトライ |
| --- | --- | --- | --- |
| MySQL | `SELECT 1` | 1s | 120 |
| Redis | `redis-cli ping` | 1s | 120 |
| Ghost | HTTP fetch | 5s | 10 |
| Mailpit | wget spider | 1s | 30 |
| Tinybird | curl /v0/health | 1s | 120 |
| Analytics | Node fetch | 1s | 120 |
| MinIO | curl /minio/health/ready | 1s | 120 |

### 依存関係管理

サービスは `depends_on` と `condition: service_healthy` で依存関係を定義：

```yaml
ghost-dev:
  depends_on:
    mysql:
      condition: service_healthy
    redis:
      condition: service_healthy
    mailpit:
      condition: service_healthy
```

### 再起動ポリシー

```yaml
restart: always  # mysql, redis, mailpit, caddy
restart: "no"    # セットアップ系（minio-setup等）
```

## バックアップ設計

| 対象 | 方式 | 頻度 | 保持期間 |
| --- | --- | --- | --- |
| MySQL データ | mysqldump / Volumeバックアップ | 日次推奨 | 設定による |
| Redis データ | RDB/AOF / Volumeバックアップ | 定期 | 設定による |
| コンテンツファイル | Volume / S3同期 | リアルタイム/日次 | 設定による |
| 設定ファイル | Git管理 | 変更時 | 無期限 |

## 備考

### 本番環境の考慮事項

本リポジトリは開発環境向けの構成が中心です。本番環境では以下を検討してください：

1. **Ghost Docker**: https://github.com/tryghost/ghost-docker を参照
2. **マネージドサービス**: Ghost(Pro) の利用
3. **データベース**: RDS等のマネージドMySQL
4. **キャッシュ**: ElastiCache等のマネージドRedis
5. **ストレージ**: AWS S3等のマネージドオブジェクトストレージ
6. **SSL**: 外部ロードバランサーまたはCDN
7. **監視**: DataDog, New Relic等の外部監視サービス

### 環境変数設定

主要な環境変数：

```bash
# データベース
database__client=mysql2
database__connection__host=mysql
database__connection__user=root
database__connection__password=root
database__connection__database=ghost_dev

# サーバー
server__host=0.0.0.0
server__port=2368

# Redis
adapters__cache__Redis__host=redis
adapters__cache__Redis__port=6379

# メール
mail__transport=SMTP
mail__options__host=mailpit
mail__options__port=1025

# Stripe
STRIPE_SECRET_KEY=sk_test_xxx
STRIPE_PUBLISHABLE_KEY=pk_test_xxx
STRIPE_ACCOUNT_ID=acct_xxx

# アナリティクス
tinybird__stats__endpoint=http://tinybird-local:7181
tinybird__tracker__endpoint=http://localhost/.ghost/analytics/api/v1/page_hit
```
