# セキュリティ設計書

## 概要

本ドキュメントは、Ghostパブリッシングプラットフォームのセキュリティ設計について記述する。Ghostは Node.js/Express ベースのバックエンドと、Ember.js/React ベースのフロントエンドで構成されるモノレポ構造のアプリケーションである。

## 認証設計

### 認証方式

Ghostは複数の認証方式をサポートしている。

#### 1. セッション認証（Admin API）
- **実装**: `ghost/core/core/server/services/auth/session/`
- **方式**: Cookie ベースのセッション認証
- **ライブラリ**: express-session
- **二要素認証（2FA）**: TOTP（Time-based One-Time Password）による検証
  - `otplib` ライブラリを使用
  - 6桁の認証コードを生成
  - 有効期間: 60秒、前後10ステップのウィンドウ

#### 2. JWT認証（Admin API Key）
- **実装**: `ghost/core/core/server/services/auth/api-key/admin.js`
- **方式**: JWT（JSON Web Token）ベースの認証
- **アルゴリズム**: HS256
- **トークン有効期間**: 5分
- **ヘッダー形式**: `Authorization: Ghost {JWT}`

#### 3. API Key認証（Content API）
- **実装**: `ghost/core/core/server/services/auth/api-key/content.js`
- **方式**: クエリパラメータによるAPIキー認証
- **形式**: `?key={API_KEY}`

#### 4. Members JWT認証
- **実装**: `ghost/core/core/server/services/auth/members/`
- **方式**: RS512アルゴリズムのJWT認証
- **ヘッダー形式**: `Authorization: GhostMembers {JWT}`

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | 6ヶ月（約180日） | Cookie maxAge設定 |
| Cookie名 | `ghost-admin-api-session` | Admin API用 |
| httpOnly | true | JavaScript からのアクセスを防止 |
| sameSite | `none`（HTTPS時）/ `lax`（HTTP時） | CSRF対策 |
| secure | サイトURLに依存 | HTTPS時はtrue |
| セッションストア | データベース（SessionModel） | express-session-store |
| セッション固定化対策 | Origin検証によるCSRF保護 | セッションのoriginと リクエストのoriginを比較 |

### パスワードセキュリティ

| 項目 | 実装 |
| --- | --- |
| ハッシュアルゴリズム | bcrypt |
| パスワード検証 | `ghost/core/core/server/lib/validate-password` |
| インポート時の処理 | ランダムUID生成後、ユーザーをロック状態に設定 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| Owner | 全権限 | サイトオーナー、全ての操作が可能 |
| Administrator | 管理者権限 | Ownerロール以外の全ての操作が可能 |
| Super Editor | 上級編集者権限 | Editor + Author/Contributor管理 |
| Editor | 編集者権限 | コンテンツの編集・公開が可能 |
| Author | 著者権限 | 自身のコンテンツの作成・編集が可能 |
| Contributor | 寄稿者権限 | 下書きの作成のみ可能 |

### アクセス制御

- **実装**: `ghost/core/core/server/services/permissions/`
- **方式**: RBAC（Role-Based Access Control）
- **権限チェックAPI**: `canThis(context).{action}.{objectType}()`
  - 例: `canThis(user).edit.post(postId)`
- **内部リクエスト**: `context.internal` が true の場合、権限チェックをバイパス

#### 認可フロー
1. コンテキスト（ユーザー/APIキー）の解析
2. ユーザーまたはAPIキーの権限をロード
3. アクション・オブジェクトタイプに対する権限チェック
4. モデル固有の `permissible` メソッドによる追加チェック

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | サイトURL設定に基づく自動判定、HTTP→HTTPSリダイレクト |
| HSTS | 設定可能（config依存） |
| Cookie Secure属性 | HTTPSサイトでは自動的に有効化 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS（HTTPS設定時） |
| パスワード | bcryptハッシュ |
| セッションシークレット | `admin_session_secret`（設定値） |
| API Key シークレット | Hex形式でデータベースに保存 |
| 2FAトークン | TOTP（HMAC-SHA1ベース） |

### 機密情報管理

- **パスワードハッシュ**: JSONシリアライズ時に自動除外（`toJSON`メソッド）
- **API シークレット**: データベースに保存、リクエスト時のみ利用
- **セッションシークレット**: 設定ファイルで管理

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Handlebars `SafeString`、`escapeExpression` によるエスケープ |
| SQLインジェクション | Bookshelf ORM / Knex.js のパラメータ化クエリ |
| CSRF | Origin検証（セッションのoriginとリクエストのoriginを比較）、SameSite Cookie属性 |
| ブルートフォース | express-brute + brute-knex によるレート制限 |
| 入力検証 | `@tryghost/validator` による検証、JSON Schema検証 |

### 入力検証の実装

- **バリデーター**: `ghost/core/core/server/api/endpoints/utils/validators/`
- **検証内容**:
  - メールアドレス形式検証（`validator.isEmail`）
  - パスワードポリシー検証
  - URL形式検証
  - 文字列長検証
  - 許可値リスト検証（`isIn`）

### ブルートフォース対策

| 対象 | 制限 | 備考 |
| --- | --- | --- |
| グローバルブロック | 50回/時間 | エンドポイント単位、IP単位 |
| ユーザーログイン | 5回/IP | フィボナッチ数列で増加するロック時間 |
| パスワードリセット | 5回/時間/メールアドレス | |
| 2FA検証 | 設定可能 | IP単位、コード単位 |
| メンバー認証 | 設定可能 | メールアドレス単位 |
| Content API Key | メモリストア使用 | 成功時にリセット |

## CORS設定

- **実装**: `ghost/core/core/server/web/api/middleware/cors.js`
- **許可オリジン**:
  - localhost
  - ローカルIPアドレス
  - サイトURL（blog/admin）
- **設定**: オリジンヘッダーの検証、動的CORS有効化/無効化

## フレーム保護

- **実装**: `ghost/core/core/server/web/admin/controller.js`
- **設定**: `adminFrameProtection` が有効な場合、`X-Frame-Options: sameorigin` を設定

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| アクションログ | CRUD操作（作成、編集、削除） | データベースに永続保存 |
| ログインイベント | ユーザーアクティベート/非アクティベート | アクションテーブル |
| セッション情報 | ユーザーエージェント、IPアドレス、ログイン時刻 | セッション有効期間 |

### 監査ログの実装

- **モデル**: `ghost/core/core/server/models/action.js`
- **プラグイン**: `ghost/core/core/server/models/base/plugins/actions.js`
- **記録項目**:
  - イベント種別（added, edited, deleted）
  - リソースID、リソースタイプ
  - アクターID、アクタータイプ
  - コンテキスト情報（プライマリ名、アクション名）

## プライベートブログ機能

- **実装**: `ghost/core/core/frontend/apps/private-blogging/`
- **認証方式**: パスワード認証（SHA-256ハッシュ + ソルト）
- **セッション**: cookie-session（30日間有効）
- **特記事項**:
  - 404エラー時にも認証を要求（情報漏洩防止）
  - プライベートRSSフィードは `public_hash` によるアクセス

## 備考

### セキュリティ関連の設定項目

- `admin_session_secret`: セッション署名用シークレット
- `require_email_mfa`: 2FA必須設定
- `is_private`: プライベートブログモード
- `password`: プライベートブログのパスワード
- `adminFrameProtection`: X-Frame-Options設定
- `spam.*`: レート制限設定

### 使用ライブラリ

| ライブラリ | 用途 |
| --- | --- |
| express-session | セッション管理 |
| jsonwebtoken | JWT生成・検証 |
| express-jwt | JWT認証ミドルウェア |
| otplib | TOTP生成・検証 |
| express-brute | レート制限 |
| brute-knex | ブルートフォース対策のDBストア |
| cors | CORS設定 |
| cookie-session | プライベートブログセッション |
| @tryghost/security | パスワードハッシュ、識別子生成 |
| @tryghost/validator | 入力検証 |
