# データベース設計書

## 概要

本ドキュメントはGhost CMSプラットフォームのデータベース設計を記載しています。Ghostは Node.js ベースのオープンソースブログ/CMSプラットフォームであり、コンテンツ管理、メンバーシップ管理、ニュースレター配信、課金管理などの機能を提供しています。

データベーススキーマは `ghost/core/core/server/data/schema/schema.js` で定義されており、Knex.js マイグレーションを使用してデータベースを管理しています。

## テーブル一覧

### コンテンツ管理

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| posts | 投稿/ページ | ブログ投稿とページのコンテンツを管理 |
| posts_meta | 投稿メタ情報 | 投稿のSEO/OGP情報を管理 |
| posts_authors | 投稿-著者関連 | 投稿と著者の多対多リレーション |
| posts_tags | 投稿-タグ関連 | 投稿とタグの多対多リレーション |
| posts_products | 投稿-商品関連 | 投稿とティア(商品)の多対多リレーション |
| post_revisions | 投稿リビジョン | 投稿の編集履歴(Lexical形式) |
| mobiledoc_revisions | Mobiledocリビジョン | 投稿の編集履歴(Mobiledoc形式) |
| tags | タグ | コンテンツのタグ/カテゴリ |
| collections | コレクション | 投稿のコレクション/グループ |
| collections_posts | コレクション-投稿関連 | コレクションと投稿の多対多リレーション |
| snippets | スニペット | 再利用可能なコンテンツスニペット |
| redirects | リダイレクト | URLリダイレクト設定 |

### ユーザー/スタッフ管理

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| users | スタッフユーザー | 管理者・編集者などのスタッフアカウント |
| roles | ロール | ユーザーの権限ロール |
| roles_users | ロール-ユーザー関連 | ロールとユーザーの多対多リレーション |
| permissions | パーミッション | 操作権限の定義 |
| permissions_users | パーミッション-ユーザー関連 | パーミッションとユーザーの直接割当 |
| permissions_roles | パーミッション-ロール関連 | パーミッションとロールの多対多リレーション |
| invites | 招待 | スタッフ招待トークン |
| sessions | セッション | ユーザーセッション管理 |

### メンバー管理

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| members | メンバー | サイトの購読者/メンバー |
| members_products | メンバー-商品関連 | メンバーとティアの多対多リレーション |
| members_labels | メンバー-ラベル関連 | メンバーとラベルの多対多リレーション |
| labels | ラベル | メンバーの分類ラベル |
| members_newsletters | メンバー-ニュースレター関連 | メンバーのニュースレター購読状況 |

### メンバーイベント

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| members_created_events | メンバー作成イベント | メンバー登録時のアトリビューション情報 |
| members_cancel_events | メンバーキャンセルイベント | サブスクリプションキャンセル履歴 |
| members_payment_events | メンバー支払いイベント | 支払い履歴 |
| members_login_events | メンバーログインイベント | ログイン履歴 |
| members_email_change_events | メンバーメール変更イベント | メールアドレス変更履歴 |
| members_status_events | メンバーステータス変更イベント | ステータス変更履歴 |
| members_product_events | メンバー商品イベント | ティア追加/削除履歴 |
| members_paid_subscription_events | 有料サブスクリプションイベント | 有料サブスクリプション変更履歴 |
| members_subscribe_events | メンバー購読イベント | ニュースレター購読/解除履歴 |
| members_subscription_created_events | サブスクリプション作成イベント | サブスクリプション作成時のアトリビューション情報 |
| members_click_events | メンバークリックイベント | メンバーのリンククリック履歴 |
| members_feedback | メンバーフィードバック | 投稿に対するフィードバック |

### 課金/Stripe連携

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| products | 商品/ティア | サブスクリプションティア(無料/有料) |
| offers | オファー | 割引/プロモーションオファー |
| offer_redemptions | オファー利用 | オファー利用履歴 |
| benefits | ベネフィット | ティアの特典定義 |
| products_benefits | 商品-ベネフィット関連 | ティアとベネフィットの多対多リレーション |
| subscriptions | サブスクリプション | メンバーのサブスクリプション情報 |
| members_stripe_customers | Stripeカスタマー | メンバーとStripeカスタマーの紐付け |
| members_stripe_customers_subscriptions | Stripeサブスクリプション | Stripeサブスクリプション詳細 |
| stripe_products | Stripe商品 | GhostとStripe商品のマッピング |
| stripe_prices | Stripe価格 | Stripeの価格情報 |
| donation_payment_events | 寄付イベント | 寄付支払い履歴 |

### ニュースレター/メール

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| newsletters | ニュースレター | ニュースレターの設定 |
| emails | メール | 送信メールの管理 |
| email_batches | メールバッチ | メール送信バッチ |
| email_recipients | メール受信者 | メール受信者リスト |
| email_recipient_failures | メール配信失敗 | メール配信エラー情報 |
| suppressions | 抑制リスト | メール送信抑制リスト |
| email_spam_complaint_events | スパム苦情イベント | スパム報告履歴 |
| automated_emails | 自動メール | 自動配信メール設定 |
| automated_email_recipients | 自動メール受信者 | 自動メール受信者リスト |
| tokens | トークン | 認証トークン管理 |

### インテグレーション/API

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| integrations | インテグレーション | 外部サービス連携設定 |
| webhooks | Webhook | Webhook設定 |
| api_keys | APIキー | API認証キー |

### コメント

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| comments | コメント | 投稿へのコメント |
| comment_likes | コメントいいね | コメントへのいいね |
| comment_reports | コメント報告 | コメントの不正報告 |

### その他

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| settings | 設定 | サイト設定のKey-Value保存 |
| actions | アクション | 監査ログ/アクション履歴 |
| brute | ブルートフォース対策 | レート制限用データ |
| jobs | ジョブ | バックグラウンドジョブ管理 |
| custom_theme_settings | カスタムテーマ設定 | テーマ固有の設定 |
| mentions | メンション | Webmention受信データ |
| milestones | マイルストーン | 達成マイルストーン |
| recommendations | レコメンデーション | 外部サイトの推薦 |
| recommendation_click_events | レコメンドクリックイベント | 推薦リンクのクリック履歴 |
| recommendation_subscribe_events | レコメンド購読イベント | 推薦経由の購読履歴 |
| outbox | アウトボックス | イベント送信キュー |

## 各テーブル定義

### 1. newsletters

ニュースレターの設定と外観をカスタマイズするためのテーブル。複数のニュースレターを作成し、それぞれ独自のデザイン設定を持つことができる。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| uuid | string(36) | NO | UUID (一意) |
| name | string(191) | NO | ニュースレター名 (一意) |
| description | string(2000) | YES | 説明文 |
| feedback_enabled | boolean | NO | フィードバック機能の有効化 (デフォルト: false) |
| slug | string(191) | NO | スラッグ (一意) |
| sender_name | string(191) | YES | 送信者名 |
| sender_email | string(191) | YES | 送信者メールアドレス |
| sender_reply_to | string(191) | NO | 返信先 (デフォルト: 'newsletter') |
| status | string(50) | NO | ステータス ('active', 'archived') デフォルト: 'active' |
| visibility | string(50) | NO | 表示設定 (デフォルト: 'members') |
| subscribe_on_signup | boolean | NO | サインアップ時に自動購読 (デフォルト: true) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |
| header_image | string(2000) | YES | ヘッダー画像URL |
| show_header_icon | boolean | NO | ヘッダーアイコン表示 (デフォルト: true) |
| show_header_title | boolean | NO | ヘッダータイトル表示 (デフォルト: true) |
| show_excerpt | boolean | NO | 抜粋表示 (デフォルト: false) |
| title_font_category | string(191) | NO | タイトルフォント ('serif', 'sans_serif') デフォルト: 'sans_serif' |
| title_alignment | string(191) | NO | タイトル配置 ('center', 'left') デフォルト: 'center' |
| show_feature_image | boolean | NO | フィーチャー画像表示 (デフォルト: true) |
| body_font_category | string(191) | NO | 本文フォント ('serif', 'sans_serif') デフォルト: 'sans_serif' |
| footer_content | text | YES | フッターコンテンツ |
| show_badge | boolean | NO | バッジ表示 (デフォルト: true) |
| show_header_name | boolean | NO | ヘッダー名表示 (デフォルト: true) |
| show_post_title_section | boolean | NO | 投稿タイトルセクション表示 (デフォルト: true) |
| show_comment_cta | boolean | NO | コメントCTA表示 (デフォルト: true) |
| show_subscription_details | boolean | NO | 購読詳細表示 (デフォルト: false) |
| show_latest_posts | boolean | NO | 最新投稿表示 (デフォルト: false) |
| background_color | string(50) | NO | 背景色 (デフォルト: 'light') |
| post_title_color | string(50) | YES | 投稿タイトル色 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| button_corners | string(50) | NO | ボタン角 ('square', 'rounded', 'pill') デフォルト: 'rounded' |
| button_style | string(50) | NO | ボタンスタイル ('fill', 'outline') デフォルト: 'fill' |
| title_font_weight | string(50) | NO | タイトルフォントウェイト ('normal', 'medium', 'semibold', 'bold') デフォルト: 'bold' |
| link_style | string(50) | NO | リンクスタイル ('underline', 'regular', 'bold') デフォルト: 'underline' |
| image_corners | string(50) | NO | 画像角 ('square', 'rounded') デフォルト: 'square' |
| header_background_color | string(50) | NO | ヘッダー背景色 (デフォルト: 'transparent') |
| section_title_color | string(50) | YES | セクションタイトル色 |
| divider_color | string(50) | YES | 区切り線色 |
| button_color | string(50) | YES | ボタン色 (デフォルト: 'accent') |
| link_color | string(50) | YES | リンク色 (デフォルト: 'accent') |

### 2. posts

ブログ投稿とページを管理するテーブル。Ghostのコアとなるコンテンツデータを保持する。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| uuid | string(36) | NO | UUID (インデックス) |
| title | string(2000) | NO | タイトル (最大255文字) |
| slug | string(191) | NO | スラッグ |
| mobiledoc | text(long) | YES | Mobiledoc形式のコンテンツ |
| lexical | text(long) | YES | Lexical形式のコンテンツ |
| html | text(long) | YES | HTML形式のコンテンツ |
| comment_id | string(50) | YES | コメント識別子 |
| plaintext | text(long) | YES | プレーンテキストコンテンツ |
| feature_image | string(2000) | YES | フィーチャー画像URL |
| featured | boolean | NO | 注目記事フラグ (デフォルト: false) |
| type | string(50) | NO | タイプ ('post', 'page') デフォルト: 'post' |
| status | string(50) | NO | ステータス ('published', 'draft', 'scheduled', 'sent') デフォルト: 'draft' |
| locale | string(6) | YES | ロケール (将来の機能用) |
| visibility | string(50) | NO | 表示設定 (デフォルト: 'public') |
| email_recipient_filter | text | NO | メール受信者フィルター |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 (インデックス) |
| published_at | dateTime | YES | 公開日時 (インデックス) |
| published_by | string(24) | YES | 公開者ID |
| custom_excerpt | string(2000) | YES | カスタム抜粋 (最大300文字) |
| codeinjection_head | text(65535) | YES | ヘッダーコード挿入 |
| codeinjection_foot | text(65535) | YES | フッターコード挿入 |
| custom_template | string(100) | YES | カスタムテンプレート |
| canonical_url | text(2000) | YES | カノニカルURL |
| newsletter_id | string(24) | YES | ニュースレターID (FK: newsletters.id) |
| show_title_and_feature_image | boolean | NO | タイトルとフィーチャー画像表示 (デフォルト: true) |

**インデックス**: [type, status, updated_at]
**ユニーク制約**: [slug, type]

### 3. posts_meta

投稿のOGP/Twitter Card/SEO情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (FK: posts.id, 一意) |
| og_image | string(2000) | YES | OG画像URL |
| og_title | string(300) | YES | OGタイトル |
| og_description | string(500) | YES | OG説明 |
| twitter_image | string(2000) | YES | Twitter画像URL |
| twitter_title | string(300) | YES | Twitterタイトル |
| twitter_description | string(500) | YES | Twitter説明 |
| meta_title | string(2000) | YES | メタタイトル (最大300文字) |
| meta_description | string(2000) | YES | メタ説明 (最大500文字) |
| email_subject | string(300) | YES | メール件名 |
| frontmatter | text(65535) | YES | フロントマター |
| feature_image_alt | string(2000) | YES | フィーチャー画像ALT (最大191文字) |
| feature_image_caption | text(65535) | YES | フィーチャー画像キャプション |
| email_only | boolean | NO | メール専用フラグ (デフォルト: false) |

### 4. users

スタッフユーザー(管理者・編集者など)を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | 表示名 |
| slug | string(191) | NO | スラッグ (一意) |
| password | string(60) | NO | パスワードハッシュ |
| email | string(191) | NO | メールアドレス (一意) |
| profile_image | string(2000) | YES | プロフィール画像URL |
| cover_image | string(2000) | YES | カバー画像URL |
| bio | text(65535) | YES | 自己紹介 (最大250文字) |
| website | string(2000) | YES | ウェブサイトURL |
| location | text(65535) | YES | 所在地 (最大150文字) |
| facebook | string(2000) | YES | Facebookアカウント |
| twitter | string(2000) | YES | Twitterアカウント |
| threads | string(191) | YES | Threadsアカウント |
| bluesky | string(191) | YES | Blueskyアカウント |
| mastodon | string(191) | YES | Mastodonアカウント |
| tiktok | string(191) | YES | TikTokアカウント |
| youtube | string(191) | YES | YouTubeアカウント |
| instagram | string(191) | YES | Instagramアカウント |
| linkedin | string(191) | YES | LinkedInアカウント |
| accessibility | text(65535) | YES | アクセシビリティ設定 |
| status | string(50) | NO | ステータス ('active', 'inactive', 'locked', 'warn-1'~'warn-4') デフォルト: 'active' |
| locale | string(6) | YES | ロケール (将来の機能用) |
| visibility | string(50) | NO | 表示設定 (デフォルト: 'public') |
| meta_title | string(2000) | YES | メタタイトル (最大300文字) |
| meta_description | string(2000) | YES | メタ説明 (最大500文字) |
| tour | text(65535) | YES | ツアー進捗 |
| last_seen | dateTime | YES | 最終アクセス日時 |
| comment_notifications | boolean | NO | コメント通知 (デフォルト: true) |
| free_member_signup_notification | boolean | NO | 無料メンバー登録通知 (デフォルト: true) |
| paid_subscription_started_notification | boolean | NO | 有料購読開始通知 (デフォルト: true) |
| paid_subscription_canceled_notification | boolean | NO | 有料購読キャンセル通知 (デフォルト: false) |
| mention_notifications | boolean | NO | メンション通知 (デフォルト: true) |
| recommendation_notifications | boolean | NO | レコメンデーション通知 (デフォルト: true) |
| milestone_notifications | boolean | NO | マイルストーン通知 (デフォルト: true) |
| donation_notifications | boolean | NO | 寄付通知 (デフォルト: true) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 5. posts_authors

投稿と著者の多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (FK: posts.id) |
| author_id | string(24) | NO | 著者ID (FK: users.id) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

### 6. roles

ユーザーの権限ロールを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(50) | NO | ロール名 (一意) |
| description | string(2000) | YES | 説明 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 7. roles_users

ロールとユーザーの多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| role_id | string(24) | NO | ロールID |
| user_id | string(24) | NO | ユーザーID |

### 8. permissions

操作権限を定義するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(50) | NO | パーミッション名 (一意) |
| object_type | string(50) | NO | オブジェクトタイプ |
| action_type | string(50) | NO | アクションタイプ |
| object_id | string(24) | YES | オブジェクトID |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 9. permissions_users

パーミッションとユーザーの直接割当を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| user_id | string(24) | NO | ユーザーID |
| permission_id | string(24) | NO | パーミッションID |

### 10. permissions_roles

パーミッションとロールの多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| role_id | string(24) | NO | ロールID |
| permission_id | string(24) | NO | パーミッションID |

### 11. settings

サイト設定をKey-Value形式で保存するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| group | string(50) | NO | グループ ('core', 'email', 'labs', 'members', 'portal', 'private', 'site', 'slack', 'theme', 'unsplash', 'views') デフォルト: 'core' |
| key | string(50) | NO | キー (一意) |
| value | text(65535) | YES | 値 |
| type | string(50) | NO | 型 ('array', 'string', 'number', 'boolean', 'object') |
| flags | string(50) | YES | フラグ |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 12. tags

コンテンツのタグを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | タグ名 (カンマ不可) |
| slug | string(191) | NO | スラッグ (一意) |
| description | text(65535) | YES | 説明 (最大500文字) |
| feature_image | string(2000) | YES | フィーチャー画像URL |
| parent_id | string | YES | 親タグID |
| visibility | string(50) | NO | 表示設定 ('public', 'internal') デフォルト: 'public' |
| og_image | string(2000) | YES | OG画像URL |
| og_title | string(300) | YES | OGタイトル |
| og_description | string(500) | YES | OG説明 |
| twitter_image | string(2000) | YES | Twitter画像URL |
| twitter_title | string(300) | YES | Twitterタイトル |
| twitter_description | string(500) | YES | Twitter説明 |
| meta_title | string(2000) | YES | メタタイトル (最大300文字) |
| meta_description | string(2000) | YES | メタ説明 (最大500文字) |
| codeinjection_head | text(65535) | YES | ヘッダーコード挿入 |
| codeinjection_foot | text(65535) | YES | フッターコード挿入 |
| canonical_url | string(2000) | YES | カノニカルURL |
| accent_color | string(50) | YES | アクセントカラー |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 13. posts_tags

投稿とタグの多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (FK: posts.id) |
| tag_id | string(24) | NO | タグID (FK: tags.id) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

**インデックス**: [post_id, tag_id]

### 14. invites

スタッフ招待を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| role_id | string(24) | NO | ロールID |
| status | string(50) | NO | ステータス ('pending', 'sent') デフォルト: 'pending' |
| token | string(191) | NO | トークン (一意) |
| email | string(191) | NO | メールアドレス (一意) |
| expires | bigInteger | NO | 有効期限 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 15. brute

ブルートフォース攻撃対策用のレート制限テーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| key | string(191) | NO | キー (主キー) |
| firstRequest | bigInteger | YES | 最初のリクエスト時刻 |
| lastRequest | bigInteger | YES | 最後のリクエスト時刻 |
| lifetime | bigInteger | YES | 有効期間 |
| count | integer | YES | リクエスト回数 |

### 16. sessions

ユーザーセッションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| session_id | string(32) | NO | セッションID (一意) |
| user_id | string(24) | NO | ユーザーID |
| session_data | string(2000) | NO | セッションデータ |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 17. integrations

外部サービス連携を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| type | string(50) | NO | タイプ ('internal', 'builtin', 'custom', 'core') デフォルト: 'custom' |
| name | string(191) | NO | 名前 |
| slug | string(191) | NO | スラッグ (一意) |
| icon_image | string(2000) | YES | アイコン画像URL |
| description | string(2000) | YES | 説明 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 18. webhooks

Webhook設定を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| event | string(50) | NO | イベント名 (小文字) |
| target_url | string(2000) | NO | ターゲットURL |
| name | string(191) | YES | 名前 |
| secret | string(191) | YES | シークレット |
| api_version | string(50) | NO | APIバージョン (デフォルト: 'v2') |
| integration_id | string(24) | NO | インテグレーションID (FK: integrations.id, カスケード削除) |
| last_triggered_at | dateTime | YES | 最終トリガー日時 |
| last_triggered_status | string(50) | YES | 最終トリガーステータス |
| last_triggered_error | string(50) | YES | 最終トリガーエラー |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 19. api_keys

API認証キーを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| type | string(50) | NO | タイプ ('content', 'admin') |
| secret | string(191) | NO | シークレット (一意, 26-128文字) |
| role_id | string(24) | YES | ロールID |
| integration_id | string(24) | YES | インテグレーションID |
| user_id | string(24) | YES | ユーザーID |
| last_seen_at | dateTime | YES | 最終使用日時 |
| last_seen_version | string(50) | YES | 最終使用バージョン |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 20. mobiledoc_revisions

Mobiledoc形式の投稿リビジョンを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (インデックス) |
| mobiledoc | text(long) | YES | Mobiledocコンテンツ |
| created_at_ts | bigInteger | NO | 作成タイムスタンプ |
| created_at | dateTime | NO | 作成日時 |

### 21. post_revisions

Lexical形式の投稿リビジョンを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (インデックス) |
| lexical | text(long) | YES | Lexicalコンテンツ |
| created_at_ts | bigInteger | NO | 作成タイムスタンプ |
| created_at | dateTime | NO | 作成日時 |
| author_id | string(24) | YES | 著者ID (FK: users.id) |
| title | string(2000) | YES | タイトル (最大255文字) |
| post_status | string(50) | YES | 投稿ステータス ('draft', 'published', 'scheduled', 'sent') |
| reason | string(50) | YES | 理由 |
| feature_image | string(2000) | YES | フィーチャー画像URL |
| feature_image_alt | string(2000) | YES | フィーチャー画像ALT (最大191文字) |
| feature_image_caption | text(65535) | YES | フィーチャー画像キャプション |
| custom_excerpt | string(2000) | YES | カスタム抜粋 (最大300文字) |

### 22. members

サイトの購読者/メンバーを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| uuid | string(36) | NO | UUID (一意) |
| transient_id | string(191) | NO | 一時ID (一意) |
| email | string(191) | NO | メールアドレス (一意) |
| status | string(50) | NO | ステータス ('free', 'paid', 'comped') デフォルト: 'free' |
| name | string(191) | YES | 名前 |
| expertise | string(191) | YES | 専門分野 (最大50文字) |
| note | string(2000) | YES | 管理者メモ |
| geolocation | string(2000) | YES | 位置情報 |
| enable_comment_notifications | boolean | NO | コメント通知有効 (デフォルト: true) |
| email_count | integer | NO | 受信メール数 (デフォルト: 0) |
| email_opened_count | integer | NO | 開封メール数 (デフォルト: 0) |
| email_open_rate | integer | YES | メール開封率 (インデックス) |
| email_disabled | boolean | NO | メール無効フラグ (デフォルト: false) |
| last_seen_at | dateTime | YES | 最終アクセス日時 |
| last_commented_at | dateTime | YES | 最終コメント日時 |
| commenting | text(65535) | YES | コメント設定 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

**インデックス**: [email_disabled]

### 23. products

サブスクリプションティア(商品)を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | 名前 |
| slug | string(191) | NO | スラッグ (一意) |
| active | boolean | NO | アクティブフラグ (デフォルト: true) |
| welcome_page_url | string(2000) | YES | ウェルカムページURL |
| visibility | string(50) | NO | 表示設定 ('public', 'none') デフォルト: 'none' |
| trial_days | integer | NO | トライアル日数 (デフォルト: 0) |
| description | string(191) | YES | 説明 |
| type | string(50) | NO | タイプ ('paid', 'free') デフォルト: 'paid' |
| currency | string(50) | YES | 通貨 |
| monthly_price | integer | YES | 月額料金 |
| yearly_price | integer | YES | 年額料金 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| monthly_price_id | string(24) | YES | 月額価格ID (非推奨) |
| yearly_price_id | string(24) | YES | 年額価格ID (非推奨) |

### 24. offers

割引/プロモーションオファーを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| active | boolean | NO | アクティブフラグ (デフォルト: true) |
| name | string(191) | NO | 名前 (一意) |
| code | string(191) | NO | コード (一意) |
| product_id | string(24) | NO | 商品ID (FK: products.id) |
| stripe_coupon_id | string(255) | YES | Stripeクーポン ID (一意) |
| interval | string(50) | NO | 間隔 ('month', 'year') |
| currency | string(50) | YES | 通貨 |
| discount_type | string(50) | NO | 割引タイプ ('percent', 'amount', 'trial') |
| discount_amount | integer | NO | 割引額 |
| duration | string(50) | NO | 期間 ('trial', 'once', 'repeating', 'forever') |
| duration_in_months | integer | YES | 月数 |
| portal_title | string(191) | YES | ポータルタイトル |
| portal_description | string(2000) | YES | ポータル説明 |
| redemption_type | string(50) | NO | 利用タイプ ('signup', 'retention') デフォルト: 'signup' |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 25. benefits

ティアの特典を定義するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | 名前 |
| slug | string(191) | NO | スラッグ (一意) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 26. products_benefits

商品とベネフィットの多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| product_id | string(24) | NO | 商品ID (FK: products.id, カスケード削除) |
| benefit_id | string(24) | NO | ベネフィットID (FK: benefits.id, カスケード削除) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

### 27. members_products

メンバーと商品(ティア)の多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| product_id | string(24) | NO | 商品ID (FK: products.id, カスケード削除) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |
| expiry_at | dateTime | YES | 有効期限 |

### 28. posts_products

投稿と商品(ティア)の多対多リレーションを管理するテーブル(アクセス制御用)。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (FK: posts.id, カスケード削除) |
| product_id | string(24) | NO | 商品ID (FK: products.id, カスケード削除) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

### 29. members_created_events

メンバー登録時のアトリビューション情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| created_at | dateTime | NO | 作成日時 |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| attribution_id | string(24) | YES | アトリビューションID (インデックス) |
| attribution_type | string(50) | YES | アトリビューションタイプ ('url', 'post', 'page', 'author', 'tag') |
| attribution_url | string(2000) | YES | アトリビューションURL |
| referrer_source | string(191) | YES | リファラーソース |
| referrer_medium | string(191) | YES | リファラーメディア |
| referrer_url | string(2000) | YES | リファラーURL |
| utm_source | string(191) | YES | UTMソース |
| utm_medium | string(191) | YES | UTMメディア |
| utm_campaign | string(191) | YES | UTMキャンペーン |
| utm_term | string(191) | YES | UTMターム |
| utm_content | string(191) | YES | UTMコンテンツ |
| source | string(50) | NO | ソース ('member', 'import', 'system', 'api', 'admin') |
| batch_id | string(24) | YES | バッチID |

### 30. members_cancel_events

サブスクリプションキャンセル履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| from_plan | string(255) | NO | キャンセル前のプラン |
| created_at | dateTime | NO | 作成日時 |

### 31. members_payment_events

メンバーの支払い履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| amount | integer | NO | 金額 |
| currency | string(191) | NO | 通貨 |
| source | string(50) | NO | ソース |
| created_at | dateTime | NO | 作成日時 |

### 32. members_login_events

メンバーのログイン履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| created_at | dateTime | NO | 作成日時 |

### 33. members_email_change_events

メンバーのメールアドレス変更履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| to_email | string(191) | NO | 変更後メールアドレス |
| from_email | string(191) | NO | 変更前メールアドレス |
| created_at | dateTime | NO | 作成日時 |

### 34. members_status_events

メンバーのステータス変更履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| from_status | string(50) | YES | 変更前ステータス ('free', 'paid', 'comped') |
| to_status | string(50) | YES | 変更後ステータス ('free', 'paid', 'comped') |
| created_at | dateTime | NO | 作成日時 |

### 35. members_product_events

メンバーのティア追加/削除履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| product_id | string(24) | NO | 商品ID (FK: products.id) |
| action | string(50) | YES | アクション ('added', 'removed') |
| created_at | dateTime | NO | 作成日時 |

### 36. members_paid_subscription_events

有料サブスクリプション変更履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| type | string(50) | YES | タイプ |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| subscription_id | string(24) | YES | サブスクリプションID |
| from_plan | string(255) | YES | 変更前プラン |
| to_plan | string(255) | YES | 変更後プラン |
| currency | string(191) | NO | 通貨 |
| source | string(50) | NO | ソース ('stripe') |
| mrr_delta | integer | NO | MRR変動額 |
| created_at | dateTime | NO | 作成日時 |

### 37. labels

メンバーの分類ラベルを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | 名前 (一意) |
| slug | string(191) | NO | スラッグ (一意) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 38. members_labels

メンバーとラベルの多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| label_id | string(24) | NO | ラベルID (FK: labels.id, カスケード削除) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

### 39. members_stripe_customers

メンバーとStripeカスタマーの紐付けを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| customer_id | string(255) | NO | StripeカスタマーID (一意) |
| name | string(191) | YES | 名前 |
| email | string(191) | YES | メールアドレス |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 40. subscriptions

メンバーのサブスクリプション情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| type | string(50) | NO | タイプ ('free', 'comped', 'paid') |
| status | string(50) | NO | ステータス ('active', 'expired', 'canceled') |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| tier_id | string(24) | NO | ティアID (FK: products.id) |
| cadence | string(50) | YES | 課金サイクル ('month', 'year') |
| currency | string(50) | YES | 通貨 |
| amount | integer | YES | 金額 |
| payment_provider | string(50) | YES | 決済プロバイダー (例: 'stripe') |
| payment_subscription_url | string(2000) | YES | Stripeサブスクリプションリンク |
| payment_user_url | string(2000) | YES | StripeカスタマーリンクURL |
| offer_id | string(24) | YES | オファーID (FK: offers.id) |
| expires_at | dateTime | YES | 有効期限 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 41. members_stripe_customers_subscriptions

Stripeサブスクリプションの詳細情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| customer_id | string(255) | NO | StripeカスタマーID (FK: members_stripe_customers.customer_id, カスケード削除) |
| ghost_subscription_id | string(24) | YES | GhostサブスクリプションID (FK: subscriptions.id, カスケード削除) |
| subscription_id | string(255) | NO | StripeサブスクリプションID (一意) |
| stripe_price_id | string(255) | NO | Stripe価格ID (インデックス) |
| status | string(50) | NO | ステータス |
| cancel_at_period_end | boolean | NO | 期間終了時キャンセル (デフォルト: false) |
| cancellation_reason | string(500) | YES | キャンセル理由 |
| current_period_end | dateTime | NO | 現在期間終了日 |
| start_date | dateTime | NO | 開始日 |
| default_payment_card_last4 | string(4) | YES | デフォルトカード下4桁 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| mrr | integer | NO | MRR (デフォルト: 0) |
| offer_id | string(24) | YES | オファーID (FK: offers.id) |
| discount_start | dateTime | YES | 割引開始日 |
| discount_end | dateTime | YES | 割引終了日 |
| trial_start_at | dateTime | YES | トライアル開始日 |
| trial_end_at | dateTime | YES | トライアル終了日 |
| plan_id | string(255) | NO | プランID (非推奨) |
| plan_nickname | string(50) | NO | プラン名 (非推奨) |
| plan_interval | string(50) | NO | プラン間隔 (非推奨) |
| plan_amount | integer | NO | プラン金額 (非推奨) |
| plan_currency | string(191) | NO | プラン通貨 (非推奨) |

### 42. members_subscription_created_events

サブスクリプション作成時のアトリビューション情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| created_at | dateTime | NO | 作成日時 |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| subscription_id | string(24) | NO | サブスクリプションID (FK: members_stripe_customers_subscriptions.id, カスケード削除) |
| attribution_id | string(24) | YES | アトリビューションID (インデックス) |
| attribution_type | string(50) | YES | アトリビューションタイプ ('url', 'post', 'page', 'author', 'tag') |
| attribution_url | string(2000) | YES | アトリビューションURL |
| referrer_source | string(191) | YES | リファラーソース |
| referrer_medium | string(191) | YES | リファラーメディア |
| referrer_url | string(2000) | YES | リファラーURL |
| utm_source | string(191) | YES | UTMソース |
| utm_medium | string(191) | YES | UTMメディア |
| utm_campaign | string(191) | YES | UTMキャンペーン |
| utm_term | string(191) | YES | UTMターム |
| utm_content | string(191) | YES | UTMコンテンツ |
| batch_id | string(24) | YES | バッチID |

### 43. offer_redemptions

オファー利用履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| offer_id | string(24) | NO | オファーID (FK: offers.id, カスケード削除) |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| subscription_id | string(24) | NO | サブスクリプションID (FK: members_stripe_customers_subscriptions.id, カスケード削除) |
| created_at | dateTime | NO | 作成日時 |

### 44. members_subscribe_events

ニュースレター購読/解除履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| subscribed | boolean | NO | 購読状態 (デフォルト: true) |
| created_at | dateTime | NO | 作成日時 |
| source | string(50) | YES | ソース ('member', 'import', 'system', 'api', 'admin') |
| newsletter_id | string(24) | YES | ニュースレターID (FK: newsletters.id) |

**インデックス**: [newsletter_id, created_at]

### 45. donation_payment_events

寄付支払い履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | YES | 寄付者名 |
| email | string(191) | NO | メールアドレス |
| member_id | string(24) | YES | メンバーID (FK: members.id, null削除) |
| amount | integer | NO | 金額 |
| currency | string(50) | NO | 通貨 |
| attribution_id | string(24) | YES | アトリビューションID |
| attribution_type | string(50) | YES | アトリビューションタイプ ('url', 'post', 'page', 'author', 'tag') |
| attribution_url | string(2000) | YES | アトリビューションURL |
| referrer_source | string(191) | YES | リファラーソース |
| referrer_medium | string(191) | YES | リファラーメディア |
| referrer_url | string(2000) | YES | リファラーURL |
| utm_source | string(191) | YES | UTMソース |
| utm_medium | string(191) | YES | UTMメディア |
| utm_campaign | string(191) | YES | UTMキャンペーン |
| utm_term | string(191) | YES | UTMターム |
| utm_content | string(191) | YES | UTMコンテンツ |
| created_at | dateTime | NO | 作成日時 |
| donation_message | string(255) | YES | 寄付メッセージ |

### 46. stripe_products

GhostとStripe商品のマッピングを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| product_id | string(24) | YES | Ghost商品ID (FK: products.id) |
| stripe_product_id | string(255) | NO | Stripe商品ID (一意) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 47. stripe_prices

Stripeの価格情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| stripe_price_id | string(255) | NO | Stripe価格ID (一意) |
| stripe_product_id | string(255) | NO | Stripe商品ID (FK: stripe_products.stripe_product_id) |
| active | boolean | NO | アクティブフラグ |
| nickname | string(255) | YES | ニックネーム |
| currency | string(191) | NO | 通貨 |
| amount | integer | NO | 金額 |
| type | string(50) | NO | タイプ ('recurring', 'one_time', 'donation') デフォルト: 'recurring' |
| interval | string(50) | YES | 間隔 |
| description | string(191) | YES | 説明 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 48. actions

監査ログ/アクション履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| resource_id | string(24) | YES | リソースID |
| resource_type | string(50) | NO | リソースタイプ |
| actor_id | string(24) | NO | アクター ID |
| actor_type | string(50) | NO | アクタータイプ |
| event | string(50) | NO | イベント名 |
| context | text | YES | コンテキスト情報 |
| created_at | dateTime | NO | 作成日時 |

### 49. emails

送信メールを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (インデックス, 一意) |
| uuid | string(36) | NO | UUID |
| status | string(50) | NO | ステータス ('pending', 'submitting', 'submitted', 'failed') デフォルト: 'pending' |
| recipient_filter | text | NO | 受信者フィルター |
| error | string(2000) | YES | エラーメッセージ |
| error_data | text(long) | YES | エラー詳細データ |
| email_count | integer | NO | メール数 (デフォルト: 0) |
| csd_email_count | integer | YES | CSDメール数 |
| delivered_count | integer | NO | 配信数 (デフォルト: 0) |
| opened_count | integer | NO | 開封数 (デフォルト: 0) |
| failed_count | integer | NO | 失敗数 (デフォルト: 0) |
| subject | string(300) | YES | 件名 |
| from | string(2000) | YES | 送信元 |
| reply_to | string(2000) | YES | 返信先 |
| html | text(long) | YES | HTMLコンテンツ |
| plaintext | text(long) | YES | プレーンテキストコンテンツ |
| source | text(long) | YES | ソースコンテンツ |
| source_type | string(50) | NO | ソースタイプ ('html', 'lexical', 'mobiledoc') デフォルト: 'html' |
| track_opens | boolean | NO | 開封追跡 (デフォルト: false) |
| track_clicks | boolean | NO | クリック追跡 (デフォルト: false) |
| feedback_enabled | boolean | NO | フィードバック有効 (デフォルト: false) |
| submitted_at | dateTime | NO | 送信日時 |
| newsletter_id | string(24) | YES | ニュースレターID (FK: newsletters.id) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 50. email_batches

メール送信バッチを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| email_id | string(24) | NO | メールID (FK: emails.id) |
| provider_id | string(255) | YES | プロバイダーID |
| fallback_sending_domain | boolean | NO | フォールバック送信ドメイン (デフォルト: false) |
| status | string(50) | NO | ステータス ('pending', 'submitting', 'submitted', 'failed') デフォルト: 'pending' |
| member_segment | text(2000) | YES | メンバーセグメント |
| error_status_code | integer | YES | エラーステータスコード |
| error_message | string(2000) | YES | エラーメッセージ |
| error_data | text(long) | YES | エラー詳細データ |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | NO | 更新日時 |

### 51. email_recipients

メール受信者を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| email_id | string(24) | NO | メールID (FK: emails.id) |
| member_id | string(24) | NO | メンバーID (インデックス) |
| batch_id | string(24) | NO | バッチID (FK: email_batches.id) |
| processed_at | dateTime | YES | 処理日時 |
| delivered_at | dateTime | YES | 配信日時 |
| opened_at | dateTime | YES | 開封日時 |
| failed_at | dateTime | YES | 失敗日時 |
| member_uuid | string(36) | NO | メンバーUUID |
| member_email | string(191) | NO | メンバーメールアドレス |
| member_name | string(191) | YES | メンバー名 |

**インデックス**: [email_id, member_email], [email_id, delivered_at], [email_id, opened_at], [email_id, failed_at]

### 52. email_recipient_failures

メール配信エラー情報を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| email_id | string(24) | NO | メールID (FK: emails.id) |
| member_id | string(24) | YES | メンバーID |
| email_recipient_id | string(24) | NO | メール受信者ID (FK: email_recipients.id) |
| code | integer | NO | エラーコード |
| enhanced_code | string(50) | YES | 拡張エラーコード |
| message | string(2000) | NO | エラーメッセージ |
| severity | string(50) | NO | 重大度 ('temporary', 'permanent') デフォルト: 'permanent' |
| failed_at | dateTime | NO | 失敗日時 |
| event_id | string(255) | YES | イベントID |

### 53. tokens

認証トークンを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| token | string(32) | NO | トークン (インデックス) |
| uuid | string(36) | NO | UUID (一意) |
| data | string(2000) | YES | トークンデータ |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| first_used_at | dateTime | YES | 初回使用日時 |
| used_count | integer | NO | 使用回数 (デフォルト: 0) |
| otc_used_count | integer | NO | OTC使用回数 (デフォルト: 0) |

### 54. snippets

再利用可能なコンテンツスニペットを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | 名前 (一意) |
| mobiledoc | text(long) | NO | Mobiledocコンテンツ |
| lexical | text(long) | YES | Lexicalコンテンツ |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 55. custom_theme_settings

テーマ固有の設定を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| theme | string(191) | NO | テーマ名 |
| key | string(191) | NO | キー |
| type | string(50) | NO | タイプ ('select', 'boolean', 'color', 'text', 'image') |
| value | text(65535) | YES | 値 |

### 56. members_newsletters

メンバーのニュースレター購読状況を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| newsletter_id | string(24) | NO | ニュースレターID (FK: newsletters.id, カスケード削除) |

**インデックス**: [newsletter_id, member_id]

### 57. comments

投稿へのコメントを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| post_id | string(24) | NO | 投稿ID (FK: posts.id, カスケード削除) |
| member_id | string(24) | YES | メンバーID (FK: members.id, null削除) |
| parent_id | string(24) | YES | 親コメントID (FK: comments.id, カスケード削除) |
| in_reply_to_id | string(24) | YES | 返信先コメントID (FK: comments.id, null削除) |
| status | string(50) | NO | ステータス ('published', 'hidden', 'deleted') デフォルト: 'published' |
| html | text(long) | YES | HTMLコンテンツ |
| edited_at | dateTime | YES | 編集日時 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | NO | 更新日時 |

### 58. comment_likes

コメントへのいいねを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| comment_id | string(24) | NO | コメントID (FK: comments.id, カスケード削除) |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | NO | 更新日時 |

### 59. comment_reports

コメントの不正報告を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| comment_id | string(24) | NO | コメントID (FK: comments.id, カスケード削除) |
| member_id | string(24) | YES | メンバーID (FK: members.id, null削除) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | NO | 更新日時 |

### 60. jobs

バックグラウンドジョブを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| name | string(191) | NO | ジョブ名 (一意) |
| status | string(50) | NO | ステータス ('started', 'finished', 'failed', 'queued') デフォルト: 'queued' |
| started_at | dateTime | YES | 開始日時 |
| finished_at | dateTime | YES | 終了日時 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| metadata | string(2000) | YES | メタデータ |
| queue_entry | integer | YES | キュー順序 |

### 61. redirects

URLリダイレクト設定を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| from | string(191) | NO | リダイレクト元 (インデックス) |
| to | string(2000) | NO | リダイレクト先 |
| post_id | string(24) | YES | 投稿ID (FK: posts.id, null削除) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 62. members_click_events

メンバーのリンククリック履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| redirect_id | string(24) | NO | リダイレクトID (FK: redirects.id, カスケード削除) |
| created_at | dateTime | NO | 作成日時 |

### 63. members_feedback

投稿に対するメンバーフィードバックを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| score | integer | NO | スコア (デフォルト: 0) |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| post_id | string(24) | NO | 投稿ID (FK: posts.id, カスケード削除) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 64. suppressions

メール送信抑制リストを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| email | string(191) | NO | メールアドレス (一意) |
| email_id | string(24) | YES | メールID (FK: emails.id) |
| reason | string(50) | NO | 理由 ('spam', 'bounce') |
| created_at | dateTime | NO | 作成日時 |

### 65. email_spam_complaint_events

スパム報告履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| member_id | string(24) | NO | メンバーID (FK: members.id, カスケード削除) |
| email_id | string(24) | NO | メールID (FK: emails.id) |
| email_address | string(191) | NO | メールアドレス |
| created_at | dateTime | NO | 作成日時 |

**ユニーク制約**: [email_id, member_id]

### 66. mentions

Webmention受信データを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| source | string(2000) | NO | ソースURL |
| source_title | string(2000) | YES | ソースタイトル |
| source_site_title | string(2000) | YES | ソースサイトタイトル |
| source_excerpt | string(2000) | YES | ソース抜粋 |
| source_author | string(2000) | YES | ソース著者 |
| source_featured_image | string(2000) | YES | ソースフィーチャー画像 |
| source_favicon | string(2000) | YES | ソースファビコン |
| target | string(2000) | NO | ターゲットURL |
| resource_id | string(24) | YES | リソースID |
| resource_type | string(50) | YES | リソースタイプ |
| created_at | dateTime | NO | 作成日時 |
| payload | text(65535) | YES | ペイロード |
| deleted | boolean | NO | 削除フラグ (デフォルト: false) |
| verified | boolean | NO | 検証済みフラグ (デフォルト: false) |

### 67. milestones

達成マイルストーンを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| type | string(24) | NO | タイプ |
| value | integer | NO | 値 |
| currency | string(24) | YES | 通貨 |
| created_at | dateTime | NO | 作成日時 |
| email_sent_at | dateTime | YES | メール送信日時 |

### 68. collections

投稿のコレクション/グループを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| title | string(191) | NO | タイトル |
| slug | string(191) | NO | スラッグ (一意) |
| description | string(2000) | YES | 説明 |
| type | string(50) | NO | タイプ |
| filter | text | YES | フィルター |
| feature_image | string(2000) | YES | フィーチャー画像URL |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 69. collections_posts

コレクションと投稿の多対多リレーションを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| collection_id | string(24) | NO | コレクションID (FK: collections.id, カスケード削除) |
| post_id | string(24) | NO | 投稿ID (FK: posts.id, カスケード削除) |
| sort_order | integer | NO | 表示順序 (デフォルト: 0) |

### 70. recommendations

外部サイトの推薦を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| url | string(2000) | NO | URL |
| title | string(2000) | NO | タイトル |
| excerpt | string(2000) | YES | 抜粋 |
| featured_image | string(2000) | YES | フィーチャー画像URL |
| favicon | string(2000) | YES | ファビコンURL |
| description | string(2000) | YES | 説明 |
| one_click_subscribe | boolean | NO | ワンクリック購読 (デフォルト: false) |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

### 71. recommendation_click_events

推薦リンクのクリック履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| recommendation_id | string(24) | NO | レコメンデーションID (FK: recommendations.id, カスケード削除) |
| member_id | string(24) | YES | メンバーID (FK: members.id, null削除) |
| created_at | dateTime | NO | 作成日時 |

### 72. recommendation_subscribe_events

推薦経由の購読履歴を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| recommendation_id | string(24) | NO | レコメンデーションID (FK: recommendations.id, カスケード削除) |
| member_id | string(24) | YES | メンバーID (FK: members.id, null削除) |
| created_at | dateTime | NO | 作成日時 |

### 73. outbox

イベント送信キューを管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| event_type | string(50) | NO | イベントタイプ |
| status | string(50) | NO | ステータス (デフォルト: 'pending') |
| payload | text(65535) | NO | ペイロード |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |
| retry_count | integer | NO | リトライ回数 (デフォルト: 0) |
| last_retry_at | dateTime | YES | 最終リトライ日時 |
| message | string(2000) | YES | メッセージ |

**インデックス**: [event_type, status, created_at]

### 74. automated_emails

自動配信メール設定を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| status | string(50) | NO | ステータス ('active', 'inactive') デフォルト: 'inactive' |
| name | string(191) | NO | 名前 (一意) |
| slug | string(191) | NO | スラッグ (一意) |
| subject | string(300) | NO | 件名 |
| lexical | text(long) | YES | Lexicalコンテンツ |
| sender_name | string(191) | YES | 送信者名 |
| sender_email | string(191) | YES | 送信者メールアドレス |
| sender_reply_to | string(191) | YES | 返信先メールアドレス |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

**インデックス**: [slug], [status]

### 75. automated_email_recipients

自動メール受信者を管理するテーブル。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| id | string(24) | NO | 主キー |
| automated_email_id | string(24) | NO | 自動メールID (FK: automated_emails.id) |
| member_id | string(24) | NO | メンバーID (インデックス) |
| member_uuid | string(36) | NO | メンバーUUID |
| member_email | string(191) | NO | メンバーメールアドレス |
| member_name | string(191) | YES | メンバー名 |
| created_at | dateTime | NO | 作成日時 |
| updated_at | dateTime | YES | 更新日時 |

## 備考

### ID形式
- すべてのプライマリキーは24文字の文字列(ObjectId形式)を使用
- UUIDカラムは36文字の標準UUID形式

### 文字列カラムサイズのガイドライン
- Small strings: 50文字
- Medium strings: 191文字 (MySQLのインデックス制限を考慮)
- Large strings: 2000文字
- Text: 65535文字 (64 KiB)
- Long text: 1,000,000,000文字

### 外部キー制約
- `cascadeDelete: true`: 親レコード削除時に子レコードも削除
- `setNullDelete: true`: 親レコード削除時に子レコードの参照をNULLに設定

### 非推奨フィールド
- `products.monthly_price_id`, `products.yearly_price_id`: 将来削除予定
- `members_stripe_customers_subscriptions.plan_*`: `stripe_prices`テーブルに移行済み

### バリデーション
- メールアドレスフィールドには`isEmail`バリデーション適用
- URLフィールドには`isEmptyOrURL`バリデーション適用
- 列挙型フィールドには`isIn`バリデーションで許可値を制限
