# 機能設計書 43-招待機能

## 概要

本ドキュメントは、Ghost CMSにおける招待機能の設計仕様を記載したものである。

### 本機能の処理概要

招待機能は、既存のサイト管理者が新規スタッフユーザーをGhostサイトに招待するための機能である。招待メールを送信し、受信者が招待リンクをクリックすることで、新規スタッフアカウントを作成できる。招待には有効期限があり、セキュアなトークンベースの認証を使用する。

**業務上の目的・背景**：Ghost CMSは複数のスタッフによる共同運営を想定しており、新規メンバーの追加プロセスを安全かつ効率的に行う必要がある。招待機能により、メールアドレスの所有確認とロール割り当てを同時に行うことで、不正アカウント作成を防止しつつ、スムーズなオンボーディングを実現する。

**機能の利用シーン**：
- サイト管理者が新しい編集者・著者を招待する場面
- チーム拡大時に複数メンバーを一括招待する場面
- 外部ライターをContributorとして招待する場面
- API経由で自動化されたスタッフ招待を行う場面

**主要な処理内容**：
1. 招待の作成（Add）：招待データとセキュアトークンの生成
2. 招待メールの送信：トークン付きURLを含むメール送信
3. 招待一覧の取得（Browse）：保留中の招待を一覧表示
4. 招待の取り消し（Destroy）：未使用の招待を削除
5. 招待の使用：受信者がトークンを使用してアカウント作成

**関連システム・外部連携**：
- メールサービス（Mail Service）との連携
- ロール管理（Role Model）との連携
- ユーザー管理（User Model）との連携
- 制限管理サービス（Limit Service）によるスタッフ数制限

**権限による制御**：
- Owner/Administratorはすべてのロール（Owner以外）を招待可能
- Editor/Super EditorはAuthor/Contributorのみ招待可能
- APIキー経由ではOwnerロールの招待は禁止
- 招待可能ロールは招待者のロールに応じて制限

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | サインアップ画面 | 主機能 | 招待トークンを使用した新規スタッフユーザーの登録処理 |
| 3 | サインアップ画面 | 補助機能 | 新規ユーザーの作成とロール割り当て |
| 36 | ユーザー・権限設定 | 補助機能 | 新規スタッフの招待 |

## 機能種別

CRUD操作 / メール送信 / トークン認証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| email | string | Yes | 招待先メールアドレス | 有効なメール形式 |
| role_id | string | Yes | 割り当てるロールのID | 有効なロールID |
| include | string | No | 関連データの取得指定 | なし |
| page | integer | No | ページ番号 | 正の整数 |
| limit | integer | No | 取得件数 | 正の整数 |
| filter | string | No | フィルター条件 | NQL形式 |
| order | string | No | ソート順 | カラム名 ASC/DESC |

### 入力データソース

- 管理画面の招待フォーム
- Admin APIエンドポイント（POST /ghost/api/admin/invites/）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| id | string | 招待の一意識別子（24文字） |
| email | string | 招待先メールアドレス |
| role_id | string | 割り当てられるロールのID |
| status | string | 招待ステータス（pending/sent） |
| expires | dateTime | 有効期限（作成から1週間） |
| created_at | dateTime | 作成日時 |
| updated_at | dateTime | 更新日時 |

### 出力先

- APIレスポンス（JSON形式）
- 招待メール（HTML/テキスト形式）

## 処理フロー

### 処理シーケンス

```
1. 招待リクエスト受信
   └─ POST /ghost/api/admin/invites/ が呼び出される
2. 権限チェック
   └─ 招待者のロールで招待可能なロールか確認
3. スタッフ制限チェック
   └─ limitServiceでスタッフ数上限を確認
4. 既存招待の削除
   └─ 同一メールの既存招待を削除
5. 招待データ作成
   ├─ セキュアトークン生成
   ├─ 有効期限設定（1週間）
   └─ DBに招待レコード作成
6. 招待メール送信
   ├─ メールテンプレート生成
   ├─ 招待リンクURL生成
   └─ メール送信
7. ステータス更新
   └─ status を 'sent' に更新
8. レスポンス返却
   └─ 招待データを返却
```

### フローチャート

```mermaid
flowchart TD
    A[招待リクエスト受信] --> B{認証済み?}
    B -->|No| C[401 Unauthorized]
    B -->|Yes| D{ロール割り当て権限あり?}
    D -->|No| E[403 Forbidden]
    D -->|Yes| F{Ownerロールへの招待?}
    F -->|Yes| E
    F -->|No| G{スタッフ数上限内?}
    G -->|No| H[HostLimitError]
    G -->|Yes| I[既存招待削除]
    I --> J[トークン生成]
    J --> K[招待レコード作成]
    K --> L[メールテンプレート生成]
    L --> M[招待メール送信]
    M --> N{送信成功?}
    N -->|Yes| O[status='sent'に更新]
    N -->|No| P[EmailError]
    O --> Q[レスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-001 | 招待有効期限 | 招待は作成から1週間で有効期限切れ | 全招待 |
| BR-43-002 | Owner招待禁止 | Ownerロールへの招待は不可 | 全招待 |
| BR-43-003 | ロール階層制限 | 招待者は自分以下のロールのみ招待可能 | ロール割り当て時 |
| BR-43-004 | 重複招待防止 | 同一メールへの新規招待時は既存招待を削除 | 招待作成時 |
| BR-43-005 | スタッフ数制限 | プラン制限によりスタッフ数を制限（Contributorは除外） | 招待作成時 |
| BR-43-006 | APIキー制限 | APIキー経由での招待可能ロールは制限 | APIキー使用時 |

### 計算ロジック

招待トークン生成:
```javascript
// security.tokens.generateFromEmailを使用
token = security.tokens.generateFromEmail({
    email: data.email,
    expires: data.expires,
    secret: settingsCache.get('db_hash')
});

// 有効期限は現在時刻から1週間
expires = moment().add(1, 'week').valueOf();
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 招待作成 | invites | INSERT | 招待レコードの作成 |
| 既存招待確認 | invites | SELECT | 同一メールの招待確認 |
| 既存招待削除 | invites | DELETE | 同一メールの招待削除 |
| ステータス更新 | invites | UPDATE | status を 'sent' に更新 |
| 招待一覧取得 | invites | SELECT | ページング付き一覧取得 |
| 招待削除 | invites | DELETE | 招待の取り消し |
| ロール確認 | roles | SELECT | 招待対象ロールの確認 |

### テーブル別操作詳細

#### invites

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | id, email, role_id, status, expires, token, created_at, updated_at | 新規招待データ | tokenはJSONには含まれない |
| UPDATE | status, updated_at | status='sent' | メール送信成功時 |
| DELETE | - | emailで検索 | 既存招待削除時 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証なしでアクセス | 認証情報を付与 |
| 403 | NoPermissionError | ロール割り当て権限なし | 適切な権限を持つユーザーで操作 |
| 403 | NoPermissionError | Owner招待を試行 | Owner以外のロールを指定 |
| 404 | NotFoundError | 指定ロールが存在しない | 有効なロールIDを指定 |
| 422 | EmailError | メール送信失敗 | メール設定を確認 |
| 429 | HostLimitError | スタッフ数上限超過 | プランをアップグレード |

### リトライ仕様

- メール送信失敗時はエラーをログに記録し、招待は作成されるが status は 'pending' のまま
- 招待メールの再送信は手動で行う

## トランザクション仕様

- 招待作成とメール送信は個別トランザクション
- メール送信失敗時も招待レコードは保持される
- ステータス更新は独立したトランザクション

## パフォーマンス要件

- 招待作成: 500ms以内（メール送信を含む）
- 招待一覧取得: 100ms以内

## セキュリティ考慮事項

- 招待トークンはdb_hashをシークレットとして生成
- トークンはAPIレスポンスには含まれない（toJSONで除外）
- 招待メールのURLはHTTPS必須
- 有効期限切れトークンは使用不可

## 備考

- 招待メールのテンプレートは invite-user または invite-user-by-api-key
- 招待者の名前がない場合（APIキー経由）は異なるテンプレートを使用
- actionsCollectCRUD: trueにより操作履歴が記録される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、招待のデータモデルとスキーマを理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | invite.js | `ghost/core/core/server/models/invite.js` | Inviteモデルの定義 |

**読解のコツ**:
- tableName: 'invites' でテーブル名を確認
- toJSON関数で token が除外されている点に注目
- permissible関数でロール階層の権限チェックロジックを確認

**主要処理フロー**:
- **21-33行目**: モデル定義とtoJSON（tokenを除外）
- **35-37行目**: orderDefaultOptionsは空オブジェクト
- **39-55行目**: add関数でトークンと有効期限を設定
- **47-52行目**: security.tokens.generateFromEmailでトークン生成
- **57-118行目**: permissible関数で招待の権限チェック

#### Step 2: APIエンドポイントを理解する

招待のAPIコントローラーを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | invites.js | `ghost/core/core/server/api/endpoints/invites.js` | APIコントローラー |

**主要処理フロー**:
1. **17-39行目**: browseアクションでfindPageを使用したページング
2. **41-67行目**: readアクションで単一招待の取得
3. **70-88行目**: destroyアクションで招待の削除
4. **90-127行目**: addアクションで招待の作成
5. **112-114行目**: permissions.unsafeAttrsでrole_idを指定
6. **116-126行目**: invites.add呼び出しでサービス層に委譲

#### Step 3: 招待サービスを理解する

招待メール送信のビジネスロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | invites.js | `ghost/core/core/server/services/invites/invites.js` | 招待サービス実装 |

**主要処理フロー**:
- **14-21行目**: コンストラクタで依存関係を注入
- **22-104行目**: add関数のメインロジック
- **26-33行目**: 既存招待を検索して削除
- **34-36行目**: 新規招待を作成
- **40-61行目**: 招待メールのデータ準備
- **47行目**: security.url.encodeBase64でトークンをエンコード
- **51-60行目**: 招待者の名前有無でテンプレートを分岐
- **63-83行目**: メールペイロードの構築
- **82行目**: api.mail.send でメール送信
- **84-91行目**: ステータスを 'sent' に更新

#### Step 4: 権限チェックの詳細を理解する

招待時の権限チェックロジックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | invite.js | `ghost/core/core/server/models/invite.js` | permissible関数 |
| 4-2 | role-utils.js | `ghost/core/core/server/models/role-utils.js` | setIsRoles関数 |

**主要処理フロー**:
- **57-68行目**: add以外のアクションの権限チェック
- **71-78行目**: ロールが見つからない場合はNotFoundError
- **80-84行目**: Ownerへの招待は禁止
- **86-90行目**: スタッフ制限チェック（Contributor以外）
- **92-102行目**: 招待可能ロールの決定ロジック
- **104-108行目**: 招待対象ロールが許可リストにあるか確認

### プログラム呼び出し階層図

```
API Request (POST /ghost/api/admin/invites/)
    │
    ├─ invites.js (controller)
    │      │
    │      ├─ permissions check (via api-framework)
    │      │      └─ Invite.permissible()
    │      │             ├─ Role.findOne() - 招待対象ロール確認
    │      │             ├─ limitService.errorIfWouldGoOverLimit()
    │      │             └─ setIsRoles() - 招待者ロール確認
    │      │
    │      └─ invites.add() (service)
    │             │
    │             ├─ InviteModel.findOne() - 既存招待確認
    │             ├─ existingInvite.destroy() - 既存招待削除
    │             ├─ InviteModel.add() - 招待作成
    │             │      └─ security.tokens.generateFromEmail()
    │             ├─ mailService.utils.generateContent()
    │             ├─ api.mail.send() - メール送信
    │             └─ InviteModel.edit() - status='sent'
    │
    └─ Response (JSON)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

email ────────────────────▶ ┌──────────────────────────┐
role_id ─────────────────▶  │   invites.js (API)       │
                            │      │                    │
                            │      ▼                    │
                            │   権限チェック             │
                            │      │                    │
                            │      ▼                    │
                            │   invites.js (Service)   │
                            │      │                    │
                            │      ├─ トークン生成        │
                            │      ├─ DB保存             │
                            │      └─ メール送信          │
                            └──────────────────────────┘
                                       │
                                       ▼
                            ┌──────────────────────────┐
                            │  invitesテーブル          │ ───▶ JSON Response
                            │  (id, email, token, ...)  │
                            └──────────────────────────┘
                                       │
                                       ▼
                            ┌──────────────────────────┐
                            │  招待メール               │ ───▶ 受信者のメールボックス
                            │  (招待リンク付き)          │
                            └──────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| invite.js | `ghost/core/core/server/models/invite.js` | ソース | Inviteモデル定義 |
| invites.js | `ghost/core/core/server/api/endpoints/invites.js` | ソース | APIエンドポイント |
| invites.js | `ghost/core/core/server/services/invites/invites.js` | ソース | 招待サービス |
| index.js | `ghost/core/core/server/services/invites/index.js` | ソース | サービスエクスポート |
| role-utils.js | `ghost/core/core/server/models/role-utils.js` | ソース | ロール判定ユーティリティ |
| limits.js | `ghost/core/core/server/services/limits.js` | ソース | 制限管理サービス |
| invite-user.hbs | `ghost/core/core/server/views/emails/invite-user.hbs` | テンプレート | 招待メールテンプレート |
