# 機能設計書 45-Admin API

## 概要

本ドキュメントは、Ghost CMSにおけるAdmin APIの設計仕様を記載したものである。

### 本機能の処理概要

Admin APIは、Ghost CMSの管理操作用のREST APIである。Content APIと異なり、コンテンツの作成・編集・削除、設定変更、ユーザー管理など、すべての管理操作を実行できる。Admin APIキーまたはスタッフユーザーセッションによる認証が必要で、操作ごとにロールベースのアクセス制御（RBAC）が適用される。

**業務上の目的・背景**：Ghost CMSの管理機能をプログラマティックに利用するためのAPIを提供する。外部アプリケーションからのコンテンツ管理自動化、カスタム管理ツールの構築、CI/CDパイプラインでのコンテンツ更新など、多様なユースケースに対応する。Ghost Admin管理画面自体もこのAPIを利用している。

**機能の利用シーン**：
- 外部CMSからのコンテンツ移行
- 自動コンテンツ公開システムの構築
- カスタムワークフローの実装
- バックアップ・リストア処理
- サードパーティツールとの連携
- 管理画面（Ghost Admin）の全操作

**主要な処理内容**：
1. コンテンツ管理: 記事、ページ、タグ、スニペットのCRUD操作
2. メディア管理: 画像、ファイル、メディアのアップロード
3. ユーザー管理: スタッフユーザーのCRUD操作、招待
4. メンバー管理: メンバーのCRUD操作、インポート/エクスポート
5. 設定管理: サイト設定、テーマ設定、ニュースレター設定
6. インテグレーション管理: APIキー、Webhook設定
7. その他: コメント管理、統計取得、データベース操作

**関連システム・外部連携**：
- Ghost Admin（Ember.jsベース管理画面）
- Admin X Settings（React管理コンポーネント）
- 外部自動化ツール
- CI/CDシステム

**権限による制御**：
- Admin APIキーまたはスタッフセッションによる認証
- ロールベースのアクセス制御（Owner/Administrator/Editor/Author/Contributor）
- 各エンドポイントで個別の権限チェック
- Staff APIキー（ユーザー紐付き）の場合はユーザー権限を使用

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 全管理画面 | - | 主機能 | 管理画面のすべての操作はAdmin API経由 |
| 58 | インテグレーション設定 | 主画面 | APIキーの発行・管理 |

## 機能種別

REST API / CRUD操作 / 認証・認可

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Authorization | Header | Yes | Admin APIキーまたはセッションCookie | 有効な認証情報 |
| include | string | No | 関連データの取得指定 | エンドポイントに依存 |
| filter | string | No | フィルター条件 | NQL形式 |
| limit | integer | No | 取得件数 | エンドポイントに依存 |
| page | integer | No | ページ番号 | 正の整数 |
| order | string | No | ソート順 | カラム名 ASC/DESC |
| fields | string | No | 取得フィールド | カンマ区切り |

### 入力データソース

- HTTPリクエスト（GET/POST/PUT/DELETE）
- リクエストボディ（JSON形式）
- クエリパラメータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| {resource} | array/object | 操作対象リソースのデータ |
| meta | object | ページング情報（一覧取得時） |
| errors | array | エラー情報（エラー時） |

### 出力先

- HTTPレスポンス（JSON形式）
- Content-Type: application/json

## 処理フロー

### 処理シーケンス

```
1. APIリクエスト受信
   └─ {METHOD} /ghost/api/admin/{resource}/
2. 認証チェック
   ├─ Admin APIキーの検証
   └─ またはセッションCookieの検証
3. 権限チェック
   └─ canThis()によるRBAC検証
4. バリデーション
   └─ 入力パラメータ・リクエストボディの検証
5. ビジネスロジック実行
   └─ サービス層での処理
6. データベース操作
   └─ モデル経由でのDB操作
7. レスポンス生成
   └─ シリアライザーでJSON変換
8. レスポンス返却
   └─ HTTPレスポンス
```

### フローチャート

```mermaid
flowchart TD
    A[APIリクエスト受信] --> B{認証方式}
    B -->|APIキー| C[APIキー検証]
    B -->|Session| D[セッション検証]
    C -->|Invalid| E[401 Unauthorized]
    D -->|Invalid| E
    C -->|Valid| F[権限チェック]
    D -->|Valid| F
    F -->|Denied| G[403 Forbidden]
    F -->|Allowed| H[バリデーション]
    H -->|Invalid| I[422 Validation Error]
    H -->|Valid| J[ビジネスロジック実行]
    J --> K[DB操作]
    K --> L[レスポンス生成]
    L --> M[レスポンス返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-45-001 | 認証必須 | すべてのAdmin APIリクエストに認証が必要 | 全リクエスト |
| BR-45-002 | RBAC適用 | 各操作にロールベースのアクセス制御を適用 | 全操作 |
| BR-45-003 | Admin APIキー制限 | Admin APIキーはOwnerロールの操作を一部制限 | APIキー認証時 |
| BR-45-004 | Staff APIキー | ユーザー紐付きAPIキーはユーザー権限を使用 | Staff APIキー使用時 |
| BR-45-005 | キャッシュ無効化 | データ変更時は関連キャッシュを無効化 | CUD操作時 |

### 計算ロジック

Admin APIキー認証:
```javascript
// Admin APIキーはJWT形式
// ヘッダー: Authorization: Ghost {token}
// トークン: {admin_key_id}:{secret_signature}

// 署名検証
const [id, secret] = token.split(':');
const apiKey = await ApiKey.findOne({id});
const expectedSecret = createHmac('sha256', apiKey.secret)
    .update(Buffer.from(id, 'hex'))
    .digest('hex');
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 記事管理 | posts, posts_tags, posts_authors | CRUD | 記事の全操作 |
| ユーザー管理 | users, roles_users | CRUD | ユーザーの全操作 |
| 設定管理 | settings | READ/UPDATE | 設定の参照・更新 |
| インテグレーション | integrations, api_keys, webhooks | CRUD | 連携設定の全操作 |
| メンバー管理 | members, members_labels, subscriptions | CRUD | メンバーの全操作 |

### テーブル別操作詳細

各エンドポイントに依存するため、個別の機能設計書を参照。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 401 | Unauthorized | 認証失敗 | 有効な認証情報を使用 |
| 403 | NoPermissionError | 権限不足 | 適切なロールを持つユーザーで操作 |
| 404 | NotFoundError | リソースが存在しない | 有効なIDを指定 |
| 422 | ValidationError | 入力値が無効 | 入力値を確認 |
| 429 | TooManyRequestsError | レートリミット超過 | 時間をおいて再試行 |
| 500 | InternalServerError | サーバーエラー | サポートに連絡 |

### リトライ仕様

- 5xxエラー時はエクスポネンシャルバックオフでリトライ推奨
- 429エラー時はRetry-Afterヘッダーに従う

## トランザクション仕様

- 単一リソースの操作はトランザクション内で実行
- 関連リソースの操作はBookshelf ORMのトランザクションサポートを使用
- エラー時は自動ロールバック

## パフォーマンス要件

- 単一リソース操作: 500ms以内
- 一覧取得: 1000ms以内
- バルク操作: タイムアウト設定に依存

## セキュリティ考慮事項

- Admin APIキーは秘密鍵として厳重に管理
- HTTPSの使用を必須
- APIキーローテーションを定期的に実施
- 監査ログ（Actions）でAPI操作を記録
- レートリミットによるブルートフォース防止

## 備考

- Admin APIキーはインテグレーション設定で発行
- 公式SDKライブラリ（@tryghost/admin-api）の使用を推奨
- API仕様はバージョン管理され、破壊的変更は事前に通知

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: エンドポイント構造を理解する

まず、Admin APIのエンドポイント一覧と構造を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | index.js | `ghost/core/core/server/api/endpoints/index.js` | 全エンドポイントのエクスポート |

**読解のコツ**: Admin APIは60以上のエンドポイントファイルで構成されている。主要なエンドポイントを優先的に確認する。

#### Step 2: 主要エンドポイントの構造を理解する

代表的なエンドポイント（posts.js）を詳細に確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | posts.js | `ghost/core/core/server/api/endpoints/posts.js` | 記事管理APIの完全な実装 |

**主要処理フロー**:
1. **コントローラー定義**: docName, browse, read, add, edit, destroy
2. **headers**: cacheInvalidate設定
3. **options/data**: パラメータ定義
4. **validation**: バリデーションルール
5. **permissions**: 権限チェック設定
6. **query**: 実際のデータ操作

#### Step 3: 認証機構を理解する

Admin API認証の仕組みを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | authentication.js | `ghost/core/core/server/api/endpoints/authentication.js` | 認証エンドポイント |
| 3-2 | session.js | `ghost/core/core/server/api/endpoints/session.js` | セッション管理 |

#### Step 4: 権限チェック機構を理解する

APIフレームワークでの権限チェックを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | permissions.js | `ghost/core/core/server/services/permissions/index.js` | 権限サービス |
| 4-2 | can-this.js | `ghost/core/core/server/services/permissions/can-this.js` | 権限チェック実装 |

### プログラム呼び出し階層図

```
API Request
    │
    ├─ Middleware Stack
    │      ├─ Authentication (API Key / Session)
    │      ├─ Error Handling
    │      └─ CORS
    │
    ├─ API Framework (@tryghost/api-framework)
    │      ├─ Input Validation
    │      ├─ Permission Check (canThis)
    │      └─ Serialization
    │
    ├─ Controller (endpoints/*.js)
    │      └─ query() function
    │
    ├─ Service Layer (services/*.js)
    │      └─ Business Logic
    │
    ├─ Model Layer (models/*.js)
    │      └─ Bookshelf ORM
    │
    └─ Database
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Authorization ───────────▶ ┌──────────────────────────┐
(APIキー/Session)          │   Authentication         │
                           │          │               │
リクエストBody ───────────▶ │          ▼               │
                           │   Permission Check       │
クエリパラメータ ──────────▶ │          │               │
                           │          ▼               │
                           │   Validation             │
                           │          │               │
                           │          ▼               │
                           │   Controller.query()     │
                           │          │               │
                           │          ▼               │
                           │   Service Layer          │
                           │          │               │
                           │          ▼               │
                           │   Model Layer            │
                           │          │               │
                           │          ▼               │
                           │   Database               │
                           └──────────────────────────┘
                                       │
                                       ▼
                           ┌──────────────────────────┐
                           │  JSON Response           │
                           │  {                       │
                           │    {resource}: [...],    │
                           │    meta: { pagination }  │
                           │  }                       │
                           └──────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| posts.js | `ghost/core/core/server/api/endpoints/posts.js` | ソース | 記事管理API |
| pages.js | `ghost/core/core/server/api/endpoints/pages.js` | ソース | ページ管理API |
| tags.js | `ghost/core/core/server/api/endpoints/tags.js` | ソース | タグ管理API |
| users.js | `ghost/core/core/server/api/endpoints/users.js` | ソース | ユーザー管理API |
| members.js | `ghost/core/core/server/api/endpoints/members.js` | ソース | メンバー管理API |
| settings.js | `ghost/core/core/server/api/endpoints/settings.js` | ソース | 設定管理API |
| integrations.js | `ghost/core/core/server/api/endpoints/integrations.js` | ソース | インテグレーションAPI |
| webhooks.js | `ghost/core/core/server/api/endpoints/webhooks.js` | ソース | Webhook管理API |
| images.js | `ghost/core/core/server/api/endpoints/images.js` | ソース | 画像アップロードAPI |
| media.js | `ghost/core/core/server/api/endpoints/media.js` | ソース | メディアアップロードAPI |
| authentication.js | `ghost/core/core/server/api/endpoints/authentication.js` | ソース | 認証API |
| session.js | `ghost/core/core/server/api/endpoints/session.js` | ソース | セッションAPI |
| db.js | `ghost/core/core/server/api/endpoints/db.js` | ソース | データベースAPI |
| themes.js | `ghost/core/core/server/api/endpoints/themes.js` | ソース | テーマ管理API |
| newsletters.js | `ghost/core/core/server/api/endpoints/newsletters.js` | ソース | ニュースレター管理API |
| index.js | `ghost/core/core/server/api/endpoints/index.js` | ソース | エンドポイント集約 |
