# 通知設計書 19-パスワードリセットメール

## 概要

本ドキュメントは、Ghost CMSにおけるパスワードリセットメール機能の設計仕様を記載したものである。

### 本通知の処理概要

パスワードリセットメールは、管理者がパスワードを忘れた際にリセット用のリンクをメールで送信する機能である。このメールにはセキュアなリセットトークンが含まれており、受信者はそのリンクから新しいパスワードを設定できる。

**業務上の目的・背景**：パスワードの忘却は一般的な問題であり、セルフサービスでのパスワードリセット機能は必須である。セキュリティを保ちながら、正当なユーザーがアカウントへのアクセスを回復できる仕組みを提供する。リセットトークンは有効期限付きで、ユーザーの現在のパスワードハッシュに基づいて生成されるため、一度使用すると無効になる。

**通知の送信タイミング**：管理者がログイン画面から「パスワードを忘れた」リンクをクリックし、メールアドレスを入力した際に、Settings APIを通じてトークンが生成され、メールが送信される。

**通知の受信者**：パスワードリセットを要求したメールアドレスの所有者（管理者ユーザー）。

**通知内容の概要**：パスワードリセット用のURL（リセットトークンをBase64エンコードしたものを含む）が含まれる。リンクは1日で期限切れとなる。

**期待されるアクション**：受信者はリセットリンクをクリックし、新しいパスワードを設定することが期待される。リンクの有効期限は24時間。

## 通知種別

メール通知（管理者向け）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（API呼び出し後にメール送信） |
| 優先度 | 高 |
| リトライ | なし |

### 送信先決定ロジック

パスワードリセットを要求したメールアドレス（`data.email`）に直接送信する。

## 通知テンプレート

### メール通知の場合

| 項目 | 内容 |
|-----|------|
| 送信元アドレス | システムデフォルト |
| 送信元名称 | システム |
| 件名 | `Reset Password` |
| 形式 | HTML/テキスト両対応 |

### 本文テンプレート

`reset-password`テンプレートを使用（mail.utils.generateContent経由）。

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| なし | - | - | 添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| resetUrl | パスワードリセットURL | urlUtils.urlJoin(adminUrl, 'reset', Base64(token), '/') | Yes |
| recipientEmail | 受信者のメールアドレス | data.email | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | sendResetNotification() | ユーザーが存在する | パスワードリセットAPIが呼び出された際 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ユーザーが存在しない | 指定されたメールアドレスのユーザーがいない場合（NotFoundErrorをスロー） |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[パスワードリセット要求] --> B[generateToken呼び出し]
    B --> C[db_hash設定を取得]
    C --> D[メールアドレスでユーザー検索]
    D --> E{ユーザー存在?}
    E -->|No| F[NotFoundError]
    E -->|Yes| G[リセットトークン生成]
    G --> H[sendResetNotification呼び出し]
    H --> I[resetUrlを生成]
    I --> J[テンプレートでコンテンツ生成]
    J --> K[mailAPI.send実行]
    K --> L[完了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| users | ユーザー検索とパスワードハッシュ取得 | email, password |
| settings | db_hash取得 | トークン生成に使用 |

### テーブル別参照項目詳細

#### users

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| email | ユーザー検索 | 要求されたメールアドレス |
| password | トークン生成 | パスワードハッシュをトークンに含める |

#### settings

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| db_hash | トークン生成 | サイト固有のハッシュ値 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| なし | - | パスワードリセットメール送信時は更新なし |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| NotFoundError | ユーザーが存在しない | エラーをスロー（セキュリティ上、存在有無は明かさない設計も検討要） |
| 送信失敗 | SMTP接続エラー等 | エラーをスロー |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | ブルートフォース保護あり（10回/email+expires） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（API呼び出し時に即時送信）

## セキュリティ考慮事項

- リセットトークンは`security.tokens.resetToken.generateHash`で生成
- トークンには有効期限（1日）、メールアドレス、db_hash、現在のパスワードハッシュが含まれる
- パスワードが変更されるとトークンは無効になる（パスワードハッシュがトークン検証に使用される）
- ブルートフォース保護：同一email+expiresの組み合わせで10回以上の試行は拒否
- トークンはBase64エンコードしてURLに含まれる

## 備考

### トークン生成の仕組み

```javascript
token = security.tokens.resetToken.generateHash({
    expires: moment().add(1, 'days').valueOf(),
    email: email,
    dbHash: dbHash,
    password: user.get('password')
});
```

### トークン検証

パスワードリセット実行時に以下を検証：
1. トークンの形式が正しいか
2. 有効期限が切れていないか
3. db_hashが一致するか
4. 現在のパスワードハッシュが一致するか（既に変更されていないか）

### 2FA連携

パスワードリセット完了時には、2FAバイパス用のメール検証トークンが生成される（OTP）。これにより、リセット直後のログインで2FAをスキップできる。

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: トークン生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | 36-64行目: `generateToken`関数 |

**読解のコツ**:
- **41-43行目**: settingsAPIからdb_hashを取得
- **45行目**: メールアドレスでユーザー検索
- **52-57行目**: トークン生成（expires, email, dbHash, password）

#### Step 2: メール送信を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | 164-191行目: `sendResetNotification`関数 |

**主要処理フロー**:
- **165行目**: adminUrlを取得
- **166行目**: トークンをBase64エンコード
- **167行目**: リセットURLを生成
- **173-176行目**: mail.utils.generateContentでテンプレート処理
- **178-188行目**: payloadの構築
- **190行目**: mailAPI.sendで送信

#### Step 3: トークン検証とパスワードリセット実行

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | 66-82行目: `extractTokenParts` |
| 3-2 | passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | 84-94行目: `protectBruteForce` |
| 3-3 | passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | 96-162行目: `doReset` |

**主要処理フロー**:
- **67行目**: Base64デコード
- **69-71行目**: トークンからパーツを抽出
- **86-91行目**: 10回以上の試行をブロック
- **115-138行目**: トークンの検証（expired, invalid等）
- **140-144行目**: パスワード変更
- **152-154行目**: 2FAバイパス用OTP生成

### プログラム呼び出し階層図

```
Password Reset Flow
    │
    ├─ generateToken(email, settingsAPI, transaction)
    │      ├─ settingsAPI.read({key: 'db_hash'})
    │      ├─ models.User.getByEmail(email)
    │      └─ security.tokens.resetToken.generateHash()
    │
    └─ sendResetNotification(data, mailAPI)
           ├─ security.url.encodeBase64(data.resetToken)
           ├─ urlUtils.urlJoin(adminUrl, 'reset', resetToken, '/')
           ├─ mail.utils.generateContent({template: 'reset-password'})
           └─ mailAPI.send(payload)

Password Reset Execution
    │
    ├─ extractTokenParts(options)
    │      └─ security.tokens.resetToken.extract()
    │
    ├─ protectBruteForce({options, tokenParts})
    │
    └─ doReset(options, tokenParts, settingsAPI)
           ├─ security.tokens.resetToken.compare()
           ├─ models.User.changePassword()
           └─ otp.generate() - 2FAバイパストークン
```

### データフロー図

```
[入力]                    [処理]                         [出力]

email               ───▶ generateToken ───▶ resetToken
                              │
                              ├─ db_hash取得
                              ├─ ユーザー検索
                              └─ トークン生成

resetToken          ───▶ sendResetNotification ───▶ パスワードリセットメール
                              │
                              ├─ Base64エンコード
                              ├─ URL生成
                              ├─ テンプレート処理
                              └─ メール送信

[出力メール]
  ├─ 件名: Reset Password
  └─ 本文: リセットURL（24時間有効）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| passwordreset.js | `ghost/core/core/server/services/auth/passwordreset.js` | ソース | パスワードリセットサービス |
| otp.js | `ghost/core/core/server/services/auth/otp.js` | ソース | 2FAバイパストークン生成 |
| reset-password | メールテンプレート（mail serviceで管理） | テンプレート | パスワードリセットメール |
| @tryghost/security | 外部パッケージ | ライブラリ | トークン生成・検証 |
