# 通知設計書 32-post.unscheduled

## 概要

本ドキュメントは、Ghost CMSにおける `post.unscheduled` Webhookイベントの設計仕様を定義する。このWebhookは投稿のスケジュール公開が解除された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：コンテンツ管理において、一度スケジュールされた投稿が取り消されるケースは少なくない。外部システム（カレンダー連携、SNS予約投稿、静的サイト生成パイプラインなど）が、スケジュール解除を検知して関連する予約処理をキャンセルする必要がある。このWebhookにより、スケジュール解除のタイミングで外部システムへリアルタイムに通知を送信し、不要な公開処理の実行を防止する。

**通知の送信タイミング**：投稿のステータスが `scheduled` から `draft` 等の非公開・非スケジュール状態に変更された際にトリガーされる。具体的には、Postモデルの `onUpdated` イベントハンドラ内で、ステータスが `scheduled` から別の状態（`draft`等、`published`以外）に変化したことを検知し、`post.unscheduled` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'post.unscheduled'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：投稿の現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。投稿データには、タイトル、スラグ、コンテンツ、変更前の公開予定日時、著者情報、タグ情報などが含まれる。変更前の情報（`previous`）には特に`published_at`（元の公開予定日時）と`status`が含まれる。

**期待されるアクション**：受信システムは、スケジュール解除された投稿の情報を基に、関連する予約処理のキャンセル（SNS予約投稿の取消、静的サイト生成キューからの削除など）を実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'post.unscheduled'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "post": {
    "current": {
      "id": "投稿ID",
      "uuid": "UUID",
      "title": "投稿タイトル",
      "slug": "スラグ",
      "html": "HTMLコンテンツ",
      "plaintext": "プレーンテキスト",
      "status": "draft",
      "published_at": null,
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "投稿URL",
      "authors": [...],
      "tags": [...],
      "tiers": [...],
      "primary_author": {...}
    },
    "previous": {
      "status": "scheduled",
      "published_at": "元の公開予定日時（ISO8601形式）",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| post.current.id | 投稿ID | posts.id | Yes |
| post.current.uuid | 投稿UUID | posts.uuid | Yes |
| post.current.title | タイトル | posts.title | Yes |
| post.current.slug | スラグ | posts.slug | Yes |
| post.current.status | ステータス | posts.status | Yes |
| post.current.published_at | 公開日時（nullの場合あり） | posts.published_at | No |
| post.previous.status | 変更前ステータス | モデル._previousAttributes.status | Yes |
| post.previous.published_at | 変更前公開予定日時 | モデル._previousAttributes.published_at | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Post更新（status: scheduled→draft） | 前ステータスがscheduled、現ステータスがdraft | スケジュール解除してドラフトに戻す |
| 画面操作/API | リソースタイプ変更 | post→pageの変換でscheduled状態から | 投稿からページへの変換時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| 変更後ステータスがpublished | published への変更は unscheduled ではない |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[投稿更新] --> B{ステータス変更?}
    B -->|No| Z[終了]
    B -->|Yes| C{前ステータスがscheduled?}
    C -->|No| Z
    C -->|Yes| D{現ステータスがpublished?}
    D -->|Yes| E[post.published イベント発火]
    D -->|No| F{現ステータスがscheduled?}
    F -->|Yes| Z
    F -->|No| G[post.unscheduled イベント発火]
    G --> H[WebhookTrigger.trigger 呼び出し]
    H --> I[webhooks テーブル検索]
    I --> J{Webhook存在?}
    J -->|No| Z
    J -->|Yes| K[各Webhookにペイロード送信]
    K --> L{送信結果}
    L -->|成功| M[last_triggered_at/status更新]
    L -->|失敗| N[エラー記録・リトライ]
    M --> Z
    N --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | 投稿データ取得 | currentとpreviousの値を取得 |
| webhooks | Webhook設定取得 | event='post.unscheduled'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |
| tags | タグ情報 | posts_tags経由 |
| products | ティア情報 | posts_products経由 |

### テーブル別参照項目詳細

#### webhooks

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| id | Webhook識別 | event = 'post.unscheduled' |
| target_url | 送信先URL | 必須 |
| secret | 署名用シークレット | オプション |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 署名検証により、受信側でリクエストの正当性を確認可能

## 備考

- `post.unscheduled` は scheduled 状態から draft 等の非公開状態への変更時にトリガー
- scheduled から published への変更は `post.published` がトリガー（`post.unscheduled` は発火しない）
- 投稿タイプが `page` の場合は `page.unscheduled` イベントがトリガーされる
- previous オブジェクトには元の公開予定日時（`published_at`）が含まれる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | schema.js | `ghost/core/core/server/data/schema/schema.js` | webhooksテーブル定義、postsテーブル定義 |
| 1-2 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'post.unscheduled'が定義されている（20行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onUpdated`メソッド（394-464行目）でunscheduledイベント発火条件を確認 |

**主要処理フロー**:
1. **400-401行目**: `wasScheduled`（前ステータスがscheduled）を判定
2. **444-446行目**: `wasScheduled && !model.isScheduled && !model.isPublished` の場合に `unscheduled` イベント発火
3. **413行目**: リソースタイプ変更時も `wasScheduled` なら `unscheduled` 発火

#### Step 3: Webhook送信処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | triggerメソッド、getAllメソッド |
| 3-2 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | previousAttributesの取得（43-64行目） |

### プログラム呼び出し階層図

```
Post.edit()
    │
    └─ onUpdated()
           │
           ├─ wasScheduled = model.previous('status') === 'scheduled'
           ├─ isScheduled = model.get('status') === 'scheduled'
           ├─ isPublished = model.get('status') === 'published'
           │
           └─ if (wasScheduled && !isScheduled && !isPublished)
                  └─ emitChange('unscheduled')
                         └─ events.emit('post.unscheduled', model)
                                │
                                └─ WebhookTrigger.trigger('post.unscheduled', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

投稿更新リクエスト ───▶ Postモデル
(status: draft)         │ (ステータス変更検知)
                        │ wasScheduled && !isScheduled && !isPublished
                        ▼
                      emitChange('unscheduled')
                        │
                        ▼
                      WebhookTrigger ───▶ HTTP POST
                        │                 {current: {..., status: 'draft'},
                        │                  previous: {status: 'scheduled', published_at: '...'}}
                        ▼
                      webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | 投稿モデル、イベント発火（444-446行目） |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（20行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | previous属性のシリアライズ |
| posts.test.js | `ghost/core/test/e2e-webhooks/posts.test.js` | テスト | E2Eテスト（364-417行目） |
