# 通知設計書 33-post.rescheduled

## 概要

本ドキュメントは、Ghost CMSにおける `post.rescheduled` Webhookイベントの設計仕様を定義する。このWebhookは既にスケジュールされている投稿の公開予定日時が変更された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：スケジュール済み投稿の公開日時変更は、コンテンツ運用において頻繁に発生する。外部システム（SNS予約投稿、静的サイト生成、カレンダー連携、分析システムなど）が公開予定日時の変更を検知し、既存の予約を更新する必要がある。このWebhookにより、公開日時変更のタイミングで外部システムへリアルタイムに通知を送信し、スケジュールの同期を維持する。

**通知の送信タイミング**：投稿のステータスが `scheduled` のまま、`published_at`（公開予定日時）が変更された際にトリガーされる。具体的には、Postモデルの `onUpdated` イベントハンドラ内で、ステータスが `scheduled` を維持しつつ `published_at` が変更されたことを検知し、`post.rescheduled` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'post.rescheduled'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：投稿の現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。特に `previous` には変更前の `published_at`（元の公開予定日時）が含まれ、`current` には新しい `published_at` が含まれる。

**期待されるアクション**：受信システムは、新旧の公開予定日時を比較し、関連する予約処理のリスケジュール（SNS予約投稿の日時変更、静的サイト生成キューの再スケジュールなど）を実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'post.rescheduled'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "post": {
    "current": {
      "id": "投稿ID",
      "uuid": "UUID",
      "title": "投稿タイトル",
      "slug": "スラグ",
      "html": "HTMLコンテンツ",
      "plaintext": "プレーンテキスト",
      "status": "scheduled",
      "published_at": "新しい公開予定日時（ISO8601形式）",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "投稿URL",
      "authors": [...],
      "tags": [...],
      "tiers": [...]
    },
    "previous": {
      "published_at": "元の公開予定日時（ISO8601形式）",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| post.current.id | 投稿ID | posts.id | Yes |
| post.current.uuid | 投稿UUID | posts.uuid | Yes |
| post.current.title | タイトル | posts.title | Yes |
| post.current.status | ステータス | posts.status | Yes |
| post.current.published_at | 新しい公開予定日時 | posts.published_at | Yes |
| post.previous.published_at | 変更前公開予定日時 | モデル._previousAttributes.published_at | Yes |
| post.previous.updated_at | 変更前更新日時 | モデル._previousAttributes.updated_at | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Post更新（published_at変更） | status=scheduledのまま、published_atが変更 | 公開予定日時を変更 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| ステータスが変更された | ステータス変更時は scheduled/unscheduled イベント |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[投稿更新] --> B{ステータス変更?}
    B -->|Yes| C[scheduled/unscheduled等のイベント]
    B -->|No| D{現在scheduled?}
    D -->|No| Z[終了]
    D -->|Yes| E{published_at変更?}
    E -->|No| F[post.edited イベントのみ]
    E -->|Yes| G[post.rescheduled イベント発火]
    G --> H[WebhookTrigger.trigger 呼び出し]
    H --> I[webhooks テーブル検索]
    I --> J{Webhook存在?}
    J -->|No| Z
    J -->|Yes| K[各Webhookにペイロード送信]
    K --> L{送信結果}
    L -->|成功| M[last_triggered_at/status更新]
    L -->|失敗| N[エラー記録・リトライ]
    M --> Z
    N --> Z
    C --> Z
    F --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | 投稿データ取得 | currentとpreviousの値を取得 |
| webhooks | Webhook設定取得 | event='post.rescheduled'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| published_at | 新旧公開予定日時比較 | 必須 |
| status | ステータス確認 | scheduled であること |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 署名検証により、受信側でリクエストの正当性を確認可能

## 備考

- `post.rescheduled` は scheduled 状態を維持したまま `published_at` を変更した場合にのみトリガー
- ステータスが変更された場合は `post.scheduled` または `post.unscheduled` がトリガー
- `post.edited` イベントも同時に発火する（rescheduledは追加イベント）
- 投稿タイプが `page` の場合は `page.rescheduled` イベントがトリガーされる
- `needsReschedule` フラグは `publishedAtHasChanged && isScheduled` で判定される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'post.rescheduled'が定義されている（21行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onUpdated`メソッド（394-464行目）でrescheduledイベント発火条件を確認 |

**主要処理フロー**:
1. **397-404行目**: 状態変更の判定変数を設定
   - `statusChanging`: ステータスが変更されたか
   - `isScheduled`: 現在のステータスがscheduled
   - `publishedAtHasChanged`: published_atが変更されたか
   - `needsReschedule`: published_at変更かつscheduled状態
2. **447-454行目**: ステータスが変更されていない場合の処理
   - `needsReschedule` が true なら `emitChange('rescheduled')` を呼び出し

#### Step 3: Webhook送信処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | triggerメソッド |
| 3-2 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | previousAttributesの取得 |

### プログラム呼び出し階層図

```
Post.edit()
    │
    └─ onUpdated()
           │
           ├─ statusChanging = get('status') !== previous('status')
           ├─ isScheduled = get('status') === 'scheduled'
           ├─ publishedAtHasChanged = hasDateChanged('published_at')
           ├─ needsReschedule = publishedAtHasChanged && isScheduled
           │
           └─ if (!statusChanging && needsReschedule)
                  └─ emitChange('rescheduled')
                         └─ events.emit('post.rescheduled', model)
                                │
                                └─ WebhookTrigger.trigger('post.rescheduled', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

投稿更新リクエスト ───▶ Postモデル
(published_at変更)      │ (日時変更検知)
                        │ needsReschedule = true
                        ▼
                      emitChange('rescheduled')
                        │
                        ▼
                      WebhookTrigger ───▶ HTTP POST
                        │                 {current: {published_at: '新日時'},
                        │                  previous: {published_at: '旧日時'}}
                        ▼
                      webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | 投稿モデル、イベント発火（452-454行目） |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（21行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | previous属性のシリアライズ |
| posts.test.js | `ghost/core/test/e2e-webhooks/posts.test.js` | テスト | E2Eテスト（633-687行目） |
