# 通知設計書 34-page.added

## 概要

本ドキュメントは、Ghost CMSにおける `page.added` Webhookイベントの設計仕様を定義する。このWebhookは新しいページが作成された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：Ghost CMSでは、投稿（post）とページ（page）を区別して管理している。ページは通常、固定コンテンツ（Aboutページ、お問い合わせページなど）に使用される。外部システム（静的サイトジェネレーター、検索インデックス、サイトマップ生成ツールなど）が新規ページの作成を検知し、適切な処理（インデックス追加、ナビゲーション更新など）を行う必要がある。このWebhookにより、ページ作成のタイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：新しいページが作成された際（Admin APIまたは管理画面から）にトリガーされる。具体的には、Postモデル（Ghost内部ではページも投稿モデルで管理、type='page'で区別）の `onSaved` イベントハンドラ内で、新規作成時（method='insert'）に `page.added` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.added'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：ページの現在の状態（`current`）を含むJSONペイロード。ページデータには、タイトル、スラグ、HTML/プレーンテキストコンテンツ、ステータス、著者情報、タグ情報、ティア情報などが含まれる。`previous` オブジェクトは新規作成のため空となる。

**期待されるアクション**：受信システムは、新規ページの情報を基に、コンテンツインデックスへの追加、サイトマップの更新、ナビゲーションメニューの再構築などを実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.added'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {
      "id": "ページID",
      "uuid": "UUID",
      "title": "ページタイトル",
      "slug": "スラグ",
      "html": "HTMLコンテンツ",
      "plaintext": "プレーンテキスト",
      "status": "draft",
      "published_at": null,
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "ページURL",
      "authors": [...],
      "tags": [...],
      "tiers": [...],
      "primary_author": {...}
    },
    "previous": {}
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current.id | ページID | posts.id (type='page') | Yes |
| page.current.uuid | ページUUID | posts.uuid | Yes |
| page.current.title | タイトル | posts.title | Yes |
| page.current.slug | スラグ | posts.slug | Yes |
| page.current.status | ステータス | posts.status | Yes |
| page.current.html | HTMLコンテンツ | posts.html | No |
| page.current.plaintext | プレーンテキスト | posts.plaintext | No |
| page.current.authors | 著者リスト | posts_authors + users | Yes |
| page.current.tags | タグリスト | posts_tags + tags | No |
| page.current.tiers | ティアリスト | posts_products + products | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page作成 | 新規ページが作成された | Admin APIまたは管理画面からページを作成 |
| 画面操作/API | リソースタイプ変更 | post→pageへの変換 | 投稿からページへの変換時にaddedイベント発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ作成] --> B{インポート中?}
    B -->|Yes| Z[終了]
    B -->|No| C[page.added イベント発火]
    C --> D{status=scheduled/published?}
    D -->|Yes| E[追加で scheduled/published イベントも発火]
    D -->|No| F[addedのみ]
    E --> G[WebhookTrigger.trigger 呼び出し]
    F --> G
    G --> H[webhooks テーブル検索]
    H --> I{Webhook存在?}
    I -->|No| Z
    I -->|Yes| J[各Webhookにペイロード送信]
    J --> K{送信結果}
    K -->|成功| L[last_triggered_at/status更新]
    K -->|失敗| M[エラー記録・リトライ]
    L --> Z
    M --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | type='page' で絞り込み |
| webhooks | Webhook設定取得 | event='page.added'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |
| tags | タグ情報 | posts_tags経由 |
| products | ティア情報 | posts_products経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| type | ページ判定 | type = 'page' |
| 全カラム | ペイロード生成 | id指定 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 署名検証により、受信側でリクエストの正当性を確認可能

## 備考

- Ghost内部では、ページ（page）と投稿（post）は同じ `posts` テーブルで管理され、`type` カラムで区別される
- ページ作成時に同時に `published` または `scheduled` ステータスの場合、`page.added` に加えて `page.published` または `page.scheduled` も発火する
- ペイロードのキー名は `page`（投稿の場合は `post`）
- シリアライズ処理で `resourceName` がイベント名から抽出され、`docName` が `pages` となる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | schema.js | `ghost/core/core/server/data/schema/schema.js` | postsテーブルの`type`カラム（73行目）で'post'/'page'を区別 |
| 1-2 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.added'が定義されている（23行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onSaved`メソッド（378-392行目）、`emitChange`メソッド（356-367行目） |

**主要処理フロー**:
1. **356-367行目**: `emitChange`メソッドで`resourceType`（'post'または'page'）を取得し、イベント名を構築
2. **358行目**: `let resourceType = this.get('type')` でページかポストかを判定
3. **364行目**: `eventToTrigger = resourceType + '.' + event` でイベント名を生成（例：'page.added'）
4. **387行目**: `model.emitChange('added', options)` でaddedイベント発火

#### Step 3: シリアライズ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | イベント名からリソース名を抽出（7行目） |

**主要処理フロー**:
- **7行目**: `const resourceName = event.match(/(\w+)\./)[1]` で 'page' を抽出
- **8行目**: `const docName = \`${resourceName}s\`` で 'pages' を生成
- **74行目**: ペイロードのキーとして `page` を使用

### プログラム呼び出し階層図

```
Page作成（Admin API / 管理画面）
    │
    └─ Post.add() (type='page')
           │
           └─ onSaved() (method='insert')
                  │
                  ├─ emitChange('added')
                  │      └─ resourceType = 'page'
                  │      └─ eventToTrigger = 'page.added'
                  │      └─ events.emit('page.added', model)
                  │
                  └─ if (status === 'published' || 'scheduled')
                         └─ emitChange(status)
                                │
                                └─ WebhookTrigger.trigger('page.added', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ページ作成リクエスト ───▶ Postモデル (type='page')
                         │ (新規作成)
                         ▼
                       onSaved (method='insert')
                         │
                         ▼
                       emitChange('added')
                         │ resourceType = 'page'
                         ▼
                       WebhookTrigger ───▶ HTTP POST
                         │                 {page: {current: {...}, previous: {}}}
                         ▼
                       webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル（type='page'）、イベント発火 |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（23行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | リソース名抽出、シリアライズ |
| schema.js | `ghost/core/core/server/data/schema/schema.js` | 設定 | postsテーブルのtype定義 |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（173-210行目） |
