# 通知設計書 35-page.deleted

## 概要

本ドキュメントは、Ghost CMSにおける `page.deleted` Webhookイベントの設計仕様を定義する。このWebhookはページが削除された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：ページの削除は、サイト構造に影響を与える重要な操作である。外部システム（静的サイトジェネレーター、検索インデックス、サイトマップ生成ツール、CDNキャッシュなど）が削除されたページを検知し、適切な処理（インデックスからの削除、リダイレクト設定、キャッシュパージなど）を行う必要がある。このWebhookにより、ページ削除のタイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：ページが削除された際（Admin APIまたは管理画面から）にトリガーされる。具体的には、Postモデルの `onDestroyed` イベントハンドラ内で、ページ（type='page'）の削除時に `page.deleted` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.deleted'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：削除されたページの情報を含むJSONペイロード。`current` は空オブジェクトとなり、`previous` に削除されたページの情報（ID、UUID、タイトル、スラグ、著者情報など）が含まれる。これにより、受信システムは削除されたページを特定できる。

**期待されるアクション**：受信システムは、削除されたページの情報を基に、検索インデックスからの削除、サイトマップの更新、キャッシュパージ、リダイレクト設定などを実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.deleted'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {},
    "previous": {
      "id": "削除されたページID",
      "uuid": "UUID",
      "comment_id": "コメントID",
      "published_at": "公開日時（公開済みの場合）",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "authors": [...]
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current | 現在のデータ | 空オブジェクト | Yes |
| page.previous.id | 削除されたページID | モデル._previousAttributes.id | Yes |
| page.previous.uuid | UUID | モデル._previousAttributes.uuid | Yes |
| page.previous.comment_id | コメントID | モデル._previousAttributes.comment_id | No |
| page.previous.published_at | 公開日時 | モデル._previousAttributes.published_at | No |
| page.previous.created_at | 作成日時 | モデル._previousAttributes.created_at | Yes |
| page.previous.updated_at | 更新日時 | モデル._previousAttributes.updated_at | Yes |
| page.previous.authors | 著者リスト | モデルのリレーション | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page削除 | ページが削除された | Admin APIまたは管理画面からページを削除 |
| 画面操作/API | リソースタイプ変更 | page→postへの変換 | ページから投稿への変換時にdeletedイベント発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ削除] --> B{インポート中?}
    B -->|Yes| Z[終了]
    B -->|No| C{公開済み?}
    C -->|Yes| D[page.unpublished イベント発火]
    C -->|No| E[skip unpublished]
    D --> F[page.deleted イベント発火]
    E --> F
    F --> G[WebhookTrigger.trigger 呼び出し]
    G --> H[webhooks テーブル検索]
    H --> I{Webhook存在?}
    I -->|No| Z
    I -->|Yes| J[各Webhookにペイロード送信]
    J --> K{送信結果}
    K -->|成功| L[last_triggered_at/status更新]
    K -->|失敗| M[エラー記録・リトライ]
    L --> Z
    M --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | 削除前のデータ（previousAttributes経由） |
| webhooks | Webhook設定取得 | event='page.deleted'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| id | ページ識別 | 削除対象 |
| type | ページ判定 | type = 'page' |
| previous('status') | 公開状態確認 | unpublishedイベント発火判定用 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| posts | DELETE | ページの削除 |
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 削除されたコンテンツの情報が外部に送信されることに注意

## 備考

- 削除時、`current` は空オブジェクトとなり、`previous` に削除前のデータが格納される
- 公開済みページの削除時は、`page.unpublished` イベントが先に発火し、その後 `page.deleted` が発火する
- リソースタイプ変更（page→post）時も `page.deleted` が発火する（usePreviousAttribute オプション付き）
- シリアライズ処理では、削除時は attributes が空のため previous のみが意味を持つ

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.deleted'が定義されている（24行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onDestroyed`メソッド（466-474行目） |

**主要処理フロー**:
1. **469-471行目**: 公開済みの場合、`unpublished` イベントを先に発火（usePreviousAttribute付き）
2. **473行目**: `model.emitChange('deleted', Object.assign({usePreviousAttribute: true}, options))` でdeletedイベント発火
3. **360-362行目**: `usePreviousAttribute` オプションにより、`previous('type')` を使用してリソースタイプを決定

#### Step 3: シリアライズ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | 削除時の処理 |

**主要処理フロー**:
- **15行目**: `Object.keys(model.attributes).length` が 0 の場合（削除時）、空オブジェクトを返す
- **37-41行目**: attributes が空の場合の処理
- **43-64行目**: previousAttributes からデータを取得

### プログラム呼び出し階層図

```
Page削除（Admin API / 管理画面）
    │
    └─ Post.destroy() (type='page')
           │
           └─ onDestroyed()
                  │
                  ├─ if (previous('status') === 'published')
                  │      └─ emitChange('unpublished', {usePreviousAttribute: true})
                  │
                  └─ emitChange('deleted', {usePreviousAttribute: true})
                         │
                         ├─ resourceType = previous('type') = 'page'
                         └─ events.emit('page.deleted', model)
                                │
                                └─ WebhookTrigger.trigger('page.deleted', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ページ削除リクエスト ───▶ Postモデル (type='page')
                         │ (削除)
                         ▼
                       onDestroyed
                         │
                         ▼
                       emitChange('deleted')
                         │ usePreviousAttribute: true
                         ▼
                       WebhookTrigger ───▶ HTTP POST
                         │                 {page: {current: {}, previous: {...}}}
                         ▼
                       webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル、onDestroyedイベント（466-474行目） |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（24行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | 削除時のシリアライズ |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（212-254行目） |
