# 通知設計書 36-page.edited

## 概要

本ドキュメントは、Ghost CMSにおける `page.edited` Webhookイベントの設計仕様を定義する。このWebhookはページが編集された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：ページの編集は最も頻繁に発生する操作の一つである。外部システム（静的サイトジェネレーター、検索インデックス、キャッシュシステム、プレビュー生成ツールなど）がページの更新を検知し、適切な処理（コンテンツの再取得、インデックスの更新、キャッシュの無効化など）を行う必要がある。このWebhookにより、ページ編集のタイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：ページが更新された際（Admin APIまたは管理画面から）にトリガーされる。具体的には、Postモデルの `onUpdated` イベントハンドラ内で、リソースタイプが変更されていない場合に `page.edited` イベントを発火する。ステータス変更を伴う場合は、他のイベント（published, unpublished等）と共に発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.edited'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：ページの現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。`current` には更新後の全データ、`previous` には変更されたフィールドの変更前の値が含まれる。

**期待されるアクション**：受信システムは、更新されたページの情報を基に、コンテンツの再取得、検索インデックスの更新、キャッシュの無効化などを実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 中 |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.edited'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {
      "id": "ページID",
      "uuid": "UUID",
      "title": "更新後のタイトル",
      "slug": "更新後のスラグ",
      "html": "更新後のHTMLコンテンツ",
      "plaintext": "更新後のプレーンテキスト",
      "status": "ステータス",
      "published_at": "公開日時",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "ページURL",
      "authors": [...],
      "tags": [...],
      "tiers": [...]
    },
    "previous": {
      "title": "変更前のタイトル（変更された場合）",
      "slug": "変更前のスラグ（変更された場合）",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current.id | ページID | posts.id | Yes |
| page.current.uuid | UUID | posts.uuid | Yes |
| page.current.title | 更新後タイトル | posts.title | Yes |
| page.current.slug | 更新後スラグ | posts.slug | Yes |
| page.current.status | ステータス | posts.status | Yes |
| page.current.html | HTMLコンテンツ | posts.html | No |
| page.previous.* | 変更前の値 | モデル._previousAttributes | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page更新 | ページが更新された | Admin APIまたは管理画面からページを編集 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| リソースタイプ変更 | post→page または page→post の変換時は別イベント |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ更新] --> B{リソースタイプ変更?}
    B -->|Yes| C[added/deleted等のイベント]
    B -->|No| D{ステータス変更?}
    D -->|Yes| E[published/unpublished等のイベント]
    D -->|No| F[編集のみ]
    E --> G[page.edited イベント発火]
    F --> G
    G --> H[WebhookTrigger.trigger 呼び出し]
    H --> I[webhooks テーブル検索]
    I --> J{Webhook存在?}
    J -->|No| Z[終了]
    J -->|Yes| K[各Webhookにペイロード送信]
    K --> L{送信結果}
    L -->|成功| M[last_triggered_at/status更新]
    L -->|失敗| N[エラー記録・リトライ]
    M --> Z
    N --> Z
    C --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | type='page'、currentとpreviousの値を取得 |
| webhooks | Webhook設定取得 | event='page.edited'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |
| tags | タグ情報 | posts_tags経由 |
| products | ティア情報 | posts_products経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| type | ページ判定 | type = 'page' |
| 全カラム | ペイロード生成 | id指定 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 署名検証により、受信側でリクエストの正当性を確認可能

## 備考

- `page.edited` は最も頻繁に発火するイベントの一つ
- ステータス変更を伴う場合は、対応するイベント（published, unpublished等）と共に発火
- リソースタイプ変更時（page→post）は `page.deleted` が発火し、`page.edited` は発火しない
- `previous` には変更されたフィールドのみが含まれる（`model._changed` に基づく）
- 公開済みページの編集時は `page.published.edited` も同時に発火する

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.edited'が定義されている（25行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onUpdated`メソッド（394-464行目） |

**主要処理フロー**:
1. **402行目**: `resourceTypeChanging = model.get('type') !== model.previous('type')` でリソースタイプ変更を判定
2. **426行目**: リソースタイプが変更されていない場合（else文）
3. **457-458行目**: `model.emitChange('edited', options)` でeditedイベント発火

#### Step 3: シリアライズ処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | previous値の取得 |

**主要処理フロー**:
- **71行目**: `const changed = model._changed ? Object.keys(model._changed) : {}` で変更されたフィールドを取得
- **76行目**: `previous: _.pick(previous, changed)` で変更されたフィールドのみをpreviousに含める

### プログラム呼び出し階層図

```
Page編集（Admin API / 管理画面）
    │
    └─ Post.edit() (type='page')
           │
           └─ onUpdated()
                  │
                  ├─ resourceTypeChanging判定
                  │
                  └─ if (!resourceTypeChanging)
                         │
                         ├─ if (statusChanging)
                         │      └─ published/unpublished等のイベント
                         │
                         └─ emitChange('edited')
                                │
                                └─ events.emit('page.edited', model)
                                       │
                                       └─ WebhookTrigger.trigger('page.edited', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ページ更新リクエスト ───▶ Postモデル (type='page')
                         │ (更新)
                         ▼
                       onUpdated
                         │ resourceTypeChanging = false
                         ▼
                       emitChange('edited')
                         │
                         ▼
                       WebhookTrigger ───▶ HTTP POST
                         │                 {page: {current: {...}, previous: {変更フィールドのみ}}}
                         ▼
                       webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル、onUpdatedイベント（457-458行目） |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（25行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | changed fieldsのシリアライズ |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（256-307行目） |
