# 通知設計書 38-page.published.edited

## 概要

本ドキュメントは、Ghost CMSにおける `page.published.edited` Webhookイベントの設計仕様を定義する。このWebhookは既に公開されているページが編集された際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：公開済みページの編集は、サイトを訪問中のユーザーに影響を与える重要な操作である。外部システム（CDNキャッシュ、静的サイトジェネレーター、検索インデックス、プレビューシステムなど）が公開コンテンツの更新を検知し、即座にキャッシュの無効化やコンテンツの再取得を行う必要がある。このWebhookにより、公開済みページの編集タイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：公開済み（status='published'）のページが編集され、ステータスが変更されないまま保存された際にトリガーされる。具体的には、Postモデルの `onUpdated` イベントハンドラ内で、ステータスが `published` を維持したまま更新が行われたことを検知し、`page.published.edited` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.published.edited'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：ページの現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。`current` には更新後の公開ページの全データ、`previous` には変更されたフィールドの変更前の値が含まれる。

**期待されるアクション**：受信システムは、更新された公開ページの情報を基に、CDNキャッシュのパージ、静的サイトの再生成、検索インデックスの更新などを即座に実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 高（公開コンテンツの更新） |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.published.edited'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {
      "id": "ページID",
      "uuid": "UUID",
      "title": "更新後のタイトル",
      "slug": "更新後のスラグ",
      "html": "更新後のHTMLコンテンツ",
      "plaintext": "更新後のプレーンテキスト",
      "status": "published",
      "published_at": "公開日時",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "ページURL",
      "authors": [...],
      "tags": [...],
      "tiers": [...]
    },
    "previous": {
      "title": "変更前のタイトル（変更された場合）",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current.id | ページID | posts.id | Yes |
| page.current.uuid | UUID | posts.uuid | Yes |
| page.current.title | 更新後タイトル | posts.title | Yes |
| page.current.status | ステータス（published） | posts.status | Yes |
| page.current.published_at | 公開日時 | posts.published_at | Yes |
| page.current.html | HTMLコンテンツ | posts.html | No |
| page.previous.* | 変更前の値 | モデル._previousAttributes | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page更新（公開済み） | status=publishedのまま編集 | 公開済みページの内容を変更 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| ステータスが変更された | published→draft等の変更は別イベント |
| 現在のステータスがpublishedでない | 公開済みでないページの編集 |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ更新] --> B{ステータス変更?}
    B -->|Yes| C[published/unpublished等のイベント]
    B -->|No| D{現在published?}
    D -->|No| E[page.edited のみ]
    D -->|Yes| F[page.published.edited イベント発火]
    F --> G[WebhookTrigger.trigger 呼び出し]
    G --> H[webhooks テーブル検索]
    H --> I{Webhook存在?}
    I -->|No| Z[終了]
    I -->|Yes| J[各Webhookにペイロード送信]
    J --> K{送信結果}
    K -->|成功| L[last_triggered_at/status更新]
    K -->|失敗| M[エラー記録・リトライ]
    L --> Z
    M --> Z
    C --> Z
    E --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | type='page'、status='published' |
| webhooks | Webhook設定取得 | event='page.published.edited'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| type | ページ判定 | type = 'page' |
| status | 公開状態確認 | status = 'published' |
| 全カラム | ペイロード生成 | id指定 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 公開コンテンツの更新通知のため、センシティブ情報の取り扱いに注意

## 備考

- `page.published.edited` は公開済みページ（status='published'）の編集時にのみトリガー
- `page.edited` イベントも同時に発火する
- ステータスが変更された場合は、`page.published`/`page.unpublished` がトリガーされ、`page.published.edited` は発火しない
- CDNキャッシュのパージなど、即時性が求められる処理に使用される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.published.edited'が定義されている（27行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onUpdated`メソッド（394-464行目） |

**主要処理フロー**:
1. **397行目**: `statusChanging = model.get('status') !== model.previous('status')` でステータス変更を判定
2. **398行目**: `isPublished = model.get('status') === 'published'` で公開状態を判定
3. **447-450行目**: `!statusChanging && isPublished` の場合に `published.edited` イベント発火

```javascript
// ghost/core/core/server/models/post.js 447-450行目
if (model.isPublished) {
    model.emitChange('published.edited', options);
}
```

#### Step 3: Webhook送信処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | triggerメソッド |
| 3-2 | serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | changed fieldsのシリアライズ |

### プログラム呼び出し階層図

```
公開済みページ編集（Admin API / 管理画面）
    │
    └─ Post.edit() (type='page', status='published')
           │
           └─ onUpdated()
                  │
                  ├─ statusChanging = false (ステータス変更なし)
                  ├─ isPublished = true
                  │
                  └─ if (!statusChanging)
                         │
                         └─ if (isPublished)
                                └─ emitChange('published.edited')
                                       │
                                       └─ events.emit('page.published.edited', model)
                                              │
                                              └─ WebhookTrigger.trigger('page.published.edited', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

公開済みページ編集 ───▶ Postモデル (type='page', status='published')
                        │ (ステータス維持)
                        ▼
                      onUpdated
                        │ statusChanging = false
                        │ isPublished = true
                        ▼
                      emitChange('published.edited')
                        │
                        ▼
                      WebhookTrigger ───▶ HTTP POST
                        │                 {page: {current: {status: 'published', ...},
                        │                         previous: {変更フィールドのみ}}}
                        ▼
                      webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル、published.editedイベント発火（448-450行目） |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（27行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | シリアライズ |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（361-410行目） |
