# 通知設計書 39-page.unpublished

## 概要

本ドキュメントは、Ghost CMSにおける `page.unpublished` Webhookイベントの設計仕様を定義する。このWebhookはページが非公開にされた際にトリガーされる。

### 本通知の処理概要

**業務上の目的・背景**：ページの非公開化は、サイトコンテンツを一般ユーザーから非表示にする重要な操作である。外部システム（検索エンジン、CDNキャッシュ、静的サイトジェネレーター、サイトマップ生成ツールなど）がページの非公開を検知し、適切な処理（インデックスからの削除、キャッシュパージ、サイトマップ更新など）を行う必要がある。このWebhookにより、ページ非公開のタイミングで外部システムへリアルタイムに通知を送信する。

**通知の送信タイミング**：ページのステータスが `published` から `draft` 等の非公開状態に変更された際にトリガーされる。また、公開済みページが削除された際やリソースタイプが変更された際にも発火する。具体的には、Postモデルの `onUpdated` または `onDestroyed` イベントハンドラ内で、公開状態から非公開状態への変化を検知し、`page.unpublished` イベントを発火する。

**通知の受信者**：`webhooks` テーブルに登録されている `event = 'page.unpublished'` のWebhookエンドポイント全て。各Webhookは特定のIntegration（連携アプリケーション）に紐づいており、カスタムIntegration制限がある場合は内部Integrationのみに制限される。

**通知内容の概要**：ページの現在の状態（`current`）と変更前の状態（`previous`）を含むJSONペイロード。`current` には非公開後のページの全データ、`previous` には変更前の状態（特に `status` = 'published'）が含まれる。

**期待されるアクション**：受信システムは、非公開になったページの情報を基に、検索インデックスからの削除、CDNキャッシュのパージ、サイトマップからの削除、静的サイトの再生成などを実行することが期待される。

## 通知種別

Webhook（HTTP POST リクエスト）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（イベント駆動） |
| 優先度 | 高（公開コンテンツの非公開） |
| リトライ | 有・5回（テスト環境では0回） |

### 送信先決定ロジック

1. `webhooks` テーブルから `event = 'page.unpublished'` に一致するレコードを取得
2. カスタムIntegration制限（`customIntegrations` limit）が有効な場合、`integration.type = 'internal'` のWebhookのみに絞り込み
3. 各Webhookの `target_url` に対してHTTP POSTリクエストを送信

## 通知テンプレート

### Webhook通知の場合

| 項目 | 内容 |
|-----|------|
| HTTPメソッド | POST |
| Content-Type | application/json |
| Content-Version | v{Ghost APIバージョン} |
| タイムアウト | 2000ms |

### ペイロード構造

```json
{
  "page": {
    "current": {
      "id": "ページID",
      "uuid": "UUID",
      "title": "ページタイトル",
      "slug": "スラグ",
      "html": "HTMLコンテンツ",
      "plaintext": "プレーンテキスト",
      "status": "draft",
      "published_at": "公開日時（以前の公開日時）",
      "created_at": "作成日時",
      "updated_at": "更新日時",
      "url": "ページURL",
      "authors": [...],
      "tags": [...],
      "tiers": [...]
    },
    "previous": {
      "status": "published",
      "updated_at": "変更前の更新日時"
    }
  }
}
```

### 署名ヘッダー

Webhookに `secret` が設定されている場合、以下の形式で署名ヘッダーが付与される：

```
X-Ghost-Signature: sha256={HMAC-SHA256署名}, t={タイムスタンプ}
```

### 添付ファイル

なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| page.current.id | ページID | posts.id | Yes |
| page.current.uuid | UUID | posts.uuid | Yes |
| page.current.title | タイトル | posts.title | Yes |
| page.current.status | ステータス | posts.status | Yes |
| page.current.published_at | 公開日時 | posts.published_at | No |
| page.previous.status | 変更前ステータス | モデル._previousAttributes.status | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 画面操作/API | Page更新（status変更） | published→draft等に変更 | 公開ページを非公開に変更 |
| 画面操作/API | Page削除 | 公開済みページの削除 | 公開ページを削除時に先にunpublished発火 |
| 画面操作/API | リソースタイプ変更 | page→postへの変換で公開状態 | ページから投稿への変換時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| options.importing = true | データインポート中は送信しない |
| 前ステータスがpublishedでない | 元々非公開のページは対象外 |
| webhooks テーブルに該当レコードなし | 登録されたWebhookがない場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ページ更新/削除] --> B{削除?}
    B -->|Yes| C{前ステータスがpublished?}
    B -->|No| D{ステータス変更?}
    C -->|Yes| E[page.unpublished イベント発火]
    C -->|No| F[page.deleted のみ]
    D -->|Yes| G{前ステータスがpublished?}
    D -->|No| Z[終了]
    G -->|Yes| E
    G -->|No| Z
    E --> H[WebhookTrigger.trigger 呼び出し]
    H --> I[webhooks テーブル検索]
    I --> J{Webhook存在?}
    J -->|No| Z
    J -->|Yes| K[各Webhookにペイロード送信]
    K --> L{送信結果}
    L -->|成功| M[last_triggered_at/status更新]
    L -->|失敗| N[エラー記録・リトライ]
    M --> Z
    N --> Z
    F --> Z
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| posts | ページデータ取得 | type='page'、currentとpreviousの値を取得 |
| webhooks | Webhook設定取得 | event='page.unpublished'で検索 |
| integrations | Integration情報 | Webhook経由で関連取得 |
| users | 著者情報 | posts_authors経由 |

### テーブル別参照項目詳細

#### posts

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| type | ページ判定 | type = 'page' |
| status | ステータス確認 | previous('status') = 'published' |
| 全カラム | ペイロード生成 | id指定 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| webhooks | UPDATE | 送信結果の記録 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 送信失敗 | ネットワークエラー、タイムアウト | リトライ実行、エラーログ記録 |
| 410 Gone | 送信先が恒久的に削除 | Webhookレコードを自動削除 |
| 4xx/5xx | サーバーエラー | ステータス記録、リトライ実行 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 5回（本番環境）、0回（テスト環境） |
| リトライ間隔 | @tryghost/requestライブラリのデフォルト設定に依存 |
| リトライ対象エラー | ネットワークエラー、一時的なサーバーエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（イベント発生ベース） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし（イベント発生時に即座に送信）

## セキュリティ考慮事項

- Webhookに `secret` を設定することで、HMAC-SHA256署名による送信元検証が可能
- HTTPS URLの使用を推奨
- 非公開コンテンツの情報が外部に送信されることに注意

## 備考

- `page.unpublished` は公開済みページ（previous status='published'）が非公開になった時にトリガー
- 公開済みページの削除時は、`page.unpublished` が先に発火し、その後 `page.deleted` が発火
- リソースタイプ変更時（page→post）で公開状態の場合、`page.unpublished` が発火
- `page.edited` イベントも同時に発火する（削除時を除く）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | listen.js | `ghost/core/core/server/services/webhooks/listen.js` | WEBHOOKS配列で'page.unpublished'が定義されている（28行目） |

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | post.js | `ghost/core/core/server/models/post.js` | `onUpdated`（394-464行目）と`onDestroyed`（466-474行目） |

**主要処理フロー**:
1. **400行目**: `wasPublished = model.previous('status') === 'published'` で前ステータスを判定
2. **427-431行目**: ステータスが変更され、wasPublished が true の場合に `unpublished` イベント発火
3. **408-410行目**: リソースタイプ変更時も、wasPublished なら `unpublished` 発火
4. **469-471行目**: 削除時、前ステータスが `published` なら `unpublished` 発火

#### Step 3: Webhook送信処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | triggerメソッド |

### プログラム呼び出し階層図

```
ページ非公開化（Admin API / 管理画面）
    │
    └─ Post.edit() / Post.destroy() (type='page')
           │
           ├─ onUpdated() (status変更時)
           │      │
           │      ├─ wasPublished = previous('status') === 'published'
           │      │
           │      └─ if (statusChanging && wasPublished)
           │              └─ emitChange('unpublished')
           │
           └─ onDestroyed() (削除時)
                  │
                  └─ if (previous('status') === 'published')
                         └─ emitChange('unpublished', {usePreviousAttribute: true})
                                │
                                └─ events.emit('page.unpublished', model)
                                       │
                                       └─ WebhookTrigger.trigger('page.unpublished', model)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ページ非公開リクエスト ───▶ Postモデル (type='page')
(status: draft)          │ (ステータス変更検知)
                         │ wasPublished = true
                         ▼
                       emitChange('unpublished')
                         │
                         ▼
                       WebhookTrigger ───▶ HTTP POST
                         │                 {page: {current: {status: 'draft', ...},
                         │                         previous: {status: 'published', ...}}}
                         ▼
                       webhooksテーブル更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| post.js | `ghost/core/core/server/models/post.js` | ソース | ページモデル、unpublishedイベント発火 |
| listen.js | `ghost/core/core/server/services/webhooks/listen.js` | ソース | イベントリスナー登録（28行目） |
| webhook-trigger.js | `ghost/core/core/server/services/webhooks/webhook-trigger.js` | ソース | Webhook送信処理 |
| serialize.js | `ghost/core/core/server/services/webhooks/serialize.js` | ソース | シリアライズ |
| pages.test.js | `ghost/core/test/e2e-webhooks/pages.test.js` | テスト | E2Eテスト（412-461行目） |
